/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the query string parameters that you want CloudFront to use for
 * caching for a cache behavior.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryStringCacheKeys implements SdkPojo, Serializable,
        ToCopyableBuilder<QueryStringCacheKeys.Builder, QueryStringCacheKeys> {
    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(QueryStringCacheKeys::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(QueryStringCacheKeys::items))
            .setter(setter(Builder::items))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Name")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Name").unmarshallLocationName("Name").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUANTITY_FIELD, ITEMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer quantity;

    private final List<String> items;

    private QueryStringCacheKeys(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * The number of <code>whitelisted</code> query string parameters for a cache behavior.
     * </p>
     * 
     * @return The number of <code>whitelisted</code> query string parameters for a cache behavior.
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * Returns true if the Items property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasItems() {
        return items != null && !(items instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list that contains the query string parameters that you want CloudFront to use as a basis for caching for a
     * cache behavior. If <code>Quantity</code> is 0, you can omit <code>Items</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasItems()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list that contains the query string parameters that you want CloudFront to use as a basis for caching
     *         for a cache behavior. If <code>Quantity</code> is 0, you can omit <code>Items</code>.
     */
    public List<String> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(items());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryStringCacheKeys)) {
            return false;
        }
        QueryStringCacheKeys other = (QueryStringCacheKeys) obj;
        return Objects.equals(quantity(), other.quantity()) && Objects.equals(items(), other.items());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("QueryStringCacheKeys").add("Quantity", quantity()).add("Items", items()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(items()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QueryStringCacheKeys, T> g) {
        return obj -> g.apply((QueryStringCacheKeys) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryStringCacheKeys> {
        /**
         * <p>
         * The number of <code>whitelisted</code> query string parameters for a cache behavior.
         * </p>
         * 
         * @param quantity
         *        The number of <code>whitelisted</code> query string parameters for a cache behavior.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A list that contains the query string parameters that you want CloudFront to use as a basis for caching for a
         * cache behavior. If <code>Quantity</code> is 0, you can omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        A list that contains the query string parameters that you want CloudFront to use as a basis for
         *        caching for a cache behavior. If <code>Quantity</code> is 0, you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * A list that contains the query string parameters that you want CloudFront to use as a basis for caching for a
         * cache behavior. If <code>Quantity</code> is 0, you can omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        A list that contains the query string parameters that you want CloudFront to use as a basis for
         *        caching for a cache behavior. If <code>Quantity</code> is 0, you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(QueryStringCacheKeys model) {
            quantity(model.quantity);
            items(model.items);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = QueryStringCacheKeysListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = QueryStringCacheKeysListCopier.copy(items);
        }

        @Override
        public QueryStringCacheKeys build() {
            return new QueryStringCacheKeys(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
