/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that controls the countries in which your content is distributed. CloudFront determines the location
 * of your users using <code>MaxMind</code> GeoIP databases.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GeoRestriction implements SdkPojo, Serializable, ToCopyableBuilder<GeoRestriction.Builder, GeoRestriction> {
    private static final SdkField<String> RESTRICTION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(GeoRestriction::restrictionTypeAsString))
            .setter(setter(Builder::restrictionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestrictionType")
                    .unmarshallLocationName("RestrictionType").build()).build();

    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(GeoRestriction::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GeoRestriction::items))
            .setter(setter(Builder::items))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Location")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Location").unmarshallLocationName("Location").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESTRICTION_TYPE_FIELD,
            QUANTITY_FIELD, ITEMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String restrictionType;

    private final Integer quantity;

    private final List<String> items;

    private GeoRestriction(BuilderImpl builder) {
        this.restrictionType = builder.restrictionType;
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * The method that you want to use to restrict distribution of your content by country:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by client geo
     * location.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     * CloudFront to distribute your content.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront to
     * distribute your content.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #restrictionType}
     * will return {@link GeoRestrictionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #restrictionTypeAsString}.
     * </p>
     * 
     * @return The method that you want to use to restrict distribution of your content by country:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by client
     *         geo location.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     * @see GeoRestrictionType
     */
    public GeoRestrictionType restrictionType() {
        return GeoRestrictionType.fromValue(restrictionType);
    }

    /**
     * <p>
     * The method that you want to use to restrict distribution of your content by country:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by client geo
     * location.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     * CloudFront to distribute your content.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront to
     * distribute your content.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #restrictionType}
     * will return {@link GeoRestrictionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #restrictionTypeAsString}.
     * </p>
     * 
     * @return The method that you want to use to restrict distribution of your content by country:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by client
     *         geo location.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     * @see GeoRestrictionType
     */
    public String restrictionTypeAsString() {
        return restrictionType;
    }

    /**
     * <p>
     * When geo restriction is <code>enabled</code>, this is the number of countries in your <code>whitelist</code> or
     * <code>blacklist</code>. Otherwise, when it is not enabled, <code>Quantity</code> is <code>0</code>, and you can
     * omit <code>Items</code>.
     * </p>
     * 
     * @return When geo restriction is <code>enabled</code>, this is the number of countries in your
     *         <code>whitelist</code> or <code>blacklist</code>. Otherwise, when it is not enabled,
     *         <code>Quantity</code> is <code>0</code>, and you can omit <code>Items</code>.
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * Returns true if the Items property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasItems() {
        return items != null && !(items instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A complex type that contains a <code>Location</code> element for each country in which you want CloudFront either
     * to distribute your content (<code>whitelist</code>) or not distribute your content (<code>blacklist</code>).
     * </p>
     * <p>
     * The <code>Location</code> element is a two-letter, uppercase country code for a country that you want to include
     * in your <code>blacklist</code> or <code>whitelist</code>. Include one <code>Location</code> element for each
     * country.
     * </p>
     * <p>
     * CloudFront and <code>MaxMind</code> both use <code>ISO 3166</code> country codes. For the current list of
     * countries and the corresponding codes, see <code>ISO 3166-1-alpha-2</code> code on the <i>International
     * Organization for Standardization</i> website. You can also refer to the country list on the CloudFront console,
     * which includes both country names and codes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasItems()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A complex type that contains a <code>Location</code> element for each country in which you want
     *         CloudFront either to distribute your content (<code>whitelist</code>) or not distribute your content (
     *         <code>blacklist</code>).</p>
     *         <p>
     *         The <code>Location</code> element is a two-letter, uppercase country code for a country that you want to
     *         include in your <code>blacklist</code> or <code>whitelist</code>. Include one <code>Location</code>
     *         element for each country.
     *         </p>
     *         <p>
     *         CloudFront and <code>MaxMind</code> both use <code>ISO 3166</code> country codes. For the current list of
     *         countries and the corresponding codes, see <code>ISO 3166-1-alpha-2</code> code on the <i>International
     *         Organization for Standardization</i> website. You can also refer to the country list on the CloudFront
     *         console, which includes both country names and codes.
     */
    public List<String> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(restrictionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(items());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GeoRestriction)) {
            return false;
        }
        GeoRestriction other = (GeoRestriction) obj;
        return Objects.equals(restrictionTypeAsString(), other.restrictionTypeAsString())
                && Objects.equals(quantity(), other.quantity()) && Objects.equals(items(), other.items());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GeoRestriction").add("RestrictionType", restrictionTypeAsString()).add("Quantity", quantity())
                .add("Items", items()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RestrictionType":
            return Optional.ofNullable(clazz.cast(restrictionTypeAsString()));
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(items()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GeoRestriction, T> g) {
        return obj -> g.apply((GeoRestriction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GeoRestriction> {
        /**
         * <p>
         * The method that you want to use to restrict distribution of your content by country:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by client geo
         * location.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
         * CloudFront to distribute your content.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront
         * to distribute your content.
         * </p>
         * </li>
         * </ul>
         * 
         * @param restrictionType
         *        The method that you want to use to restrict distribution of your content by country:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by
         *        client geo location.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't
         *        want CloudFront to distribute your content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
         *        CloudFront to distribute your content.
         *        </p>
         *        </li>
         * @see GeoRestrictionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeoRestrictionType
         */
        Builder restrictionType(String restrictionType);

        /**
         * <p>
         * The method that you want to use to restrict distribution of your content by country:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by client geo
         * location.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
         * CloudFront to distribute your content.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront
         * to distribute your content.
         * </p>
         * </li>
         * </ul>
         * 
         * @param restrictionType
         *        The method that you want to use to restrict distribution of your content by country:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>none</code>: No geo restriction is enabled, meaning access to content is not restricted by
         *        client geo location.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't
         *        want CloudFront to distribute your content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
         *        CloudFront to distribute your content.
         *        </p>
         *        </li>
         * @see GeoRestrictionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeoRestrictionType
         */
        Builder restrictionType(GeoRestrictionType restrictionType);

        /**
         * <p>
         * When geo restriction is <code>enabled</code>, this is the number of countries in your <code>whitelist</code>
         * or <code>blacklist</code>. Otherwise, when it is not enabled, <code>Quantity</code> is <code>0</code>, and
         * you can omit <code>Items</code>.
         * </p>
         * 
         * @param quantity
         *        When geo restriction is <code>enabled</code>, this is the number of countries in your
         *        <code>whitelist</code> or <code>blacklist</code>. Otherwise, when it is not enabled,
         *        <code>Quantity</code> is <code>0</code>, and you can omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains a <code>Location</code> element for each country in which you want CloudFront
         * either to distribute your content (<code>whitelist</code>) or not distribute your content (
         * <code>blacklist</code>).
         * </p>
         * <p>
         * The <code>Location</code> element is a two-letter, uppercase country code for a country that you want to
         * include in your <code>blacklist</code> or <code>whitelist</code>. Include one <code>Location</code> element
         * for each country.
         * </p>
         * <p>
         * CloudFront and <code>MaxMind</code> both use <code>ISO 3166</code> country codes. For the current list of
         * countries and the corresponding codes, see <code>ISO 3166-1-alpha-2</code> code on the <i>International
         * Organization for Standardization</i> website. You can also refer to the country list on the CloudFront
         * console, which includes both country names and codes.
         * </p>
         * 
         * @param items
         *        A complex type that contains a <code>Location</code> element for each country in which you want
         *        CloudFront either to distribute your content (<code>whitelist</code>) or not distribute your content (
         *        <code>blacklist</code>).</p>
         *        <p>
         *        The <code>Location</code> element is a two-letter, uppercase country code for a country that you want
         *        to include in your <code>blacklist</code> or <code>whitelist</code>. Include one <code>Location</code>
         *        element for each country.
         *        </p>
         *        <p>
         *        CloudFront and <code>MaxMind</code> both use <code>ISO 3166</code> country codes. For the current list
         *        of countries and the corresponding codes, see <code>ISO 3166-1-alpha-2</code> code on the
         *        <i>International Organization for Standardization</i> website. You can also refer to the country list
         *        on the CloudFront console, which includes both country names and codes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * A complex type that contains a <code>Location</code> element for each country in which you want CloudFront
         * either to distribute your content (<code>whitelist</code>) or not distribute your content (
         * <code>blacklist</code>).
         * </p>
         * <p>
         * The <code>Location</code> element is a two-letter, uppercase country code for a country that you want to
         * include in your <code>blacklist</code> or <code>whitelist</code>. Include one <code>Location</code> element
         * for each country.
         * </p>
         * <p>
         * CloudFront and <code>MaxMind</code> both use <code>ISO 3166</code> country codes. For the current list of
         * countries and the corresponding codes, see <code>ISO 3166-1-alpha-2</code> code on the <i>International
         * Organization for Standardization</i> website. You can also refer to the country list on the CloudFront
         * console, which includes both country names and codes.
         * </p>
         * 
         * @param items
         *        A complex type that contains a <code>Location</code> element for each country in which you want
         *        CloudFront either to distribute your content (<code>whitelist</code>) or not distribute your content (
         *        <code>blacklist</code>).</p>
         *        <p>
         *        The <code>Location</code> element is a two-letter, uppercase country code for a country that you want
         *        to include in your <code>blacklist</code> or <code>whitelist</code>. Include one <code>Location</code>
         *        element for each country.
         *        </p>
         *        <p>
         *        CloudFront and <code>MaxMind</code> both use <code>ISO 3166</code> country codes. For the current list
         *        of countries and the corresponding codes, see <code>ISO 3166-1-alpha-2</code> code on the
         *        <i>International Organization for Standardization</i> website. You can also refer to the country list
         *        on the CloudFront console, which includes both country names and codes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    static final class BuilderImpl implements Builder {
        private String restrictionType;

        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GeoRestriction model) {
            restrictionType(model.restrictionType);
            quantity(model.quantity);
            items(model.items);
        }

        public final String getRestrictionType() {
            return restrictionType;
        }

        @Override
        public final Builder restrictionType(String restrictionType) {
            this.restrictionType = restrictionType;
            return this;
        }

        @Override
        public final Builder restrictionType(GeoRestrictionType restrictionType) {
            this.restrictionType(restrictionType == null ? null : restrictionType.toString());
            return this;
        }

        public final void setRestrictionType(String restrictionType) {
            this.restrictionType = restrictionType;
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = LocationListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = LocationListCopier.copy(items);
        }

        @Override
        public GeoRestriction build() {
            return new GeoRestriction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
