/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloud9;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloud9.model.BadRequestException;
import software.amazon.awssdk.services.cloud9.model.Cloud9Exception;
import software.amazon.awssdk.services.cloud9.model.Cloud9Request;
import software.amazon.awssdk.services.cloud9.model.ConflictException;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentEc2Request;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentEc2Response;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentStatusRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentStatusResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentsResponse;
import software.amazon.awssdk.services.cloud9.model.ForbiddenException;
import software.amazon.awssdk.services.cloud9.model.InternalServerErrorException;
import software.amazon.awssdk.services.cloud9.model.LimitExceededException;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.cloud9.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloud9.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloud9.model.NotFoundException;
import software.amazon.awssdk.services.cloud9.model.TagResourceRequest;
import software.amazon.awssdk.services.cloud9.model.TagResourceResponse;
import software.amazon.awssdk.services.cloud9.model.TooManyRequestsException;
import software.amazon.awssdk.services.cloud9.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloud9.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsPublisher;
import software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsPublisher;
import software.amazon.awssdk.services.cloud9.transform.CreateEnvironmentEc2RequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.CreateEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DeleteEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentMembershipsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentStatusRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UpdateEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Cloud9AsyncClient}.
 *
 * @see Cloud9AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloud9AsyncClient implements Cloud9AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloud9AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloud9AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance,
     * and then connects from the instance to the environment.
     * </p>
     *
     * @param createEnvironmentEc2Request
     * @return A Java Future containing the result of the CreateEnvironmentEC2 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.CreateEnvironmentEC2
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/CreateEnvironmentEC2" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentEc2Response> createEnvironmentEC2(
            CreateEnvironmentEc2Request createEnvironmentEc2Request) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentEc2Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentEc2Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentEc2Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentEc2Request, CreateEnvironmentEc2Response>()
                            .withOperationName("CreateEnvironmentEC2")
                            .withMarshaller(new CreateEnvironmentEc2RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createEnvironmentEc2Request));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds an environment member to an AWS Cloud9 development environment.
     * </p>
     *
     * @param createEnvironmentMembershipRequest
     * @return A Java Future containing the result of the CreateEnvironmentMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.CreateEnvironmentMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/CreateEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentMembershipResponse> createEnvironmentMembership(
            CreateEnvironmentMembershipRequest createEnvironmentMembershipRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentMembershipRequest, CreateEnvironmentMembershipResponse>()
                            .withOperationName("CreateEnvironmentMembership")
                            .withMarshaller(new CreateEnvironmentMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createEnvironmentMembershipRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AWS Cloud9 development environment. If an Amazon EC2 instance is connected to the environment, also
     * terminates the instance.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DeleteEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment")
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEnvironmentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an environment member from an AWS Cloud9 development environment.
     * </p>
     *
     * @param deleteEnvironmentMembershipRequest
     * @return A Java Future containing the result of the DeleteEnvironmentMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DeleteEnvironmentMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DeleteEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentMembershipResponse> deleteEnvironmentMembership(
            DeleteEnvironmentMembershipRequest deleteEnvironmentMembershipRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentMembershipRequest, DeleteEnvironmentMembershipResponse>()
                            .withOperationName("DeleteEnvironmentMembership")
                            .withMarshaller(new DeleteEnvironmentMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEnvironmentMembershipRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about environment members for an AWS Cloud9 development environment.
     * </p>
     *
     * @param describeEnvironmentMembershipsRequest
     * @return A Java Future containing the result of the DescribeEnvironmentMemberships operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironmentMemberships
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEnvironmentMembershipsResponse> describeEnvironmentMemberships(
            DescribeEnvironmentMembershipsRequest describeEnvironmentMembershipsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEnvironmentMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEnvironmentMembershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEnvironmentMembershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentMembershipsRequest, DescribeEnvironmentMembershipsResponse>()
                            .withOperationName("DescribeEnvironmentMemberships")
                            .withMarshaller(new DescribeEnvironmentMembershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEnvironmentMembershipsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about environment members for an AWS Cloud9 development environment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEnvironmentMemberships(software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsPublisher publisher = client.describeEnvironmentMembershipsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.DescribeEnvironmentMembershipsPublisher publisher = client.describeEnvironmentMembershipsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEnvironmentMemberships(software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEnvironmentMembershipsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironmentMemberships
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeEnvironmentMembershipsPublisher describeEnvironmentMembershipsPaginator(
            DescribeEnvironmentMembershipsRequest describeEnvironmentMembershipsRequest) {
        return new DescribeEnvironmentMembershipsPublisher(this, applyPaginatorUserAgent(describeEnvironmentMembershipsRequest));
    }

    /**
     * <p>
     * Gets status information for an AWS Cloud9 development environment.
     * </p>
     *
     * @param describeEnvironmentStatusRequest
     * @return A Java Future containing the result of the DescribeEnvironmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEnvironmentStatusResponse> describeEnvironmentStatus(
            DescribeEnvironmentStatusRequest describeEnvironmentStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEnvironmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEnvironmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEnvironmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentStatusRequest, DescribeEnvironmentStatusResponse>()
                            .withOperationName("DescribeEnvironmentStatus")
                            .withMarshaller(new DescribeEnvironmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEnvironmentStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about AWS Cloud9 development environments.
     * </p>
     *
     * @param describeEnvironmentsRequest
     * @return A Java Future containing the result of the DescribeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEnvironmentsResponse> describeEnvironments(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentsRequest, DescribeEnvironmentsResponse>()
                            .withOperationName("DescribeEnvironments")
                            .withMarshaller(new DescribeEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEnvironmentsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of AWS Cloud9 development environment identifiers.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments")
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEnvironmentsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of AWS Cloud9 development environment identifiers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    public ListEnvironmentsPublisher listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest) {
        return new ListEnvironmentsPublisher(this, applyPaginatorUserAgent(listEnvironmentsRequest));
    }

    /**
     * <p>
     * Gets a list of the tags associated with an AWS Cloud9 development environment.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to an AWS Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Tags that you add to an AWS Cloud9 environment by using this method will NOT be automatically propagated to
     * underlying resources.
     * </p>
     * </important>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an AWS Cloud9 development environment.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the settings of an existing AWS Cloud9 development environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return A Java Future containing the result of the UpdateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.UpdateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentResponse> updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                            .withOperationName("UpdateEnvironment")
                            .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateEnvironmentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the settings of an existing environment member for an AWS Cloud9 development environment.
     * </p>
     *
     * @param updateEnvironmentMembershipRequest
     * @return A Java Future containing the result of the UpdateEnvironmentMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.UpdateEnvironmentMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UpdateEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentMembershipResponse> updateEnvironmentMembership(
            UpdateEnvironmentMembershipRequest updateEnvironmentMembershipRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentMembershipRequest, UpdateEnvironmentMembershipResponse>()
                            .withOperationName("UpdateEnvironmentMembership")
                            .withMarshaller(new UpdateEnvironmentMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateEnvironmentMembershipRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Cloud9Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private <T extends Cloud9Request> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
