/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockruntime.internal.BedrockRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockruntime.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.bedrockruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailResponse;
import software.amazon.awssdk.services.bedrockruntime.model.BedrockRuntimeException;
import software.amazon.awssdk.services.bedrockruntime.model.ConflictException;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseResponse;
import software.amazon.awssdk.services.bedrockruntime.model.CountTokensRequest;
import software.amazon.awssdk.services.bedrockruntime.model.CountTokensResponse;
import software.amazon.awssdk.services.bedrockruntime.model.GetAsyncInvokeRequest;
import software.amazon.awssdk.services.bedrockruntime.model.GetAsyncInvokeResponse;
import software.amazon.awssdk.services.bedrockruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ModelErrorException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelNotReadyException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelStreamErrorException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelTimeoutException;
import software.amazon.awssdk.services.bedrockruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockruntime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.bedrockruntime.model.StartAsyncInvokeRequest;
import software.amazon.awssdk.services.bedrockruntime.model.StartAsyncInvokeResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockruntime.transform.ApplyGuardrailRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.ConverseRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.CountTokensRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.GetAsyncInvokeRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.InvokeModelRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.ListAsyncInvokesRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.StartAsyncInvokeRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BedrockRuntimeClient}.
 *
 * @see BedrockRuntimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockRuntimeClient implements BedrockRuntimeClient {
    private static final Logger log = Logger.loggerFor(DefaultBedrockRuntimeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBedrockRuntimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Bedrock_Runtime" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * The action to apply a guardrail.
     * </p>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>ApplyGuardrail</code> API,
     * see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     *
     * @param applyGuardrailRequest
     * @return Result of the ApplyGuardrail operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ApplyGuardrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ApplyGuardrail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ApplyGuardrailResponse applyGuardrail(ApplyGuardrailRequest applyGuardrailRequest) throws AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, ServiceUnavailableException,
            ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ApplyGuardrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ApplyGuardrailResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(applyGuardrailRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, applyGuardrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ApplyGuardrail");

            return clientHandler.execute(new ClientExecutionParams<ApplyGuardrailRequest, ApplyGuardrailResponse>()
                    .withOperationName("ApplyGuardrail").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(applyGuardrailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ApplyGuardrailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model. <code>Converse</code> provides a consistent interface that
     * works with all models that support messages. This allows you to write code once and use it with different models.
     * If a model has unique inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * You can submit a prompt by including it in the <code>messages</code> field, specifying the <code>modelId</code>
     * of a foundation model or inference profile to run inference on it, and including any other fields that are
     * relevant to your use case.
     * </p>
     * <p>
     * You can also submit a prompt from Prompt management by specifying the ARN of the prompt version and including a
     * map of variables to values in the <code>promptVariables</code> field. You can append more messages to the prompt
     * by using the <code>messages</code> field. If you use a prompt from Prompt management, you can't include the
     * following fields in the request: <code>additionalModelRequestFields</code>, <code>inferenceConfig</code>,
     * <code>system</code>, or <code>toolConfig</code>. Instead, these fields must be defined through Prompt management.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-management-use.html">Use a prompt from Prompt
     * management</a>.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Converse API examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the base inference actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html">InvokeModel</a> and
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModelWithResponseStream.html">
     * InvokeModelWithResponseStream</a>). For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>Converse</code> API, see
     * <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     *
     * @param converseRequest
     * @return Result of the Converse operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ModelTimeoutException
     *         The request took too long to process. Processing time exceeded the model timeout length.
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ModelNotReadyException
     *         The model specified in the request is not ready to serve inference requests. The AWS SDK will
     *         automatically retry the operation up to 5 times. For information about configuring automatic retries, see
     *         <a href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a>
     *         in the <i>AWS SDKs and Tools</i> reference guide.
     * @throws ModelErrorException
     *         The request failed due to an error while processing the model.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.Converse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/Converse" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ConverseResponse converse(ConverseRequest converseRequest) throws AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, ModelTimeoutException, InternalServerException, ServiceUnavailableException,
            ValidationException, ModelNotReadyException, ModelErrorException, AwsServiceException, SdkClientException,
            BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ConverseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ConverseResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(converseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, converseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Converse");

            return clientHandler.execute(new ClientExecutionParams<ConverseRequest, ConverseResponse>()
                    .withOperationName("Converse").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(converseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ConverseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the token count for a given inference request. This operation helps you estimate token usage before
     * sending requests to foundation models by returning the token count that would be used if the same input were sent
     * to the model in an inference request.
     * </p>
     * <p>
     * Token counting is model-specific because different models use different tokenization strategies. The token count
     * returned by this operation will match the token count that would be charged if the same input were sent to the
     * model in an <code>InvokeModel</code> or <code>Converse</code> request.
     * </p>
     * <p>
     * You can use this operation to:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Estimate costs before sending inference requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optimize prompts to fit within token limits.
     * </p>
     * </li>
     * <li>
     * <p>
     * Plan for token usage in your applications.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation accepts the same input formats as <code>InvokeModel</code> and <code>Converse</code>, allowing you
     * to count tokens for both raw text inputs and structured conversation formats.
     * </p>
     * <p>
     * The following operations are related to <code>CountTokens</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/API/API_runtime_InvokeModel.html">InvokeModel</a> - Sends
     * inference requests to foundation models
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/API/API_runtime_Converse.html">Converse</a> - Sends
     * conversation-based inference requests to foundation models
     * </p>
     * </li>
     * </ul>
     *
     * @param countTokensRequest
     * @return Result of the CountTokens operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.CountTokens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/CountTokens" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CountTokensResponse countTokens(CountTokensRequest countTokensRequest) throws AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, ServiceUnavailableException,
            ValidationException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CountTokensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CountTokensResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(countTokensRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, countTokensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CountTokens");

            return clientHandler.execute(new ClientExecutionParams<CountTokensRequest, CountTokensResponse>()
                    .withOperationName("CountTokens").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(countTokensRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CountTokensRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve information about an asynchronous invocation.
     * </p>
     *
     * @param getAsyncInvokeRequest
     * @return Result of the GetAsyncInvoke operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.GetAsyncInvoke
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/GetAsyncInvoke"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAsyncInvokeResponse getAsyncInvoke(GetAsyncInvokeRequest getAsyncInvokeRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAsyncInvokeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAsyncInvokeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAsyncInvokeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAsyncInvokeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAsyncInvoke");

            return clientHandler.execute(new ClientExecutionParams<GetAsyncInvokeRequest, GetAsyncInvokeResponse>()
                    .withOperationName("GetAsyncInvoke").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAsyncInvokeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAsyncInvokeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the Converse API actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html">Converse</a> and <a
     * href=
     * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html">ConverseStream</a>).
     * For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>InvokeModel</code> API,
     * see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     *
     * @param invokeModelRequest
     * @return Result of the InvokeModel operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ModelTimeoutException
     *         The request took too long to process. Processing time exceeded the model timeout length.
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws ModelNotReadyException
     *         The model specified in the request is not ready to serve inference requests. The AWS SDK will
     *         automatically retry the operation up to 5 times. For information about configuring automatic retries, see
     *         <a href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a>
     *         in the <i>AWS SDKs and Tools</i> reference guide.
     * @throws ModelErrorException
     *         The request failed due to an error while processing the model.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InvokeModelResponse invokeModel(InvokeModelRequest invokeModelRequest) throws AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, ModelTimeoutException, InternalServerException,
            ServiceUnavailableException, ValidationException, ServiceQuotaExceededException, ModelNotReadyException,
            ModelErrorException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<InvokeModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InvokeModelResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeModel");

            return clientHandler.execute(new ClientExecutionParams<InvokeModelRequest, InvokeModelResponse>()
                    .withOperationName("InvokeModel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(invokeModelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InvokeModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists asynchronous invocations.
     * </p>
     *
     * @param listAsyncInvokesRequest
     * @return Result of the ListAsyncInvokes operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ListAsyncInvokes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ListAsyncInvokes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAsyncInvokesResponse listAsyncInvokes(ListAsyncInvokesRequest listAsyncInvokesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAsyncInvokesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAsyncInvokesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAsyncInvokesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAsyncInvokesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAsyncInvokes");

            return clientHandler.execute(new ClientExecutionParams<ListAsyncInvokesRequest, ListAsyncInvokesResponse>()
                    .withOperationName("ListAsyncInvokes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAsyncInvokesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAsyncInvokesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an asynchronous invocation.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the Converse API actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html">Converse</a> and <a
     * href=
     * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html">ConverseStream</a>).
     * For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     *
     * @param startAsyncInvokeRequest
     * @return Result of the StartAsyncInvoke operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.StartAsyncInvoke
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/StartAsyncInvoke"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartAsyncInvokeResponse startAsyncInvoke(StartAsyncInvokeRequest startAsyncInvokeRequest)
            throws AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException,
            ServiceUnavailableException, ValidationException, ServiceQuotaExceededException, ConflictException,
            AwsServiceException, SdkClientException, BedrockRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartAsyncInvokeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartAsyncInvokeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ModelErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelErrorException::builder).build());
            case "ModelNotReadyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ModelStreamErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelStreamErrorException").httpStatusCode(424)
                        .exceptionBuilderSupplier(ModelStreamErrorException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ModelTimeoutException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ModelTimeoutException").httpStatusCode(408)
                        .exceptionBuilderSupplier(ModelTimeoutException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startAsyncInvokeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAsyncInvokeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartAsyncInvoke");

            return clientHandler.execute(new ClientExecutionParams<StartAsyncInvokeRequest, StartAsyncInvokeResponse>()
                    .withOperationName("StartAsyncInvoke").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startAsyncInvokeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartAsyncInvokeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BedrockRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(BedrockRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final BedrockRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
