/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.bedrockruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailResponse;
import software.amazon.awssdk.services.bedrockruntime.model.BedrockRuntimeException;
import software.amazon.awssdk.services.bedrockruntime.model.ConflictException;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseResponse;
import software.amazon.awssdk.services.bedrockruntime.model.GetAsyncInvokeRequest;
import software.amazon.awssdk.services.bedrockruntime.model.GetAsyncInvokeResponse;
import software.amazon.awssdk.services.bedrockruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ModelErrorException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelNotReadyException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelTimeoutException;
import software.amazon.awssdk.services.bedrockruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockruntime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.bedrockruntime.model.StartAsyncInvokeRequest;
import software.amazon.awssdk.services.bedrockruntime.model.StartAsyncInvokeResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable;

/**
 * Service client for accessing Amazon Bedrock Runtime. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Describes the API operations for running inference using Amazon Bedrock models.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BedrockRuntimeClient extends AwsClient {
    String SERVICE_NAME = "bedrock";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "bedrock-runtime";

    /**
     * <p>
     * The action to apply a guardrail.
     * </p>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>ApplyGuardrail</code> API,
     * see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     *
     * @param applyGuardrailRequest
     * @return Result of the ApplyGuardrail operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ApplyGuardrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ApplyGuardrail"
     *      target="_top">AWS API Documentation</a>
     */
    default ApplyGuardrailResponse applyGuardrail(ApplyGuardrailRequest applyGuardrailRequest) throws AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The action to apply a guardrail.
     * </p>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>ApplyGuardrail</code> API,
     * see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ApplyGuardrailRequest.Builder} avoiding the need to
     * create one manually via {@link ApplyGuardrailRequest#builder()}
     * </p>
     *
     * @param applyGuardrailRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailRequest.Builder} to create a
     *        request.
     * @return Result of the ApplyGuardrail operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ApplyGuardrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ApplyGuardrail"
     *      target="_top">AWS API Documentation</a>
     */
    default ApplyGuardrailResponse applyGuardrail(Consumer<ApplyGuardrailRequest.Builder> applyGuardrailRequest)
            throws AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        return applyGuardrail(ApplyGuardrailRequest.builder().applyMutation(applyGuardrailRequest).build());
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model. <code>Converse</code> provides a consistent interface that
     * works with all models that support messages. This allows you to write code once and use it with different models.
     * If a model has unique inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * You can submit a prompt by including it in the <code>messages</code> field, specifying the <code>modelId</code>
     * of a foundation model or inference profile to run inference on it, and including any other fields that are
     * relevant to your use case.
     * </p>
     * <p>
     * You can also submit a prompt from Prompt management by specifying the ARN of the prompt version and including a
     * map of variables to values in the <code>promptVariables</code> field. You can append more messages to the prompt
     * by using the <code>messages</code> field. If you use a prompt from Prompt management, you can't include the
     * following fields in the request: <code>additionalModelRequestFields</code>, <code>inferenceConfig</code>,
     * <code>system</code>, or <code>toolConfig</code>. Instead, these fields must be defined through Prompt management.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-management-use.html">Use a prompt from Prompt
     * management</a>.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Converse API examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the base inference actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html">InvokeModel</a> and
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModelWithResponseStream.html">
     * InvokeModelWithResponseStream</a>). For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>Converse</code> API, see
     * <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     *
     * @param converseRequest
     * @return Result of the Converse operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ModelTimeoutException
     *         The request took too long to process. Processing time exceeded the model timeout length.
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ModelNotReadyException
     *         The model specified in the request is not ready to serve inference requests. The AWS SDK will
     *         automatically retry the operation up to 5 times. For information about configuring automatic retries, see
     *         <a href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a>
     *         in the <i>AWS SDKs and Tools</i> reference guide.
     * @throws ModelErrorException
     *         The request failed due to an error while processing the model.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.Converse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/Converse" target="_top">AWS API
     *      Documentation</a>
     */
    default ConverseResponse converse(ConverseRequest converseRequest) throws AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, ModelTimeoutException, InternalServerException, ServiceUnavailableException,
            ValidationException, ModelNotReadyException, ModelErrorException, AwsServiceException, SdkClientException,
            BedrockRuntimeException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model. <code>Converse</code> provides a consistent interface that
     * works with all models that support messages. This allows you to write code once and use it with different models.
     * If a model has unique inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * You can submit a prompt by including it in the <code>messages</code> field, specifying the <code>modelId</code>
     * of a foundation model or inference profile to run inference on it, and including any other fields that are
     * relevant to your use case.
     * </p>
     * <p>
     * You can also submit a prompt from Prompt management by specifying the ARN of the prompt version and including a
     * map of variables to values in the <code>promptVariables</code> field. You can append more messages to the prompt
     * by using the <code>messages</code> field. If you use a prompt from Prompt management, you can't include the
     * following fields in the request: <code>additionalModelRequestFields</code>, <code>inferenceConfig</code>,
     * <code>system</code>, or <code>toolConfig</code>. Instead, these fields must be defined through Prompt management.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-management-use.html">Use a prompt from Prompt
     * management</a>.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Converse API examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the base inference actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html">InvokeModel</a> and
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModelWithResponseStream.html">
     * InvokeModelWithResponseStream</a>). For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>Converse</code> API, see
     * <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ConverseRequest.Builder} avoiding the need to
     * create one manually via {@link ConverseRequest#builder()}
     * </p>
     *
     * @param converseRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ConverseRequest.Builder} to create a request.
     * @return Result of the Converse operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ModelTimeoutException
     *         The request took too long to process. Processing time exceeded the model timeout length.
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ModelNotReadyException
     *         The model specified in the request is not ready to serve inference requests. The AWS SDK will
     *         automatically retry the operation up to 5 times. For information about configuring automatic retries, see
     *         <a href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a>
     *         in the <i>AWS SDKs and Tools</i> reference guide.
     * @throws ModelErrorException
     *         The request failed due to an error while processing the model.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.Converse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/Converse" target="_top">AWS API
     *      Documentation</a>
     */
    default ConverseResponse converse(Consumer<ConverseRequest.Builder> converseRequest) throws AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, ModelTimeoutException, InternalServerException,
            ServiceUnavailableException, ValidationException, ModelNotReadyException, ModelErrorException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        return converse(ConverseRequest.builder().applyMutation(converseRequest).build());
    }

    /**
     * <p>
     * Retrieve information about an asynchronous invocation.
     * </p>
     *
     * @param getAsyncInvokeRequest
     * @return Result of the GetAsyncInvoke operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.GetAsyncInvoke
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/GetAsyncInvoke"
     *      target="_top">AWS API Documentation</a>
     */
    default GetAsyncInvokeResponse getAsyncInvoke(GetAsyncInvokeRequest getAsyncInvokeRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            BedrockRuntimeException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieve information about an asynchronous invocation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAsyncInvokeRequest.Builder} avoiding the need to
     * create one manually via {@link GetAsyncInvokeRequest#builder()}
     * </p>
     *
     * @param getAsyncInvokeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.GetAsyncInvokeRequest.Builder} to create a
     *        request.
     * @return Result of the GetAsyncInvoke operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.GetAsyncInvoke
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/GetAsyncInvoke"
     *      target="_top">AWS API Documentation</a>
     */
    default GetAsyncInvokeResponse getAsyncInvoke(Consumer<GetAsyncInvokeRequest.Builder> getAsyncInvokeRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        return getAsyncInvoke(GetAsyncInvokeRequest.builder().applyMutation(getAsyncInvokeRequest).build());
    }

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the Converse API actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html">Converse</a> and <a
     * href=
     * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html">ConverseStream</a>).
     * For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>InvokeModel</code> API,
     * see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     *
     * @param invokeModelRequest
     * @return Result of the InvokeModel operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ModelTimeoutException
     *         The request took too long to process. Processing time exceeded the model timeout length.
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ModelNotReadyException
     *         The model specified in the request is not ready to serve inference requests. The AWS SDK will
     *         automatically retry the operation up to 5 times. For information about configuring automatic retries, see
     *         <a href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a>
     *         in the <i>AWS SDKs and Tools</i> reference guide.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws ModelErrorException
     *         The request failed due to an error while processing the model.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    default InvokeModelResponse invokeModel(InvokeModelRequest invokeModelRequest) throws AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, ModelTimeoutException, InternalServerException,
            ServiceUnavailableException, ValidationException, ModelNotReadyException, ServiceQuotaExceededException,
            ModelErrorException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the Converse API actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html">Converse</a> and <a
     * href=
     * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html">ConverseStream</a>).
     * For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     * <p>
     * For troubleshooting some of the common errors you might encounter when using the <code>InvokeModel</code> API,
     * see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html">
     * Troubleshooting Amazon Bedrock API Error Codes</a> in the Amazon Bedrock User Guide
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link InvokeModelRequest.Builder} avoiding the need to
     * create one manually via {@link InvokeModelRequest#builder()}
     * </p>
     *
     * @param invokeModelRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest.Builder} to create a
     *        request.
     * @return Result of the InvokeModel operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws ModelTimeoutException
     *         The request took too long to process. Processing time exceeded the model timeout length.
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ModelNotReadyException
     *         The model specified in the request is not ready to serve inference requests. The AWS SDK will
     *         automatically retry the operation up to 5 times. For information about configuring automatic retries, see
     *         <a href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a>
     *         in the <i>AWS SDKs and Tools</i> reference guide.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws ModelErrorException
     *         The request failed due to an error while processing the model.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    default InvokeModelResponse invokeModel(Consumer<InvokeModelRequest.Builder> invokeModelRequest)
            throws AccessDeniedException, ResourceNotFoundException, ThrottlingException, ModelTimeoutException,
            InternalServerException, ServiceUnavailableException, ValidationException, ModelNotReadyException,
            ServiceQuotaExceededException, ModelErrorException, AwsServiceException, SdkClientException, BedrockRuntimeException {
        return invokeModel(InvokeModelRequest.builder().applyMutation(invokeModelRequest).build());
    }

    /**
     * <p>
     * Lists asynchronous invocations.
     * </p>
     *
     * @param listAsyncInvokesRequest
     * @return Result of the ListAsyncInvokes operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ListAsyncInvokes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ListAsyncInvokes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAsyncInvokesResponse listAsyncInvokes(ListAsyncInvokesRequest listAsyncInvokesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists asynchronous invocations.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAsyncInvokesRequest.Builder} avoiding the need
     * to create one manually via {@link ListAsyncInvokesRequest#builder()}
     * </p>
     *
     * @param listAsyncInvokesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest.Builder} to create a
     *        request.
     * @return Result of the ListAsyncInvokes operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ListAsyncInvokes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ListAsyncInvokes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAsyncInvokesResponse listAsyncInvokes(Consumer<ListAsyncInvokesRequest.Builder> listAsyncInvokesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        return listAsyncInvokes(ListAsyncInvokesRequest.builder().applyMutation(listAsyncInvokesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAsyncInvokes(software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable responses = client.listAsyncInvokesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable responses = client
     *             .listAsyncInvokesPaginator(request);
     *     for (software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable responses = client.listAsyncInvokesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAsyncInvokes(software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAsyncInvokesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ListAsyncInvokes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ListAsyncInvokes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAsyncInvokesIterable listAsyncInvokesPaginator(ListAsyncInvokesRequest listAsyncInvokesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        return new ListAsyncInvokesIterable(this, listAsyncInvokesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAsyncInvokes(software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable responses = client.listAsyncInvokesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable responses = client
     *             .listAsyncInvokesPaginator(request);
     *     for (software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bedrockruntime.paginators.ListAsyncInvokesIterable responses = client.listAsyncInvokesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAsyncInvokes(software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAsyncInvokesRequest.Builder} avoiding the need
     * to create one manually via {@link ListAsyncInvokesRequest#builder()}
     * </p>
     *
     * @param listAsyncInvokesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ListAsyncInvokesRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.ListAsyncInvokes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ListAsyncInvokes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAsyncInvokesIterable listAsyncInvokesPaginator(Consumer<ListAsyncInvokesRequest.Builder> listAsyncInvokesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BedrockRuntimeException {
        return listAsyncInvokesPaginator(ListAsyncInvokesRequest.builder().applyMutation(listAsyncInvokesRequest).build());
    }

    /**
     * <p>
     * Starts an asynchronous invocation.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the Converse API actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html">Converse</a> and <a
     * href=
     * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html">ConverseStream</a>).
     * For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important>
     *
     * @param startAsyncInvokeRequest
     * @return Result of the StartAsyncInvoke operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.StartAsyncInvoke
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/StartAsyncInvoke"
     *      target="_top">AWS API Documentation</a>
     */
    default StartAsyncInvokeResponse startAsyncInvoke(StartAsyncInvokeRequest startAsyncInvokeRequest)
            throws AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            ServiceUnavailableException, ValidationException, ServiceQuotaExceededException, ConflictException,
            AwsServiceException, SdkClientException, BedrockRuntimeException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts an asynchronous invocation.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <important>
     * <p>
     * To deny all inference access to resources that you specify in the modelId field, you need to deny access to the
     * <code>bedrock:InvokeModel</code> and <code>bedrock:InvokeModelWithResponseStream</code> actions. Doing this also
     * denies access to the resource through the Converse API actions (<a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html">Converse</a> and <a
     * href=
     * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html">ConverseStream</a>).
     * For more information see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-deny-inference"
     * >Deny access for inference on specific models</a>.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartAsyncInvokeRequest.Builder} avoiding the need
     * to create one manually via {@link StartAsyncInvokeRequest#builder()}
     * </p>
     *
     * @param startAsyncInvokeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.StartAsyncInvokeRequest.Builder} to create a
     *        request.
     * @return Result of the StartAsyncInvoke operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because you do not have sufficient permissions to perform the requested action. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-access-denied"
     *         >AccessDeniedException</a> in the Amazon Bedrock User Guide
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-resource-not-found"
     *         >ResourceNotFound</a> in the Amazon Bedrock User Guide
     * @throws ThrottlingException
     *         Your request was denied due to exceeding the account quotas for <i>Amazon Bedrock</i>. For
     *         troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-throttling-exception"
     *         >ThrottlingException</a> in the Amazon Bedrock User Guide
     * @throws InternalServerException
     *         An internal server error occurred. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-internal-failure"
     *         >InternalFailure</a> in the Amazon Bedrock User Guide
     * @throws ServiceUnavailableException
     *         The service isn't currently available. For troubleshooting this error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-service-unavailable"
     *         >ServiceUnavailable</a> in the Amazon Bedrock User Guide
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by <i>Amazon Bedrock</i>. For troubleshooting this
     *         error, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/troubleshooting-api-error-codes.html#ts-validation-error"
     *         >ValidationError</a> in the Amazon Bedrock User Guide
     * @throws ServiceQuotaExceededException
     *         Your request exceeds the service quota for your account. You can view your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockRuntimeClient.StartAsyncInvoke
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/StartAsyncInvoke"
     *      target="_top">AWS API Documentation</a>
     */
    default StartAsyncInvokeResponse startAsyncInvoke(Consumer<StartAsyncInvokeRequest.Builder> startAsyncInvokeRequest)
            throws AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            ServiceUnavailableException, ValidationException, ServiceQuotaExceededException, ConflictException,
            AwsServiceException, SdkClientException, BedrockRuntimeException {
        return startAsyncInvoke(StartAsyncInvokeRequest.builder().applyMutation(startAsyncInvokeRequest).build());
    }

    /**
     * Create a {@link BedrockRuntimeClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BedrockRuntimeClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BedrockRuntimeClient}.
     */
    static BedrockRuntimeClientBuilder builder() {
        return new DefaultBedrockRuntimeClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default BedrockRuntimeServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
