/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the image data for multimodal knowledge base queries, including format and content.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Retrieve.html#API_agent-runtime_Retrieve_RequestSyntax"
 * >Retrieve request</a> – in the <code>image</code> field
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InputImage implements SdkPojo, Serializable, ToCopyableBuilder<InputImage.Builder, InputImage> {
    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(InputImage::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<SdkBytes> INLINE_CONTENT_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("inlineContent").getter(getter(InputImage::inlineContent)).setter(setter(Builder::inlineContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inlineContent").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORMAT_FIELD,
            INLINE_CONTENT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String format;

    private final SdkBytes inlineContent;

    private InputImage(BuilderImpl builder) {
        this.format = builder.format;
        this.inlineContent = builder.inlineContent;
    }

    /**
     * <p>
     * The format of the input image. Supported formats include png, gif, jpeg, and webp.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link InputImageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the input image. Supported formats include png, gif, jpeg, and webp.
     * @see InputImageFormat
     */
    public final InputImageFormat format() {
        return InputImageFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the input image. Supported formats include png, gif, jpeg, and webp.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link InputImageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the input image. Supported formats include png, gif, jpeg, and webp.
     * @see InputImageFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The base64-encoded image data for inline image content. Maximum size is 5MB.
     * </p>
     * 
     * @return The base64-encoded image data for inline image content. Maximum size is 5MB.
     */
    public final SdkBytes inlineContent() {
        return inlineContent;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inlineContent());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InputImage)) {
            return false;
        }
        InputImage other = (InputImage) obj;
        return Objects.equals(formatAsString(), other.formatAsString()) && Objects.equals(inlineContent(), other.inlineContent());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InputImage").add("Format", formatAsString())
                .add("InlineContent", inlineContent() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "inlineContent":
            return Optional.ofNullable(clazz.cast(inlineContent()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("format", FORMAT_FIELD);
        map.put("inlineContent", INLINE_CONTENT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InputImage, T> g) {
        return obj -> g.apply((InputImage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InputImage> {
        /**
         * <p>
         * The format of the input image. Supported formats include png, gif, jpeg, and webp.
         * </p>
         * 
         * @param format
         *        The format of the input image. Supported formats include png, gif, jpeg, and webp.
         * @see InputImageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputImageFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the input image. Supported formats include png, gif, jpeg, and webp.
         * </p>
         * 
         * @param format
         *        The format of the input image. Supported formats include png, gif, jpeg, and webp.
         * @see InputImageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputImageFormat
         */
        Builder format(InputImageFormat format);

        /**
         * <p>
         * The base64-encoded image data for inline image content. Maximum size is 5MB.
         * </p>
         * 
         * @param inlineContent
         *        The base64-encoded image data for inline image content. Maximum size is 5MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inlineContent(SdkBytes inlineContent);
    }

    static final class BuilderImpl implements Builder {
        private String format;

        private SdkBytes inlineContent;

        private BuilderImpl() {
        }

        private BuilderImpl(InputImage model) {
            format(model.format);
            inlineContent(model.inlineContent);
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(InputImageFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final ByteBuffer getInlineContent() {
            return inlineContent == null ? null : inlineContent.asByteBuffer();
        }

        public final void setInlineContent(ByteBuffer inlineContent) {
            inlineContent(inlineContent == null ? null : SdkBytes.fromByteBuffer(inlineContent));
        }

        @Override
        public final Builder inlineContent(SdkBytes inlineContent) {
            this.inlineContent = inlineContent;
            return this;
        }

        @Override
        public InputImage build() {
            return new InputImage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
