/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents a container instance host device.
 * </p>
 * <note>
 * <p>
 * This object isn't applicable to jobs that are running on Fargate resources and shouldn't be provided.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Device implements SdkPojo, Serializable, ToCopyableBuilder<Device.Builder, Device> {
    private static final SdkField<String> HOST_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hostPath").getter(getter(Device::hostPath)).setter(setter(Builder::hostPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPath").build()).build();

    private static final SdkField<String> CONTAINER_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerPath").getter(getter(Device::containerPath)).setter(setter(Builder::containerPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPath").build()).build();

    private static final SdkField<List<String>> PERMISSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("permissions")
            .getter(getter(Device::permissionsAsStrings))
            .setter(setter(Builder::permissionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("permissions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOST_PATH_FIELD,
            CONTAINER_PATH_FIELD, PERMISSIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String hostPath;

    private final String containerPath;

    private final List<String> permissions;

    private Device(BuilderImpl builder) {
        this.hostPath = builder.hostPath;
        this.containerPath = builder.containerPath;
        this.permissions = builder.permissions;
    }

    /**
     * <p>
     * The path for the device on the host container instance.
     * </p>
     * 
     * @return The path for the device on the host container instance.
     */
    public final String hostPath() {
        return hostPath;
    }

    /**
     * <p>
     * The path inside the container that's used to expose the host device. By default, the <code>hostPath</code> value
     * is used.
     * </p>
     * 
     * @return The path inside the container that's used to expose the host device. By default, the
     *         <code>hostPath</code> value is used.
     */
    public final String containerPath() {
        return containerPath;
    }

    /**
     * <p>
     * The explicit permissions to provide to the container for the device. By default, the container has permissions
     * for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPermissions} method.
     * </p>
     * 
     * @return The explicit permissions to provide to the container for the device. By default, the container has
     *         permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     */
    public final List<DeviceCgroupPermission> permissions() {
        return DeviceCgroupPermissionsCopier.copyStringToEnum(permissions);
    }

    /**
     * For responses, this returns true if the service returned a value for the Permissions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPermissions() {
        return permissions != null && !(permissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The explicit permissions to provide to the container for the device. By default, the container has permissions
     * for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPermissions} method.
     * </p>
     * 
     * @return The explicit permissions to provide to the container for the device. By default, the container has
     *         permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     */
    public final List<String> permissionsAsStrings() {
        return permissions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hostPath());
        hashCode = 31 * hashCode + Objects.hashCode(containerPath());
        hashCode = 31 * hashCode + Objects.hashCode(hasPermissions() ? permissionsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Device)) {
            return false;
        }
        Device other = (Device) obj;
        return Objects.equals(hostPath(), other.hostPath()) && Objects.equals(containerPath(), other.containerPath())
                && hasPermissions() == other.hasPermissions()
                && Objects.equals(permissionsAsStrings(), other.permissionsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Device").add("HostPath", hostPath()).add("ContainerPath", containerPath())
                .add("Permissions", hasPermissions() ? permissionsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "hostPath":
            return Optional.ofNullable(clazz.cast(hostPath()));
        case "containerPath":
            return Optional.ofNullable(clazz.cast(containerPath()));
        case "permissions":
            return Optional.ofNullable(clazz.cast(permissionsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("hostPath", HOST_PATH_FIELD);
        map.put("containerPath", CONTAINER_PATH_FIELD);
        map.put("permissions", PERMISSIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Device, T> g) {
        return obj -> g.apply((Device) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Device> {
        /**
         * <p>
         * The path for the device on the host container instance.
         * </p>
         * 
         * @param hostPath
         *        The path for the device on the host container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPath(String hostPath);

        /**
         * <p>
         * The path inside the container that's used to expose the host device. By default, the <code>hostPath</code>
         * value is used.
         * </p>
         * 
         * @param containerPath
         *        The path inside the container that's used to expose the host device. By default, the
         *        <code>hostPath</code> value is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPath(String containerPath);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsWithStrings(Collection<String> permissions);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsWithStrings(String... permissions);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissions(Collection<DeviceCgroupPermission> permissions);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissions(DeviceCgroupPermission... permissions);
    }

    static final class BuilderImpl implements Builder {
        private String hostPath;

        private String containerPath;

        private List<String> permissions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Device model) {
            hostPath(model.hostPath);
            containerPath(model.containerPath);
            permissionsWithStrings(model.permissions);
        }

        public final String getHostPath() {
            return hostPath;
        }

        public final void setHostPath(String hostPath) {
            this.hostPath = hostPath;
        }

        @Override
        public final Builder hostPath(String hostPath) {
            this.hostPath = hostPath;
            return this;
        }

        public final String getContainerPath() {
            return containerPath;
        }

        public final void setContainerPath(String containerPath) {
            this.containerPath = containerPath;
        }

        @Override
        public final Builder containerPath(String containerPath) {
            this.containerPath = containerPath;
            return this;
        }

        public final Collection<String> getPermissions() {
            if (permissions instanceof SdkAutoConstructList) {
                return null;
            }
            return permissions;
        }

        public final void setPermissions(Collection<String> permissions) {
            this.permissions = DeviceCgroupPermissionsCopier.copy(permissions);
        }

        @Override
        public final Builder permissionsWithStrings(Collection<String> permissions) {
            this.permissions = DeviceCgroupPermissionsCopier.copy(permissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permissionsWithStrings(String... permissions) {
            permissionsWithStrings(Arrays.asList(permissions));
            return this;
        }

        @Override
        public final Builder permissions(Collection<DeviceCgroupPermission> permissions) {
            this.permissions = DeviceCgroupPermissionsCopier.copyEnumToString(permissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permissions(DeviceCgroupPermission... permissions) {
            permissions(Arrays.asList(permissions));
            return this;
        }

        @Override
        public Device build() {
            return new Device(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
