/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Object representing any Kubernetes overrides to a job definition that's used in a <a
 * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> API operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EksContainerOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<EksContainerOverride.Builder, EksContainerOverride> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(EksContainerOverride::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(EksContainerOverride::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(EksContainerOverride::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("args")
            .getter(getter(EksContainerOverride::args))
            .setter(setter(Builder::args))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("args").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EksContainerEnvironmentVariable>> ENV_FIELD = SdkField
            .<List<EksContainerEnvironmentVariable>> builder(MarshallingType.LIST)
            .memberName("env")
            .getter(getter(EksContainerOverride::env))
            .setter(setter(Builder::env))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("env").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EksContainerEnvironmentVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(EksContainerEnvironmentVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EksContainerResourceRequirements> RESOURCES_FIELD = SdkField
            .<EksContainerResourceRequirements> builder(MarshallingType.SDK_POJO).memberName("resources")
            .getter(getter(EksContainerOverride::resources)).setter(setter(Builder::resources))
            .constructor(EksContainerResourceRequirements::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resources").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, IMAGE_FIELD,
            COMMAND_FIELD, ARGS_FIELD, ENV_FIELD, RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String image;

    private final List<String> command;

    private final List<String> args;

    private final List<EksContainerEnvironmentVariable> env;

    private final EksContainerResourceRequirements resources;

    private EksContainerOverride(BuilderImpl builder) {
        this.name = builder.name;
        this.image = builder.image;
        this.command = builder.command;
        this.args = builder.args;
        this.env = builder.env;
        this.resources = builder.resources;
    }

    /**
     * <p>
     * A pointer to the container that you want to override. The name must match a unique container name that you wish
     * to override.
     * </p>
     * 
     * @return A pointer to the container that you want to override. The name must match a unique container name that
     *         you wish to override.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The override of the Docker image that's used to start the container.
     * </p>
     * 
     * @return The override of the Docker image that's used to start the container.
     */
    public final String image() {
        return image;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command to send to the container that overrides the default command from the Docker image or the job
     * definition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return The command to send to the container that overrides the default command from the Docker image or the job
     *         definition.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * For responses, this returns true if the service returned a value for the Args property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasArgs() {
        return args != null && !(args instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The arguments to the entrypoint to send to the container that overrides the default arguments from the Docker
     * image or the job definition. For more information, see <a
     * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
     * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
     * command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArgs} method.
     * </p>
     * 
     * @return The arguments to the entrypoint to send to the container that overrides the default arguments from the
     *         Docker image or the job definition. For more information, see <a
     *         href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
     *         href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define
     *         a command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.
     */
    public final List<String> args() {
        return args;
    }

    /**
     * For responses, this returns true if the service returned a value for the Env property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasEnv() {
        return env != null && !(env instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to send to the container. You can add new environment variables, which are added to the
     * container at launch. Or, you can override the existing environment variables from the Docker image or the job
     * definition.
     * </p>
     * <note>
     * <p>
     * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
     * variables that Batch sets.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnv} method.
     * </p>
     * 
     * @return The environment variables to send to the container. You can add new environment variables, which are
     *         added to the container at launch. Or, you can override the existing environment variables from the Docker
     *         image or the job definition.</p> <note>
     *         <p>
     *         Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
     *         variables that Batch sets.
     *         </p>
     */
    public final List<EksContainerEnvironmentVariable> env() {
        return env;
    }

    /**
     * <p>
     * The type and amount of resources to assign to a container. These override the settings in the job definition. The
     * supported resources include <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more
     * information, see <a
     * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management for
     * pods and containers</a> in the <i>Kubernetes documentation</i>.
     * </p>
     * 
     * @return The type and amount of resources to assign to a container. These override the settings in the job
     *         definition. The supported resources include <code>memory</code>, <code>cpu</code>, and
     *         <code>nvidia.com/gpu</code>. For more information, see <a
     *         href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management
     *         for pods and containers</a> in the <i>Kubernetes documentation</i>.
     */
    public final EksContainerResourceRequirements resources() {
        return resources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasArgs() ? args() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnv() ? env() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EksContainerOverride)) {
            return false;
        }
        EksContainerOverride other = (EksContainerOverride) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(image(), other.image())
                && hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && hasArgs() == other.hasArgs() && Objects.equals(args(), other.args()) && hasEnv() == other.hasEnv()
                && Objects.equals(env(), other.env()) && Objects.equals(resources(), other.resources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EksContainerOverride").add("Name", name()).add("Image", image())
                .add("Command", hasCommand() ? command() : null).add("Args", hasArgs() ? args() : null)
                .add("Env", hasEnv() ? env() : null).add("Resources", resources()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "args":
            return Optional.ofNullable(clazz.cast(args()));
        case "env":
            return Optional.ofNullable(clazz.cast(env()));
        case "resources":
            return Optional.ofNullable(clazz.cast(resources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("image", IMAGE_FIELD);
        map.put("command", COMMAND_FIELD);
        map.put("args", ARGS_FIELD);
        map.put("env", ENV_FIELD);
        map.put("resources", RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EksContainerOverride, T> g) {
        return obj -> g.apply((EksContainerOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EksContainerOverride> {
        /**
         * <p>
         * A pointer to the container that you want to override. The name must match a unique container name that you
         * wish to override.
         * </p>
         * 
         * @param name
         *        A pointer to the container that you want to override. The name must match a unique container name that
         *        you wish to override.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The override of the Docker image that's used to start the container.
         * </p>
         * 
         * @param image
         *        The override of the Docker image that's used to start the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The arguments to the entrypoint to send to the container that overrides the default arguments from the Docker
         * image or the job definition. For more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
         * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         * command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param args
         *        The arguments to the entrypoint to send to the container that overrides the default arguments from the
         *        Docker image or the job definition. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
         *        href=
         *        "https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         *        command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(Collection<String> args);

        /**
         * <p>
         * The arguments to the entrypoint to send to the container that overrides the default arguments from the Docker
         * image or the job definition. For more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
         * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         * command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param args
         *        The arguments to the entrypoint to send to the container that overrides the default arguments from the
         *        Docker image or the job definition. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
         *        href=
         *        "https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         *        command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(String... args);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch. Or, you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param env
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch. Or, you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved
         *        for variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder env(Collection<EksContainerEnvironmentVariable> env);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch. Or, you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param env
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch. Or, you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved
         *        for variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder env(EksContainerEnvironmentVariable... env);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch. Or, you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable.Builder#build()} is called
         * immediately and its result is passed to {@link #env(List<EksContainerEnvironmentVariable>)}.
         * 
         * @param env
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #env(java.util.Collection<EksContainerEnvironmentVariable>)
         */
        Builder env(Consumer<EksContainerEnvironmentVariable.Builder>... env);

        /**
         * <p>
         * The type and amount of resources to assign to a container. These override the settings in the job definition.
         * The supported resources include <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For
         * more information, see <a
         * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management for
         * pods and containers</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param resources
         *        The type and amount of resources to assign to a container. These override the settings in the job
         *        definition. The supported resources include <code>memory</code>, <code>cpu</code>, and
         *        <code>nvidia.com/gpu</code>. For more information, see <a
         *        href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource
         *        management for pods and containers</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(EksContainerResourceRequirements resources);

        /**
         * <p>
         * The type and amount of resources to assign to a container. These override the settings in the job definition.
         * The supported resources include <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For
         * more information, see <a
         * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management for
         * pods and containers</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksContainerResourceRequirements.Builder}
         * avoiding the need to create one manually via {@link EksContainerResourceRequirements#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksContainerResourceRequirements.Builder#build()} is called
         * immediately and its result is passed to {@link #resources(EksContainerResourceRequirements)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link EksContainerResourceRequirements.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(EksContainerResourceRequirements)
         */
        default Builder resources(Consumer<EksContainerResourceRequirements.Builder> resources) {
            return resources(EksContainerResourceRequirements.builder().applyMutation(resources).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String image;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private List<String> args = DefaultSdkAutoConstructList.getInstance();

        private List<EksContainerEnvironmentVariable> env = DefaultSdkAutoConstructList.getInstance();

        private EksContainerResourceRequirements resources;

        private BuilderImpl() {
        }

        private BuilderImpl(EksContainerOverride model) {
            name(model.name);
            image(model.image);
            command(model.command);
            args(model.args);
            env(model.env);
            resources(model.resources);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Collection<String> getArgs() {
            if (args instanceof SdkAutoConstructList) {
                return null;
            }
            return args;
        }

        public final void setArgs(Collection<String> args) {
            this.args = StringListCopier.copy(args);
        }

        @Override
        public final Builder args(Collection<String> args) {
            this.args = StringListCopier.copy(args);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder args(String... args) {
            args(Arrays.asList(args));
            return this;
        }

        public final List<EksContainerEnvironmentVariable.Builder> getEnv() {
            List<EksContainerEnvironmentVariable.Builder> result = EksContainerEnvironmentVariablesCopier.copyToBuilder(this.env);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnv(Collection<EksContainerEnvironmentVariable.BuilderImpl> env) {
            this.env = EksContainerEnvironmentVariablesCopier.copyFromBuilder(env);
        }

        @Override
        public final Builder env(Collection<EksContainerEnvironmentVariable> env) {
            this.env = EksContainerEnvironmentVariablesCopier.copy(env);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder env(EksContainerEnvironmentVariable... env) {
            env(Arrays.asList(env));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder env(Consumer<EksContainerEnvironmentVariable.Builder>... env) {
            env(Stream.of(env).map(c -> EksContainerEnvironmentVariable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final EksContainerResourceRequirements.Builder getResources() {
            return resources != null ? resources.toBuilder() : null;
        }

        public final void setResources(EksContainerResourceRequirements.BuilderImpl resources) {
            this.resources = resources != null ? resources.build() : null;
        }

        @Override
        public final Builder resources(EksContainerResourceRequirements resources) {
            this.resources = resources;
            return this;
        }

        @Override
        public EksContainerOverride build() {
            return new EksContainerOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
