/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents an Batch compute environment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComputeEnvironmentDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<ComputeEnvironmentDetail.Builder, ComputeEnvironmentDetail> {
    private static final SdkField<String> COMPUTE_ENVIRONMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("computeEnvironmentName").getter(getter(ComputeEnvironmentDetail::computeEnvironmentName))
            .setter(setter(Builder::computeEnvironmentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeEnvironmentName").build())
            .build();

    private static final SdkField<String> COMPUTE_ENVIRONMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("computeEnvironmentArn").getter(getter(ComputeEnvironmentDetail::computeEnvironmentArn))
            .setter(setter(Builder::computeEnvironmentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeEnvironmentArn").build())
            .build();

    private static final SdkField<Integer> UNMANAGEDV_CPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("unmanagedvCpus").getter(getter(ComputeEnvironmentDetail::unmanagedvCpus))
            .setter(setter(Builder::unmanagedvCpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unmanagedvCpus").build()).build();

    private static final SdkField<String> ECS_CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ecsClusterArn").getter(getter(ComputeEnvironmentDetail::ecsClusterArn))
            .setter(setter(Builder::ecsClusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ecsClusterArn").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(ComputeEnvironmentDetail::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ComputeEnvironmentDetail::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(ComputeEnvironmentDetail::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ComputeEnvironmentDetail::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(ComputeEnvironmentDetail::statusReason))
            .setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<ComputeResource> COMPUTE_RESOURCES_FIELD = SdkField
            .<ComputeResource> builder(MarshallingType.SDK_POJO).memberName("computeResources")
            .getter(getter(ComputeEnvironmentDetail::computeResources)).setter(setter(Builder::computeResources))
            .constructor(ComputeResource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeResources").build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceRole").getter(getter(ComputeEnvironmentDetail::serviceRole)).setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRole").build()).build();

    private static final SdkField<UpdatePolicy> UPDATE_POLICY_FIELD = SdkField.<UpdatePolicy> builder(MarshallingType.SDK_POJO)
            .memberName("updatePolicy").getter(getter(ComputeEnvironmentDetail::updatePolicy))
            .setter(setter(Builder::updatePolicy)).constructor(UpdatePolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatePolicy").build()).build();

    private static final SdkField<EksConfiguration> EKS_CONFIGURATION_FIELD = SdkField
            .<EksConfiguration> builder(MarshallingType.SDK_POJO).memberName("eksConfiguration")
            .getter(getter(ComputeEnvironmentDetail::eksConfiguration)).setter(setter(Builder::eksConfiguration))
            .constructor(EksConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eksConfiguration").build()).build();

    private static final SdkField<String> CONTAINER_ORCHESTRATION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("containerOrchestrationType")
            .getter(getter(ComputeEnvironmentDetail::containerOrchestrationTypeAsString))
            .setter(setter(Builder::containerOrchestrationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerOrchestrationType").build())
            .build();

    private static final SdkField<String> UUID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("uuid")
            .getter(getter(ComputeEnvironmentDetail::uuid)).setter(setter(Builder::uuid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("uuid").build()).build();

    private static final SdkField<String> CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("context")
            .getter(getter(ComputeEnvironmentDetail::context)).setter(setter(Builder::context))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("context").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            COMPUTE_ENVIRONMENT_NAME_FIELD, COMPUTE_ENVIRONMENT_ARN_FIELD, UNMANAGEDV_CPUS_FIELD, ECS_CLUSTER_ARN_FIELD,
            TAGS_FIELD, TYPE_FIELD, STATE_FIELD, STATUS_FIELD, STATUS_REASON_FIELD, COMPUTE_RESOURCES_FIELD, SERVICE_ROLE_FIELD,
            UPDATE_POLICY_FIELD, EKS_CONFIGURATION_FIELD, CONTAINER_ORCHESTRATION_TYPE_FIELD, UUID_FIELD, CONTEXT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String computeEnvironmentName;

    private final String computeEnvironmentArn;

    private final Integer unmanagedvCpus;

    private final String ecsClusterArn;

    private final Map<String, String> tags;

    private final String type;

    private final String state;

    private final String status;

    private final String statusReason;

    private final ComputeResource computeResources;

    private final String serviceRole;

    private final UpdatePolicy updatePolicy;

    private final EksConfiguration eksConfiguration;

    private final String containerOrchestrationType;

    private final String uuid;

    private final String context;

    private ComputeEnvironmentDetail(BuilderImpl builder) {
        this.computeEnvironmentName = builder.computeEnvironmentName;
        this.computeEnvironmentArn = builder.computeEnvironmentArn;
        this.unmanagedvCpus = builder.unmanagedvCpus;
        this.ecsClusterArn = builder.ecsClusterArn;
        this.tags = builder.tags;
        this.type = builder.type;
        this.state = builder.state;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.computeResources = builder.computeResources;
        this.serviceRole = builder.serviceRole;
        this.updatePolicy = builder.updatePolicy;
        this.eksConfiguration = builder.eksConfiguration;
        this.containerOrchestrationType = builder.containerOrchestrationType;
        this.uuid = builder.uuid;
        this.context = builder.context;
    }

    /**
     * <p>
     * The name of the compute environment. It can be up to 128 characters long. It can contain uppercase and lowercase
     * letters, numbers, hyphens (-), and underscores (_).
     * </p>
     * 
     * @return The name of the compute environment. It can be up to 128 characters long. It can contain uppercase and
     *         lowercase letters, numbers, hyphens (-), and underscores (_).
     */
    public final String computeEnvironmentName() {
        return computeEnvironmentName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the compute environment.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the compute environment.
     */
    public final String computeEnvironmentArn() {
        return computeEnvironmentArn;
    }

    /**
     * <p>
     * The maximum number of VCPUs expected to be used for an unmanaged compute environment.
     * </p>
     * 
     * @return The maximum number of VCPUs expected to be used for an unmanaged compute environment.
     */
    public final Integer unmanagedvCpus() {
        return unmanagedvCpus;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the underlying Amazon ECS cluster that the compute environment uses.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the underlying Amazon ECS cluster that the compute environment uses.
     */
    public final String ecsClusterArn() {
        return ecsClusterArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags applied to the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags applied to the compute environment.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute environments</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         environments</a> in the <i>Batch User Guide</i>.
     * @see CEType
     */
    public final CEType type() {
        return CEType.fromValue(type);
    }

    /**
     * <p>
     * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute environments</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         environments</a> in the <i>Batch User Guide</i>.
     * @see CEType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The state of the compute environment. The valid values are <code>ENABLED</code> or <code>DISABLED</code>.
     * </p>
     * <p>
     * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated job
     * queue on the compute resources within the environment. If the compute environment is managed, then it can scale
     * its instances out or in automatically based on the job queue demand.
     * </p>
     * <p>
     * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally. Managed
     * compute environments in the <code>DISABLED</code> state don't scale out.
     * </p>
     * <note>
     * <p>
     * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     * additional charges, turn off and then delete the compute environment. For more information, see <a href=
     * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     * >State</a> in the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance size
     * doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code> value of
     * <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale down to a
     * <code>c5.large</code> instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CEState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the compute environment. The valid values are <code>ENABLED</code> or <code>DISABLED</code>
     *         .</p>
     *         <p>
     *         If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
     *         associated job queue on the compute resources within the environment. If the compute environment is
     *         managed, then it can scale its instances out or in automatically based on the job queue demand.
     *         </p>
     *         <p>
     *         If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     *         environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
     *         Managed compute environments in the <code>DISABLED</code> state don't scale out.
     *         </p>
     *         <note>
     *         <p>
     *         Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     *         additional charges, turn off and then delete the compute environment. For more information, see <a href=
     *         "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     *         >State</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
     *         instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
     *         <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>.
     *         This instance doesn't scale down to a <code>c5.large</code> instance.
     * @see CEState
     */
    public final CEState state() {
        return CEState.fromValue(state);
    }

    /**
     * <p>
     * The state of the compute environment. The valid values are <code>ENABLED</code> or <code>DISABLED</code>.
     * </p>
     * <p>
     * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated job
     * queue on the compute resources within the environment. If the compute environment is managed, then it can scale
     * its instances out or in automatically based on the job queue demand.
     * </p>
     * <p>
     * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally. Managed
     * compute environments in the <code>DISABLED</code> state don't scale out.
     * </p>
     * <note>
     * <p>
     * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     * additional charges, turn off and then delete the compute environment. For more information, see <a href=
     * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     * >State</a> in the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance size
     * doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code> value of
     * <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale down to a
     * <code>c5.large</code> instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CEState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the compute environment. The valid values are <code>ENABLED</code> or <code>DISABLED</code>
     *         .</p>
     *         <p>
     *         If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
     *         associated job queue on the compute resources within the environment. If the compute environment is
     *         managed, then it can scale its instances out or in automatically based on the job queue demand.
     *         </p>
     *         <p>
     *         If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     *         environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
     *         Managed compute environments in the <code>DISABLED</code> state don't scale out.
     *         </p>
     *         <note>
     *         <p>
     *         Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     *         additional charges, turn off and then delete the compute environment. For more information, see <a href=
     *         "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     *         >State</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
     *         instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
     *         <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>.
     *         This instance doesn't scale down to a <code>c5.large</code> instance.
     * @see CEState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The current status of the compute environment (for example, <code>CREATING</code> or <code>VALID</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CEStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the compute environment (for example, <code>CREATING</code> or <code>VALID</code>).
     * @see CEStatus
     */
    public final CEStatus status() {
        return CEStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the compute environment (for example, <code>CREATING</code> or <code>VALID</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CEStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the compute environment (for example, <code>CREATING</code> or <code>VALID</code>).
     * @see CEStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A short, human-readable string to provide additional details for the current status of the compute environment.
     * </p>
     * 
     * @return A short, human-readable string to provide additional details for the current status of the compute
     *         environment.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The compute resources defined for the compute environment. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute environments</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * 
     * @return The compute resources defined for the compute environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         environments</a> in the <i>Batch User Guide</i>.
     */
    public final ComputeResource computeResources() {
        return computeResources;
    }

    /**
     * <p>
     * The service role that's associated with the compute environment that allows Batch to make calls to Amazon Web
     * Services API operations on your behalf. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM role</a> in the
     * <i>Batch User Guide</i>.
     * </p>
     * 
     * @return The service role that's associated with the compute environment that allows Batch to make calls to Amazon
     *         Web Services API operations on your behalf. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM
     *         role</a> in the <i>Batch User Guide</i>.
     */
    public final String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * Specifies the infrastructure update policy for the compute environment. For more information about infrastructure
     * updates, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
     * environments</a> in the <i>Batch User Guide</i>.
     * </p>
     * 
     * @return Specifies the infrastructure update policy for the compute environment. For more information about
     *         infrastructure updates, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating
     *         compute environments</a> in the <i>Batch User Guide</i>.
     */
    public final UpdatePolicy updatePolicy() {
        return updatePolicy;
    }

    /**
     * <p>
     * The configuration for the Amazon EKS cluster that supports the Batch compute environment. Only specify this
     * parameter if the <code>containerOrchestrationType</code> is <code>EKS</code>.
     * </p>
     * 
     * @return The configuration for the Amazon EKS cluster that supports the Batch compute environment. Only specify
     *         this parameter if the <code>containerOrchestrationType</code> is <code>EKS</code>.
     */
    public final EksConfiguration eksConfiguration() {
        return eksConfiguration;
    }

    /**
     * <p>
     * The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
     * <code>EKS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #containerOrchestrationType} will return {@link OrchestrationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #containerOrchestrationTypeAsString}.
     * </p>
     * 
     * @return The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
     *         <code>EKS</code>.
     * @see OrchestrationType
     */
    public final OrchestrationType containerOrchestrationType() {
        return OrchestrationType.fromValue(containerOrchestrationType);
    }

    /**
     * <p>
     * The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
     * <code>EKS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #containerOrchestrationType} will return {@link OrchestrationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #containerOrchestrationTypeAsString}.
     * </p>
     * 
     * @return The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
     *         <code>EKS</code>.
     * @see OrchestrationType
     */
    public final String containerOrchestrationTypeAsString() {
        return containerOrchestrationType;
    }

    /**
     * <p>
     * Unique identifier for the compute environment.
     * </p>
     * 
     * @return Unique identifier for the compute environment.
     */
    public final String uuid() {
        return uuid;
    }

    /**
     * <p>
     * Reserved.
     * </p>
     * 
     * @return Reserved.
     */
    public final String context() {
        return context;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(computeEnvironmentName());
        hashCode = 31 * hashCode + Objects.hashCode(computeEnvironmentArn());
        hashCode = 31 * hashCode + Objects.hashCode(unmanagedvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(ecsClusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(computeResources());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(updatePolicy());
        hashCode = 31 * hashCode + Objects.hashCode(eksConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(containerOrchestrationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(uuid());
        hashCode = 31 * hashCode + Objects.hashCode(context());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComputeEnvironmentDetail)) {
            return false;
        }
        ComputeEnvironmentDetail other = (ComputeEnvironmentDetail) obj;
        return Objects.equals(computeEnvironmentName(), other.computeEnvironmentName())
                && Objects.equals(computeEnvironmentArn(), other.computeEnvironmentArn())
                && Objects.equals(unmanagedvCpus(), other.unmanagedvCpus())
                && Objects.equals(ecsClusterArn(), other.ecsClusterArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(computeResources(), other.computeResources())
                && Objects.equals(serviceRole(), other.serviceRole()) && Objects.equals(updatePolicy(), other.updatePolicy())
                && Objects.equals(eksConfiguration(), other.eksConfiguration())
                && Objects.equals(containerOrchestrationTypeAsString(), other.containerOrchestrationTypeAsString())
                && Objects.equals(uuid(), other.uuid()) && Objects.equals(context(), other.context());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComputeEnvironmentDetail").add("ComputeEnvironmentName", computeEnvironmentName())
                .add("ComputeEnvironmentArn", computeEnvironmentArn()).add("UnmanagedvCpus", unmanagedvCpus())
                .add("EcsClusterArn", ecsClusterArn()).add("Tags", hasTags() ? tags() : null).add("Type", typeAsString())
                .add("State", stateAsString()).add("Status", statusAsString()).add("StatusReason", statusReason())
                .add("ComputeResources", computeResources()).add("ServiceRole", serviceRole())
                .add("UpdatePolicy", updatePolicy()).add("EksConfiguration", eksConfiguration())
                .add("ContainerOrchestrationType", containerOrchestrationTypeAsString()).add("Uuid", uuid())
                .add("Context", context()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "computeEnvironmentName":
            return Optional.ofNullable(clazz.cast(computeEnvironmentName()));
        case "computeEnvironmentArn":
            return Optional.ofNullable(clazz.cast(computeEnvironmentArn()));
        case "unmanagedvCpus":
            return Optional.ofNullable(clazz.cast(unmanagedvCpus()));
        case "ecsClusterArn":
            return Optional.ofNullable(clazz.cast(ecsClusterArn()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "computeResources":
            return Optional.ofNullable(clazz.cast(computeResources()));
        case "serviceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "updatePolicy":
            return Optional.ofNullable(clazz.cast(updatePolicy()));
        case "eksConfiguration":
            return Optional.ofNullable(clazz.cast(eksConfiguration()));
        case "containerOrchestrationType":
            return Optional.ofNullable(clazz.cast(containerOrchestrationTypeAsString()));
        case "uuid":
            return Optional.ofNullable(clazz.cast(uuid()));
        case "context":
            return Optional.ofNullable(clazz.cast(context()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("computeEnvironmentName", COMPUTE_ENVIRONMENT_NAME_FIELD);
        map.put("computeEnvironmentArn", COMPUTE_ENVIRONMENT_ARN_FIELD);
        map.put("unmanagedvCpus", UNMANAGEDV_CPUS_FIELD);
        map.put("ecsClusterArn", ECS_CLUSTER_ARN_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("state", STATE_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("computeResources", COMPUTE_RESOURCES_FIELD);
        map.put("serviceRole", SERVICE_ROLE_FIELD);
        map.put("updatePolicy", UPDATE_POLICY_FIELD);
        map.put("eksConfiguration", EKS_CONFIGURATION_FIELD);
        map.put("containerOrchestrationType", CONTAINER_ORCHESTRATION_TYPE_FIELD);
        map.put("uuid", UUID_FIELD);
        map.put("context", CONTEXT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ComputeEnvironmentDetail, T> g) {
        return obj -> g.apply((ComputeEnvironmentDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComputeEnvironmentDetail> {
        /**
         * <p>
         * The name of the compute environment. It can be up to 128 characters long. It can contain uppercase and
         * lowercase letters, numbers, hyphens (-), and underscores (_).
         * </p>
         * 
         * @param computeEnvironmentName
         *        The name of the compute environment. It can be up to 128 characters long. It can contain uppercase and
         *        lowercase letters, numbers, hyphens (-), and underscores (_).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentName(String computeEnvironmentName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the compute environment.
         * </p>
         * 
         * @param computeEnvironmentArn
         *        The Amazon Resource Name (ARN) of the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentArn(String computeEnvironmentArn);

        /**
         * <p>
         * The maximum number of VCPUs expected to be used for an unmanaged compute environment.
         * </p>
         * 
         * @param unmanagedvCpus
         *        The maximum number of VCPUs expected to be used for an unmanaged compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unmanagedvCpus(Integer unmanagedvCpus);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the underlying Amazon ECS cluster that the compute environment uses.
         * </p>
         * 
         * @param ecsClusterArn
         *        The Amazon Resource Name (ARN) of the underlying Amazon ECS cluster that the compute environment uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ecsClusterArn(String ecsClusterArn);

        /**
         * <p>
         * The tags applied to the compute environment.
         * </p>
         * 
         * @param tags
         *        The tags applied to the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information,
         * see <a href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         * environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        environments</a> in the <i>Batch User Guide</i>.
         * @see CEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information,
         * see <a href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         * environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        environments</a> in the <i>Batch User Guide</i>.
         * @see CEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEType
         */
        Builder type(CEType type);

        /**
         * <p>
         * The state of the compute environment. The valid values are <code>ENABLED</code> or <code>DISABLED</code>.
         * </p>
         * <p>
         * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated
         * job queue on the compute resources within the environment. If the compute environment is managed, then it can
         * scale its instances out or in automatically based on the job queue demand.
         * </p>
         * <p>
         * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
         * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
         * Managed compute environments in the <code>DISABLED</code> state don't scale out.
         * </p>
         * <note>
         * <p>
         * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
         * additional charges, turn off and then delete the compute environment. For more information, see <a href=
         * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         * >State</a> in the <i>Batch User Guide</i>.
         * </p>
         * </note>
         * <p>
         * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance
         * size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code>
         * value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale
         * down to a <code>c5.large</code> instance.
         * </p>
         * 
         * @param state
         *        The state of the compute environment. The valid values are <code>ENABLED</code> or
         *        <code>DISABLED</code>.</p>
         *        <p>
         *        If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
         *        associated job queue on the compute resources within the environment. If the compute environment is
         *        managed, then it can scale its instances out or in automatically based on the job queue demand.
         *        </p>
         *        <p>
         *        If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within
         *        the environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress
         *        normally. Managed compute environments in the <code>DISABLED</code> state don't scale out.
         *        </p>
         *        <note>
         *        <p>
         *        Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To
         *        prevent additional charges, turn off and then delete the compute environment. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         *        >State</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
         *        instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
         *        <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>
         *        . This instance doesn't scale down to a <code>c5.large</code> instance.
         * @see CEState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the compute environment. The valid values are <code>ENABLED</code> or <code>DISABLED</code>.
         * </p>
         * <p>
         * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated
         * job queue on the compute resources within the environment. If the compute environment is managed, then it can
         * scale its instances out or in automatically based on the job queue demand.
         * </p>
         * <p>
         * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
         * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
         * Managed compute environments in the <code>DISABLED</code> state don't scale out.
         * </p>
         * <note>
         * <p>
         * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
         * additional charges, turn off and then delete the compute environment. For more information, see <a href=
         * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         * >State</a> in the <i>Batch User Guide</i>.
         * </p>
         * </note>
         * <p>
         * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance
         * size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code>
         * value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale
         * down to a <code>c5.large</code> instance.
         * </p>
         * 
         * @param state
         *        The state of the compute environment. The valid values are <code>ENABLED</code> or
         *        <code>DISABLED</code>.</p>
         *        <p>
         *        If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
         *        associated job queue on the compute resources within the environment. If the compute environment is
         *        managed, then it can scale its instances out or in automatically based on the job queue demand.
         *        </p>
         *        <p>
         *        If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within
         *        the environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress
         *        normally. Managed compute environments in the <code>DISABLED</code> state don't scale out.
         *        </p>
         *        <note>
         *        <p>
         *        Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To
         *        prevent additional charges, turn off and then delete the compute environment. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         *        >State</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
         *        instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
         *        <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>
         *        . This instance doesn't scale down to a <code>c5.large</code> instance.
         * @see CEState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEState
         */
        Builder state(CEState state);

        /**
         * <p>
         * The current status of the compute environment (for example, <code>CREATING</code> or <code>VALID</code>).
         * </p>
         * 
         * @param status
         *        The current status of the compute environment (for example, <code>CREATING</code> or
         *        <code>VALID</code>).
         * @see CEStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the compute environment (for example, <code>CREATING</code> or <code>VALID</code>).
         * </p>
         * 
         * @param status
         *        The current status of the compute environment (for example, <code>CREATING</code> or
         *        <code>VALID</code>).
         * @see CEStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEStatus
         */
        Builder status(CEStatus status);

        /**
         * <p>
         * A short, human-readable string to provide additional details for the current status of the compute
         * environment.
         * </p>
         * 
         * @param statusReason
         *        A short, human-readable string to provide additional details for the current status of the compute
         *        environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The compute resources defined for the compute environment. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute environments</a>
         * in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param computeResources
         *        The compute resources defined for the compute environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        environments</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeResources(ComputeResource computeResources);

        /**
         * <p>
         * The compute resources defined for the compute environment. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute environments</a>
         * in the <i>Batch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ComputeResource.Builder} avoiding the
         * need to create one manually via {@link ComputeResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ComputeResource.Builder#build()} is called immediately and its
         * result is passed to {@link #computeResources(ComputeResource)}.
         * 
         * @param computeResources
         *        a consumer that will call methods on {@link ComputeResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #computeResources(ComputeResource)
         */
        default Builder computeResources(Consumer<ComputeResource.Builder> computeResources) {
            return computeResources(ComputeResource.builder().applyMutation(computeResources).build());
        }

        /**
         * <p>
         * The service role that's associated with the compute environment that allows Batch to make calls to Amazon Web
         * Services API operations on your behalf. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM role</a> in
         * the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param serviceRole
         *        The service role that's associated with the compute environment that allows Batch to make calls to
         *        Amazon Web Services API operations on your behalf. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM
         *        role</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * Specifies the infrastructure update policy for the compute environment. For more information about
         * infrastructure updates, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
         * environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param updatePolicy
         *        Specifies the infrastructure update policy for the compute environment. For more information about
         *        infrastructure updates, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating
         *        compute environments</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatePolicy(UpdatePolicy updatePolicy);

        /**
         * <p>
         * Specifies the infrastructure update policy for the compute environment. For more information about
         * infrastructure updates, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
         * environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link UpdatePolicy.Builder} avoiding the need
         * to create one manually via {@link UpdatePolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UpdatePolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #updatePolicy(UpdatePolicy)}.
         * 
         * @param updatePolicy
         *        a consumer that will call methods on {@link UpdatePolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #updatePolicy(UpdatePolicy)
         */
        default Builder updatePolicy(Consumer<UpdatePolicy.Builder> updatePolicy) {
            return updatePolicy(UpdatePolicy.builder().applyMutation(updatePolicy).build());
        }

        /**
         * <p>
         * The configuration for the Amazon EKS cluster that supports the Batch compute environment. Only specify this
         * parameter if the <code>containerOrchestrationType</code> is <code>EKS</code>.
         * </p>
         * 
         * @param eksConfiguration
         *        The configuration for the Amazon EKS cluster that supports the Batch compute environment. Only specify
         *        this parameter if the <code>containerOrchestrationType</code> is <code>EKS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksConfiguration(EksConfiguration eksConfiguration);

        /**
         * <p>
         * The configuration for the Amazon EKS cluster that supports the Batch compute environment. Only specify this
         * parameter if the <code>containerOrchestrationType</code> is <code>EKS</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksConfiguration.Builder} avoiding the
         * need to create one manually via {@link EksConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #eksConfiguration(EksConfiguration)}.
         * 
         * @param eksConfiguration
         *        a consumer that will call methods on {@link EksConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eksConfiguration(EksConfiguration)
         */
        default Builder eksConfiguration(Consumer<EksConfiguration.Builder> eksConfiguration) {
            return eksConfiguration(EksConfiguration.builder().applyMutation(eksConfiguration).build());
        }

        /**
         * <p>
         * The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
         * <code>EKS</code>.
         * </p>
         * 
         * @param containerOrchestrationType
         *        The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
         *        <code>EKS</code>.
         * @see OrchestrationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrchestrationType
         */
        Builder containerOrchestrationType(String containerOrchestrationType);

        /**
         * <p>
         * The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
         * <code>EKS</code>.
         * </p>
         * 
         * @param containerOrchestrationType
         *        The orchestration type of the compute environment. The valid values are <code>ECS</code> (default) or
         *        <code>EKS</code>.
         * @see OrchestrationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrchestrationType
         */
        Builder containerOrchestrationType(OrchestrationType containerOrchestrationType);

        /**
         * <p>
         * Unique identifier for the compute environment.
         * </p>
         * 
         * @param uuid
         *        Unique identifier for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uuid(String uuid);

        /**
         * <p>
         * Reserved.
         * </p>
         * 
         * @param context
         *        Reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder context(String context);
    }

    static final class BuilderImpl implements Builder {
        private String computeEnvironmentName;

        private String computeEnvironmentArn;

        private Integer unmanagedvCpus;

        private String ecsClusterArn;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String type;

        private String state;

        private String status;

        private String statusReason;

        private ComputeResource computeResources;

        private String serviceRole;

        private UpdatePolicy updatePolicy;

        private EksConfiguration eksConfiguration;

        private String containerOrchestrationType;

        private String uuid;

        private String context;

        private BuilderImpl() {
        }

        private BuilderImpl(ComputeEnvironmentDetail model) {
            computeEnvironmentName(model.computeEnvironmentName);
            computeEnvironmentArn(model.computeEnvironmentArn);
            unmanagedvCpus(model.unmanagedvCpus);
            ecsClusterArn(model.ecsClusterArn);
            tags(model.tags);
            type(model.type);
            state(model.state);
            status(model.status);
            statusReason(model.statusReason);
            computeResources(model.computeResources);
            serviceRole(model.serviceRole);
            updatePolicy(model.updatePolicy);
            eksConfiguration(model.eksConfiguration);
            containerOrchestrationType(model.containerOrchestrationType);
            uuid(model.uuid);
            context(model.context);
        }

        public final String getComputeEnvironmentName() {
            return computeEnvironmentName;
        }

        public final void setComputeEnvironmentName(String computeEnvironmentName) {
            this.computeEnvironmentName = computeEnvironmentName;
        }

        @Override
        public final Builder computeEnvironmentName(String computeEnvironmentName) {
            this.computeEnvironmentName = computeEnvironmentName;
            return this;
        }

        public final String getComputeEnvironmentArn() {
            return computeEnvironmentArn;
        }

        public final void setComputeEnvironmentArn(String computeEnvironmentArn) {
            this.computeEnvironmentArn = computeEnvironmentArn;
        }

        @Override
        public final Builder computeEnvironmentArn(String computeEnvironmentArn) {
            this.computeEnvironmentArn = computeEnvironmentArn;
            return this;
        }

        public final Integer getUnmanagedvCpus() {
            return unmanagedvCpus;
        }

        public final void setUnmanagedvCpus(Integer unmanagedvCpus) {
            this.unmanagedvCpus = unmanagedvCpus;
        }

        @Override
        public final Builder unmanagedvCpus(Integer unmanagedvCpus) {
            this.unmanagedvCpus = unmanagedvCpus;
            return this;
        }

        public final String getEcsClusterArn() {
            return ecsClusterArn;
        }

        public final void setEcsClusterArn(String ecsClusterArn) {
            this.ecsClusterArn = ecsClusterArn;
        }

        @Override
        public final Builder ecsClusterArn(String ecsClusterArn) {
            this.ecsClusterArn = ecsClusterArn;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CEType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CEState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CEStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final ComputeResource.Builder getComputeResources() {
            return computeResources != null ? computeResources.toBuilder() : null;
        }

        public final void setComputeResources(ComputeResource.BuilderImpl computeResources) {
            this.computeResources = computeResources != null ? computeResources.build() : null;
        }

        @Override
        public final Builder computeResources(ComputeResource computeResources) {
            this.computeResources = computeResources;
            return this;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final UpdatePolicy.Builder getUpdatePolicy() {
            return updatePolicy != null ? updatePolicy.toBuilder() : null;
        }

        public final void setUpdatePolicy(UpdatePolicy.BuilderImpl updatePolicy) {
            this.updatePolicy = updatePolicy != null ? updatePolicy.build() : null;
        }

        @Override
        public final Builder updatePolicy(UpdatePolicy updatePolicy) {
            this.updatePolicy = updatePolicy;
            return this;
        }

        public final EksConfiguration.Builder getEksConfiguration() {
            return eksConfiguration != null ? eksConfiguration.toBuilder() : null;
        }

        public final void setEksConfiguration(EksConfiguration.BuilderImpl eksConfiguration) {
            this.eksConfiguration = eksConfiguration != null ? eksConfiguration.build() : null;
        }

        @Override
        public final Builder eksConfiguration(EksConfiguration eksConfiguration) {
            this.eksConfiguration = eksConfiguration;
            return this;
        }

        public final String getContainerOrchestrationType() {
            return containerOrchestrationType;
        }

        public final void setContainerOrchestrationType(String containerOrchestrationType) {
            this.containerOrchestrationType = containerOrchestrationType;
        }

        @Override
        public final Builder containerOrchestrationType(String containerOrchestrationType) {
            this.containerOrchestrationType = containerOrchestrationType;
            return this;
        }

        @Override
        public final Builder containerOrchestrationType(OrchestrationType containerOrchestrationType) {
            this.containerOrchestrationType(containerOrchestrationType == null ? null : containerOrchestrationType.toString());
            return this;
        }

        public final String getUuid() {
            return uuid;
        }

        public final void setUuid(String uuid) {
            this.uuid = uuid;
        }

        @Override
        public final Builder uuid(String uuid) {
            this.uuid = uuid;
            return this;
        }

        public final String getContext() {
            return context;
        }

        public final void setContext(String context) {
            this.context = context;
        }

        @Override
        public final Builder context(String context) {
            this.context = context;
            return this;
        }

        @Override
        public ComputeEnvironmentDetail build() {
            return new ComputeEnvironmentDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
