/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartBackupJobRequest extends BackupRequest implements
        ToCopyableBuilder<StartBackupJobRequest.Builder, StartBackupJobRequest> {
    private static final SdkField<String> BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultName").getter(getter(StartBackupJobRequest::backupVaultName))
            .setter(setter(Builder::backupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultName").build()).build();

    private static final SdkField<String> LOGICALLY_AIR_GAPPED_BACKUP_VAULT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogicallyAirGappedBackupVaultArn")
            .getter(getter(StartBackupJobRequest::logicallyAirGappedBackupVaultArn))
            .setter(setter(Builder::logicallyAirGappedBackupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicallyAirGappedBackupVaultArn")
                    .build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(StartBackupJobRequest::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(StartBackupJobRequest::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<String> IDEMPOTENCY_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IdempotencyToken")
            .getter(getter(StartBackupJobRequest::idempotencyToken))
            .setter(setter(Builder::idempotencyToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdempotencyToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Long> START_WINDOW_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("StartWindowMinutes").getter(getter(StartBackupJobRequest::startWindowMinutes))
            .setter(setter(Builder::startWindowMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartWindowMinutes").build())
            .build();

    private static final SdkField<Long> COMPLETE_WINDOW_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CompleteWindowMinutes").getter(getter(StartBackupJobRequest::completeWindowMinutes))
            .setter(setter(Builder::completeWindowMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompleteWindowMinutes").build())
            .build();

    private static final SdkField<Lifecycle> LIFECYCLE_FIELD = SdkField.<Lifecycle> builder(MarshallingType.SDK_POJO)
            .memberName("Lifecycle").getter(getter(StartBackupJobRequest::lifecycle)).setter(setter(Builder::lifecycle))
            .constructor(Lifecycle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<Map<String, String>> RECOVERY_POINT_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RecoveryPointTags")
            .getter(getter(StartBackupJobRequest::recoveryPointTags))
            .setter(setter(Builder::recoveryPointTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> BACKUP_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("BackupOptions")
            .getter(getter(StartBackupJobRequest::backupOptions))
            .setter(setter(Builder::backupOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> INDEX_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Index")
            .getter(getter(StartBackupJobRequest::indexAsString)).setter(setter(Builder::index))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Index").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BACKUP_VAULT_NAME_FIELD,
            LOGICALLY_AIR_GAPPED_BACKUP_VAULT_ARN_FIELD, RESOURCE_ARN_FIELD, IAM_ROLE_ARN_FIELD, IDEMPOTENCY_TOKEN_FIELD,
            START_WINDOW_MINUTES_FIELD, COMPLETE_WINDOW_MINUTES_FIELD, LIFECYCLE_FIELD, RECOVERY_POINT_TAGS_FIELD,
            BACKUP_OPTIONS_FIELD, INDEX_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String backupVaultName;

    private final String logicallyAirGappedBackupVaultArn;

    private final String resourceArn;

    private final String iamRoleArn;

    private final String idempotencyToken;

    private final Long startWindowMinutes;

    private final Long completeWindowMinutes;

    private final Lifecycle lifecycle;

    private final Map<String, String> recoveryPointTags;

    private final Map<String, String> backupOptions;

    private final String index;

    private StartBackupJobRequest(BuilderImpl builder) {
        super(builder);
        this.backupVaultName = builder.backupVaultName;
        this.logicallyAirGappedBackupVaultArn = builder.logicallyAirGappedBackupVaultArn;
        this.resourceArn = builder.resourceArn;
        this.iamRoleArn = builder.iamRoleArn;
        this.idempotencyToken = builder.idempotencyToken;
        this.startWindowMinutes = builder.startWindowMinutes;
        this.completeWindowMinutes = builder.completeWindowMinutes;
        this.lifecycle = builder.lifecycle;
        this.recoveryPointTags = builder.recoveryPointTags;
        this.backupOptions = builder.backupOptions;
        this.index = builder.index;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the Amazon Web Services Region where they are created.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the Amazon Web Services Region where they are created.
     */
    public final String backupVaultName() {
        return backupVaultName;
    }

    /**
     * <p>
     * The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided, supported fully
     * managed resources back up directly to logically air-gapped vault, while other supported resources create a
     * temporary (billable) snapshot in backup vault, then copy it to logically air-gapped vault. Unsupported resources
     * only back up to the specified backup vault.
     * </p>
     * 
     * @return The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided,
     *         supported fully managed resources back up directly to logically air-gapped vault, while other supported
     *         resources create a temporary (billable) snapshot in backup vault, then copy it to logically air-gapped
     *         vault. Unsupported resources only back up to the specified backup vault.
     */
    public final String logicallyAirGappedBackupVaultArn() {
        return logicallyAirGappedBackupVaultArn;
    }

    /**
     * <p>
     * An Amazon Resource Name (ARN) that uniquely identifies a resource. The format of the ARN depends on the resource
     * type.
     * </p>
     * 
     * @return An Amazon Resource Name (ARN) that uniquely identifies a resource. The format of the ARN depends on the
     *         resource type.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * Specifies the IAM role ARN used to create the target recovery point; for example,
     * <code>arn:aws:iam::123456789012:role/S3Access</code>.
     * </p>
     * 
     * @return Specifies the IAM role ARN used to create the target recovery point; for example,
     *         <code>arn:aws:iam::123456789012:role/S3Access</code>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * A customer-chosen string that you can use to distinguish between otherwise identical calls to
     * <code>StartBackupJob</code>. Retrying a successful request with the same idempotency token results in a success
     * message with no action taken.
     * </p>
     * 
     * @return A customer-chosen string that you can use to distinguish between otherwise identical calls to
     *         <code>StartBackupJob</code>. Retrying a successful request with the same idempotency token results in a
     *         success message with no action taken.
     */
    public final String idempotencyToken() {
        return idempotencyToken;
    }

    /**
     * <p>
     * A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start successfully.
     * This value is optional, and the default is 8 hours. If this value is included, it must be at least 60 minutes to
     * avoid errors.
     * </p>
     * <p>
     * This parameter has a maximum value of 100 years (52,560,000 minutes).
     * </p>
     * <p>
     * During the start window, the backup job status remains in <code>CREATED</code> status until it has successfully
     * begun or until the start window time has run out. If within the start window time Backup receives an error that
     * allows the job to be retried, Backup will automatically retry to begin the job at least every 10 minutes until
     * the backup successfully begins (the job status changes to <code>RUNNING</code>) or until the job status changes
     * to <code>EXPIRED</code> (which is expected to occur when the start window time is over).
     * </p>
     * 
     * @return A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start
     *         successfully. This value is optional, and the default is 8 hours. If this value is included, it must be
     *         at least 60 minutes to avoid errors.</p>
     *         <p>
     *         This parameter has a maximum value of 100 years (52,560,000 minutes).
     *         </p>
     *         <p>
     *         During the start window, the backup job status remains in <code>CREATED</code> status until it has
     *         successfully begun or until the start window time has run out. If within the start window time Backup
     *         receives an error that allows the job to be retried, Backup will automatically retry to begin the job at
     *         least every 10 minutes until the backup successfully begins (the job status changes to
     *         <code>RUNNING</code>) or until the job status changes to <code>EXPIRED</code> (which is expected to occur
     *         when the start window time is over).
     */
    public final Long startWindowMinutes() {
        return startWindowMinutes;
    }

    /**
     * <p>
     * A value in minutes during which a successfully started backup must complete, or else Backup will cancel the job.
     * This value is optional. This value begins counting down from when the backup was scheduled. It does not add
     * additional time for <code>StartWindowMinutes</code>, or if the backup started later than scheduled.
     * </p>
     * <p>
     * Like <code>StartWindowMinutes</code>, this parameter has a maximum value of 100 years (52,560,000 minutes).
     * </p>
     * 
     * @return A value in minutes during which a successfully started backup must complete, or else Backup will cancel
     *         the job. This value is optional. This value begins counting down from when the backup was scheduled. It
     *         does not add additional time for <code>StartWindowMinutes</code>, or if the backup started later than
     *         scheduled.</p>
     *         <p>
     *         Like <code>StartWindowMinutes</code>, this parameter has a maximum value of 100 years (52,560,000
     *         minutes).
     */
    public final Long completeWindowMinutes() {
        return completeWindowMinutes;
    }

    /**
     * <p>
     * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup will
     * transition and expire backups automatically according to the lifecycle that you define.
     * </p>
     * <p>
     * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
     * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition to
     * cold after days” setting cannot be changed after a backup has been transitioned to cold.
     * </p>
     * <p>
     * Resource types that can transition to cold storage are listed in the <a href=
     * "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
     * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
     * </p>
     * <p>
     * This parameter has a maximum value of 100 years (36,500 days).
     * </p>
     * 
     * @return The lifecycle defines when a protected resource is transitioned to cold storage and when it expires.
     *         Backup will transition and expire backups automatically according to the lifecycle that you define. </p>
     *         <p>
     *         Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore,
     *         the “retention” setting must be 90 days greater than the “transition to cold after days” setting. The
     *         “transition to cold after days” setting cannot be changed after a backup has been transitioned to cold.
     *         </p>
     *         <p>
     *         Resource types that can transition to cold storage are listed in the <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
     *         >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
     *         </p>
     *         <p>
     *         This parameter has a maximum value of 100 years (36,500 days).
     */
    public final Lifecycle lifecycle() {
        return lifecycle;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecoveryPointTags property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecoveryPointTags() {
        return recoveryPointTags != null && !(recoveryPointTags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags to assign to the resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecoveryPointTags} method.
     * </p>
     * 
     * @return The tags to assign to the resources.
     */
    public final Map<String, String> recoveryPointTags() {
        return recoveryPointTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the BackupOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBackupOptions() {
        return backupOptions != null && !(backupOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The backup option for a selected resource. This option is only available for Windows Volume Shadow Copy Service
     * (VSS) backup jobs.
     * </p>
     * <p>
     * Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option and
     * create a Windows VSS backup. Set to <code>"WindowsVSS""disabled"</code> to create a regular backup. The
     * <code>WindowsVSS</code> option is not enabled by default.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBackupOptions} method.
     * </p>
     * 
     * @return The backup option for a selected resource. This option is only available for Windows Volume Shadow Copy
     *         Service (VSS) backup jobs.</p>
     *         <p>
     *         Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup
     *         option and create a Windows VSS backup. Set to <code>"WindowsVSS""disabled"</code> to create a regular
     *         backup. The <code>WindowsVSS</code> option is not enabled by default.
     */
    public final Map<String, String> backupOptions() {
        return backupOptions;
    }

    /**
     * <p>
     * Include this parameter to enable index creation if your backup job has a resource type that supports backup
     * indexes.
     * </p>
     * <p>
     * Resource types that support backup indexes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EBS</code> for Amazon Elastic Block Store
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
     * </p>
     * </li>
     * </ul>
     * <p>
     * Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
     * </p>
     * <p>
     * To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a backup
     * index, set value to <code>ENABLED</code>.
     * </p>
     * <p>
     * To delete a backup index, set value to <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #index} will return
     * {@link Index#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #indexAsString}.
     * </p>
     * 
     * @return Include this parameter to enable index creation if your backup job has a resource type that supports
     *         backup indexes.</p>
     *         <p>
     *         Resource types that support backup indexes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EBS</code> for Amazon Elastic Block Store
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
     *         </p>
     *         <p>
     *         To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a
     *         backup index, set value to <code>ENABLED</code>.
     *         </p>
     *         <p>
     *         To delete a backup index, set value to <code>DISABLED</code>.
     * @see Index
     */
    public final Index index() {
        return Index.fromValue(index);
    }

    /**
     * <p>
     * Include this parameter to enable index creation if your backup job has a resource type that supports backup
     * indexes.
     * </p>
     * <p>
     * Resource types that support backup indexes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EBS</code> for Amazon Elastic Block Store
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
     * </p>
     * </li>
     * </ul>
     * <p>
     * Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
     * </p>
     * <p>
     * To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a backup
     * index, set value to <code>ENABLED</code>.
     * </p>
     * <p>
     * To delete a backup index, set value to <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #index} will return
     * {@link Index#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #indexAsString}.
     * </p>
     * 
     * @return Include this parameter to enable index creation if your backup job has a resource type that supports
     *         backup indexes.</p>
     *         <p>
     *         Resource types that support backup indexes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EBS</code> for Amazon Elastic Block Store
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
     *         </p>
     *         <p>
     *         To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a
     *         backup index, set value to <code>ENABLED</code>.
     *         </p>
     *         <p>
     *         To delete a backup index, set value to <code>DISABLED</code>.
     * @see Index
     */
    public final String indexAsString() {
        return index;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(logicallyAirGappedBackupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(idempotencyToken());
        hashCode = 31 * hashCode + Objects.hashCode(startWindowMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(completeWindowMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycle());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecoveryPointTags() ? recoveryPointTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBackupOptions() ? backupOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(indexAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartBackupJobRequest)) {
            return false;
        }
        StartBackupJobRequest other = (StartBackupJobRequest) obj;
        return Objects.equals(backupVaultName(), other.backupVaultName())
                && Objects.equals(logicallyAirGappedBackupVaultArn(), other.logicallyAirGappedBackupVaultArn())
                && Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(iamRoleArn(), other.iamRoleArn())
                && Objects.equals(idempotencyToken(), other.idempotencyToken())
                && Objects.equals(startWindowMinutes(), other.startWindowMinutes())
                && Objects.equals(completeWindowMinutes(), other.completeWindowMinutes())
                && Objects.equals(lifecycle(), other.lifecycle()) && hasRecoveryPointTags() == other.hasRecoveryPointTags()
                && Objects.equals(recoveryPointTags(), other.recoveryPointTags())
                && hasBackupOptions() == other.hasBackupOptions() && Objects.equals(backupOptions(), other.backupOptions())
                && Objects.equals(indexAsString(), other.indexAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartBackupJobRequest").add("BackupVaultName", backupVaultName())
                .add("LogicallyAirGappedBackupVaultArn", logicallyAirGappedBackupVaultArn()).add("ResourceArn", resourceArn())
                .add("IamRoleArn", iamRoleArn()).add("IdempotencyToken", idempotencyToken())
                .add("StartWindowMinutes", startWindowMinutes()).add("CompleteWindowMinutes", completeWindowMinutes())
                .add("Lifecycle", lifecycle())
                .add("RecoveryPointTags", recoveryPointTags() == null ? null : "*** Sensitive Data Redacted ***")
                .add("BackupOptions", hasBackupOptions() ? backupOptions() : null).add("Index", indexAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BackupVaultName":
            return Optional.ofNullable(clazz.cast(backupVaultName()));
        case "LogicallyAirGappedBackupVaultArn":
            return Optional.ofNullable(clazz.cast(logicallyAirGappedBackupVaultArn()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "IdempotencyToken":
            return Optional.ofNullable(clazz.cast(idempotencyToken()));
        case "StartWindowMinutes":
            return Optional.ofNullable(clazz.cast(startWindowMinutes()));
        case "CompleteWindowMinutes":
            return Optional.ofNullable(clazz.cast(completeWindowMinutes()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycle()));
        case "RecoveryPointTags":
            return Optional.ofNullable(clazz.cast(recoveryPointTags()));
        case "BackupOptions":
            return Optional.ofNullable(clazz.cast(backupOptions()));
        case "Index":
            return Optional.ofNullable(clazz.cast(indexAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("BackupVaultName", BACKUP_VAULT_NAME_FIELD);
        map.put("LogicallyAirGappedBackupVaultArn", LOGICALLY_AIR_GAPPED_BACKUP_VAULT_ARN_FIELD);
        map.put("ResourceArn", RESOURCE_ARN_FIELD);
        map.put("IamRoleArn", IAM_ROLE_ARN_FIELD);
        map.put("IdempotencyToken", IDEMPOTENCY_TOKEN_FIELD);
        map.put("StartWindowMinutes", START_WINDOW_MINUTES_FIELD);
        map.put("CompleteWindowMinutes", COMPLETE_WINDOW_MINUTES_FIELD);
        map.put("Lifecycle", LIFECYCLE_FIELD);
        map.put("RecoveryPointTags", RECOVERY_POINT_TAGS_FIELD);
        map.put("BackupOptions", BACKUP_OPTIONS_FIELD);
        map.put("Index", INDEX_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StartBackupJobRequest, T> g) {
        return obj -> g.apply((StartBackupJobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends BackupRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartBackupJobRequest> {
        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the Amazon Web Services Region where they are created.
         * </p>
         * 
         * @param backupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the Amazon Web Services Region where they are
         *        created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultName(String backupVaultName);

        /**
         * <p>
         * The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided, supported
         * fully managed resources back up directly to logically air-gapped vault, while other supported resources
         * create a temporary (billable) snapshot in backup vault, then copy it to logically air-gapped vault.
         * Unsupported resources only back up to the specified backup vault.
         * </p>
         * 
         * @param logicallyAirGappedBackupVaultArn
         *        The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided,
         *        supported fully managed resources back up directly to logically air-gapped vault, while other
         *        supported resources create a temporary (billable) snapshot in backup vault, then copy it to logically
         *        air-gapped vault. Unsupported resources only back up to the specified backup vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicallyAirGappedBackupVaultArn(String logicallyAirGappedBackupVaultArn);

        /**
         * <p>
         * An Amazon Resource Name (ARN) that uniquely identifies a resource. The format of the ARN depends on the
         * resource type.
         * </p>
         * 
         * @param resourceArn
         *        An Amazon Resource Name (ARN) that uniquely identifies a resource. The format of the ARN depends on
         *        the resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * Specifies the IAM role ARN used to create the target recovery point; for example,
         * <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * </p>
         * 
         * @param iamRoleArn
         *        Specifies the IAM role ARN used to create the target recovery point; for example,
         *        <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * A customer-chosen string that you can use to distinguish between otherwise identical calls to
         * <code>StartBackupJob</code>. Retrying a successful request with the same idempotency token results in a
         * success message with no action taken.
         * </p>
         * 
         * @param idempotencyToken
         *        A customer-chosen string that you can use to distinguish between otherwise identical calls to
         *        <code>StartBackupJob</code>. Retrying a successful request with the same idempotency token results in
         *        a success message with no action taken.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idempotencyToken(String idempotencyToken);

        /**
         * <p>
         * A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start
         * successfully. This value is optional, and the default is 8 hours. If this value is included, it must be at
         * least 60 minutes to avoid errors.
         * </p>
         * <p>
         * This parameter has a maximum value of 100 years (52,560,000 minutes).
         * </p>
         * <p>
         * During the start window, the backup job status remains in <code>CREATED</code> status until it has
         * successfully begun or until the start window time has run out. If within the start window time Backup
         * receives an error that allows the job to be retried, Backup will automatically retry to begin the job at
         * least every 10 minutes until the backup successfully begins (the job status changes to <code>RUNNING</code>)
         * or until the job status changes to <code>EXPIRED</code> (which is expected to occur when the start window
         * time is over).
         * </p>
         * 
         * @param startWindowMinutes
         *        A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start
         *        successfully. This value is optional, and the default is 8 hours. If this value is included, it must
         *        be at least 60 minutes to avoid errors.</p>
         *        <p>
         *        This parameter has a maximum value of 100 years (52,560,000 minutes).
         *        </p>
         *        <p>
         *        During the start window, the backup job status remains in <code>CREATED</code> status until it has
         *        successfully begun or until the start window time has run out. If within the start window time Backup
         *        receives an error that allows the job to be retried, Backup will automatically retry to begin the job
         *        at least every 10 minutes until the backup successfully begins (the job status changes to
         *        <code>RUNNING</code>) or until the job status changes to <code>EXPIRED</code> (which is expected to
         *        occur when the start window time is over).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startWindowMinutes(Long startWindowMinutes);

        /**
         * <p>
         * A value in minutes during which a successfully started backup must complete, or else Backup will cancel the
         * job. This value is optional. This value begins counting down from when the backup was scheduled. It does not
         * add additional time for <code>StartWindowMinutes</code>, or if the backup started later than scheduled.
         * </p>
         * <p>
         * Like <code>StartWindowMinutes</code>, this parameter has a maximum value of 100 years (52,560,000 minutes).
         * </p>
         * 
         * @param completeWindowMinutes
         *        A value in minutes during which a successfully started backup must complete, or else Backup will
         *        cancel the job. This value is optional. This value begins counting down from when the backup was
         *        scheduled. It does not add additional time for <code>StartWindowMinutes</code>, or if the backup
         *        started later than scheduled.</p>
         *        <p>
         *        Like <code>StartWindowMinutes</code>, this parameter has a maximum value of 100 years (52,560,000
         *        minutes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completeWindowMinutes(Long completeWindowMinutes);

        /**
         * <p>
         * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup
         * will transition and expire backups automatically according to the lifecycle that you define.
         * </p>
         * <p>
         * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
         * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition
         * to cold after days” setting cannot be changed after a backup has been transitioned to cold.
         * </p>
         * <p>
         * Resource types that can transition to cold storage are listed in the <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
         * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
         * </p>
         * <p>
         * This parameter has a maximum value of 100 years (36,500 days).
         * </p>
         * 
         * @param lifecycle
         *        The lifecycle defines when a protected resource is transitioned to cold storage and when it expires.
         *        Backup will transition and expire backups automatically according to the lifecycle that you define.
         *        </p>
         *        <p>
         *        Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days.
         *        Therefore, the “retention” setting must be 90 days greater than the “transition to cold after days”
         *        setting. The “transition to cold after days” setting cannot be changed after a backup has been
         *        transitioned to cold.
         *        </p>
         *        <p>
         *        Resource types that can transition to cold storage are listed in the <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
         *        >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
         *        </p>
         *        <p>
         *        This parameter has a maximum value of 100 years (36,500 days).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycle(Lifecycle lifecycle);

        /**
         * <p>
         * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup
         * will transition and expire backups automatically according to the lifecycle that you define.
         * </p>
         * <p>
         * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
         * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition
         * to cold after days” setting cannot be changed after a backup has been transitioned to cold.
         * </p>
         * <p>
         * Resource types that can transition to cold storage are listed in the <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
         * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
         * </p>
         * <p>
         * This parameter has a maximum value of 100 years (36,500 days).
         * </p>
         * This is a convenience method that creates an instance of the {@link Lifecycle.Builder} avoiding the need to
         * create one manually via {@link Lifecycle#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Lifecycle.Builder#build()} is called immediately and its result
         * is passed to {@link #lifecycle(Lifecycle)}.
         * 
         * @param lifecycle
         *        a consumer that will call methods on {@link Lifecycle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycle(Lifecycle)
         */
        default Builder lifecycle(Consumer<Lifecycle.Builder> lifecycle) {
            return lifecycle(Lifecycle.builder().applyMutation(lifecycle).build());
        }

        /**
         * <p>
         * The tags to assign to the resources.
         * </p>
         * 
         * @param recoveryPointTags
         *        The tags to assign to the resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTags(Map<String, String> recoveryPointTags);

        /**
         * <p>
         * The backup option for a selected resource. This option is only available for Windows Volume Shadow Copy
         * Service (VSS) backup jobs.
         * </p>
         * <p>
         * Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option
         * and create a Windows VSS backup. Set to <code>"WindowsVSS""disabled"</code> to create a regular backup. The
         * <code>WindowsVSS</code> option is not enabled by default.
         * </p>
         * 
         * @param backupOptions
         *        The backup option for a selected resource. This option is only available for Windows Volume Shadow
         *        Copy Service (VSS) backup jobs.</p>
         *        <p>
         *        Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup
         *        option and create a Windows VSS backup. Set to <code>"WindowsVSS""disabled"</code> to create a regular
         *        backup. The <code>WindowsVSS</code> option is not enabled by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupOptions(Map<String, String> backupOptions);

        /**
         * <p>
         * Include this parameter to enable index creation if your backup job has a resource type that supports backup
         * indexes.
         * </p>
         * <p>
         * Resource types that support backup indexes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EBS</code> for Amazon Elastic Block Store
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
         * </p>
         * </li>
         * </ul>
         * <p>
         * Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
         * </p>
         * <p>
         * To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a backup
         * index, set value to <code>ENABLED</code>.
         * </p>
         * <p>
         * To delete a backup index, set value to <code>DISABLED</code>.
         * </p>
         * 
         * @param index
         *        Include this parameter to enable index creation if your backup job has a resource type that supports
         *        backup indexes.</p>
         *        <p>
         *        Resource types that support backup indexes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EBS</code> for Amazon Elastic Block Store
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
         *        </p>
         *        <p>
         *        To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a
         *        backup index, set value to <code>ENABLED</code>.
         *        </p>
         *        <p>
         *        To delete a backup index, set value to <code>DISABLED</code>.
         * @see Index
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Index
         */
        Builder index(String index);

        /**
         * <p>
         * Include this parameter to enable index creation if your backup job has a resource type that supports backup
         * indexes.
         * </p>
         * <p>
         * Resource types that support backup indexes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EBS</code> for Amazon Elastic Block Store
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
         * </p>
         * </li>
         * </ul>
         * <p>
         * Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
         * </p>
         * <p>
         * To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a backup
         * index, set value to <code>ENABLED</code>.
         * </p>
         * <p>
         * To delete a backup index, set value to <code>DISABLED</code>.
         * </p>
         * 
         * @param index
         *        Include this parameter to enable index creation if your backup job has a resource type that supports
         *        backup indexes.</p>
         *        <p>
         *        Resource types that support backup indexes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EBS</code> for Amazon Elastic Block Store
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code> for Amazon Simple Storage Service (Amazon S3)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Index can have 1 of 2 possible values, either <code>ENABLED</code> or <code>DISABLED</code>.
         *        </p>
         *        <p>
         *        To create a backup index for an eligible <code>ACTIVE</code> recovery point that does not yet have a
         *        backup index, set value to <code>ENABLED</code>.
         *        </p>
         *        <p>
         *        To delete a backup index, set value to <code>DISABLED</code>.
         * @see Index
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Index
         */
        Builder index(Index index);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BackupRequest.BuilderImpl implements Builder {
        private String backupVaultName;

        private String logicallyAirGappedBackupVaultArn;

        private String resourceArn;

        private String iamRoleArn;

        private String idempotencyToken;

        private Long startWindowMinutes;

        private Long completeWindowMinutes;

        private Lifecycle lifecycle;

        private Map<String, String> recoveryPointTags = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> backupOptions = DefaultSdkAutoConstructMap.getInstance();

        private String index;

        private BuilderImpl() {
        }

        private BuilderImpl(StartBackupJobRequest model) {
            super(model);
            backupVaultName(model.backupVaultName);
            logicallyAirGappedBackupVaultArn(model.logicallyAirGappedBackupVaultArn);
            resourceArn(model.resourceArn);
            iamRoleArn(model.iamRoleArn);
            idempotencyToken(model.idempotencyToken);
            startWindowMinutes(model.startWindowMinutes);
            completeWindowMinutes(model.completeWindowMinutes);
            lifecycle(model.lifecycle);
            recoveryPointTags(model.recoveryPointTags);
            backupOptions(model.backupOptions);
            index(model.index);
        }

        public final String getBackupVaultName() {
            return backupVaultName;
        }

        public final void setBackupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
        }

        @Override
        public final Builder backupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
            return this;
        }

        public final String getLogicallyAirGappedBackupVaultArn() {
            return logicallyAirGappedBackupVaultArn;
        }

        public final void setLogicallyAirGappedBackupVaultArn(String logicallyAirGappedBackupVaultArn) {
            this.logicallyAirGappedBackupVaultArn = logicallyAirGappedBackupVaultArn;
        }

        @Override
        public final Builder logicallyAirGappedBackupVaultArn(String logicallyAirGappedBackupVaultArn) {
            this.logicallyAirGappedBackupVaultArn = logicallyAirGappedBackupVaultArn;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final String getIdempotencyToken() {
            return idempotencyToken;
        }

        public final void setIdempotencyToken(String idempotencyToken) {
            this.idempotencyToken = idempotencyToken;
        }

        @Override
        public final Builder idempotencyToken(String idempotencyToken) {
            this.idempotencyToken = idempotencyToken;
            return this;
        }

        public final Long getStartWindowMinutes() {
            return startWindowMinutes;
        }

        public final void setStartWindowMinutes(Long startWindowMinutes) {
            this.startWindowMinutes = startWindowMinutes;
        }

        @Override
        public final Builder startWindowMinutes(Long startWindowMinutes) {
            this.startWindowMinutes = startWindowMinutes;
            return this;
        }

        public final Long getCompleteWindowMinutes() {
            return completeWindowMinutes;
        }

        public final void setCompleteWindowMinutes(Long completeWindowMinutes) {
            this.completeWindowMinutes = completeWindowMinutes;
        }

        @Override
        public final Builder completeWindowMinutes(Long completeWindowMinutes) {
            this.completeWindowMinutes = completeWindowMinutes;
            return this;
        }

        public final Lifecycle.Builder getLifecycle() {
            return lifecycle != null ? lifecycle.toBuilder() : null;
        }

        public final void setLifecycle(Lifecycle.BuilderImpl lifecycle) {
            this.lifecycle = lifecycle != null ? lifecycle.build() : null;
        }

        @Override
        public final Builder lifecycle(Lifecycle lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        public final Map<String, String> getRecoveryPointTags() {
            if (recoveryPointTags instanceof SdkAutoConstructMap) {
                return null;
            }
            return recoveryPointTags;
        }

        public final void setRecoveryPointTags(Map<String, String> recoveryPointTags) {
            this.recoveryPointTags = TagsCopier.copy(recoveryPointTags);
        }

        @Override
        public final Builder recoveryPointTags(Map<String, String> recoveryPointTags) {
            this.recoveryPointTags = TagsCopier.copy(recoveryPointTags);
            return this;
        }

        public final Map<String, String> getBackupOptions() {
            if (backupOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return backupOptions;
        }

        public final void setBackupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
        }

        @Override
        public final Builder backupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
            return this;
        }

        public final String getIndex() {
            return index;
        }

        public final void setIndex(String index) {
            this.index = index;
        }

        @Override
        public final Builder index(String index) {
            this.index = index;
            return this;
        }

        @Override
        public final Builder index(Index index) {
            this.index(index == null ? null : index.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartBackupJobRequest build() {
            return new StartBackupJobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
