/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration settings for malware scanning, including the scanner type, target resource types, and scanner
 * role.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScanSetting implements SdkPojo, Serializable, ToCopyableBuilder<ScanSetting.Builder, ScanSetting> {
    private static final SdkField<String> MALWARE_SCANNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MalwareScanner").getter(getter(ScanSetting::malwareScannerAsString))
            .setter(setter(Builder::malwareScanner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MalwareScanner").build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypes")
            .getter(getter(ScanSetting::resourceTypes))
            .setter(setter(Builder::resourceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SCANNER_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScannerRoleArn").getter(getter(ScanSetting::scannerRoleArn)).setter(setter(Builder::scannerRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScannerRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MALWARE_SCANNER_FIELD,
            RESOURCE_TYPES_FIELD, SCANNER_ROLE_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String malwareScanner;

    private final List<String> resourceTypes;

    private final String scannerRoleArn;

    private ScanSetting(BuilderImpl builder) {
        this.malwareScanner = builder.malwareScanner;
        this.resourceTypes = builder.resourceTypes;
        this.scannerRoleArn = builder.scannerRoleArn;
    }

    /**
     * <p>
     * The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #malwareScanner}
     * will return {@link MalwareScanner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #malwareScannerAsString}.
     * </p>
     * 
     * @return The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
     * @see MalwareScanner
     */
    public final MalwareScanner malwareScanner() {
        return MalwareScanner.fromValue(malwareScanner);
    }

    /**
     * <p>
     * The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #malwareScanner}
     * will return {@link MalwareScanner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #malwareScannerAsString}.
     * </p>
     * 
     * @return The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
     * @see MalwareScanner
     */
    public final String malwareScannerAsString() {
        return malwareScanner;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of resource types to be scanned for malware.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return An array of resource types to be scanned for malware.
     */
    public final List<String> resourceTypes() {
        return resourceTypes;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that the scanner uses to access resources; for example,
     * <code>arn:aws:iam::123456789012:role/ScannerRole</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that the scanner uses to access resources; for example,
     *         <code>arn:aws:iam::123456789012:role/ScannerRole</code>.
     */
    public final String scannerRoleArn() {
        return scannerRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(malwareScannerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scannerRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScanSetting)) {
            return false;
        }
        ScanSetting other = (ScanSetting) obj;
        return Objects.equals(malwareScannerAsString(), other.malwareScannerAsString())
                && hasResourceTypes() == other.hasResourceTypes() && Objects.equals(resourceTypes(), other.resourceTypes())
                && Objects.equals(scannerRoleArn(), other.scannerRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScanSetting").add("MalwareScanner", malwareScannerAsString())
                .add("ResourceTypes", hasResourceTypes() ? resourceTypes() : null).add("ScannerRoleArn", scannerRoleArn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MalwareScanner":
            return Optional.ofNullable(clazz.cast(malwareScannerAsString()));
        case "ResourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypes()));
        case "ScannerRoleArn":
            return Optional.ofNullable(clazz.cast(scannerRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MalwareScanner", MALWARE_SCANNER_FIELD);
        map.put("ResourceTypes", RESOURCE_TYPES_FIELD);
        map.put("ScannerRoleArn", SCANNER_ROLE_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ScanSetting, T> g) {
        return obj -> g.apply((ScanSetting) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScanSetting> {
        /**
         * <p>
         * The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
         * </p>
         * 
         * @param malwareScanner
         *        The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
         * @see MalwareScanner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MalwareScanner
         */
        Builder malwareScanner(String malwareScanner);

        /**
         * <p>
         * The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
         * </p>
         * 
         * @param malwareScanner
         *        The malware scanner to use for scanning. Currently only <code>GUARDDUTY</code> is supported.
         * @see MalwareScanner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MalwareScanner
         */
        Builder malwareScanner(MalwareScanner malwareScanner);

        /**
         * <p>
         * An array of resource types to be scanned for malware.
         * </p>
         * 
         * @param resourceTypes
         *        An array of resource types to be scanned for malware.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<String> resourceTypes);

        /**
         * <p>
         * An array of resource types to be scanned for malware.
         * </p>
         * 
         * @param resourceTypes
         *        An array of resource types to be scanned for malware.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(String... resourceTypes);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that the scanner uses to access resources; for example,
         * <code>arn:aws:iam::123456789012:role/ScannerRole</code>.
         * </p>
         * 
         * @param scannerRoleArn
         *        The Amazon Resource Name (ARN) of the IAM role that the scanner uses to access resources; for example,
         *        <code>arn:aws:iam::123456789012:role/ScannerRole</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scannerRoleArn(String scannerRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String malwareScanner;

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private String scannerRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ScanSetting model) {
            malwareScanner(model.malwareScanner);
            resourceTypes(model.resourceTypes);
            scannerRoleArn(model.scannerRoleArn);
        }

        public final String getMalwareScanner() {
            return malwareScanner;
        }

        public final void setMalwareScanner(String malwareScanner) {
            this.malwareScanner = malwareScanner;
        }

        @Override
        public final Builder malwareScanner(String malwareScanner) {
            this.malwareScanner = malwareScanner;
            return this;
        }

        @Override
        public final Builder malwareScanner(MalwareScanner malwareScanner) {
            this.malwareScanner(malwareScanner == null ? null : malwareScanner.toString());
            return this;
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypesCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypesCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(String... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final String getScannerRoleArn() {
            return scannerRoleArn;
        }

        public final void setScannerRoleArn(String scannerRoleArn) {
            this.scannerRoleArn = scannerRoleArn;
        }

        @Override
        public final Builder scannerRoleArn(String scannerRoleArn) {
            this.scannerRoleArn = scannerRoleArn;
            return this;
        }

        @Override
        public ScanSetting build() {
            return new ScanSetting(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
