/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth.credentials.internal;

import java.lang.reflect.InvocationTargetException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProviderChain;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.credentials.ChildProfileCredentialsProviderFactory;
import software.amazon.awssdk.auth.credentials.ContainerCredentialsProvider;
import software.amazon.awssdk.auth.credentials.EnvironmentVariableCredentialsProvider;
import software.amazon.awssdk.auth.credentials.InstanceProfileCredentialsProvider;
import software.amazon.awssdk.auth.credentials.ProcessCredentialsProvider;
import software.amazon.awssdk.auth.credentials.ProfileCredentialsProviderFactory;
import software.amazon.awssdk.auth.credentials.ProfileProviderCredentialsContext;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.auth.credentials.SystemPropertyCredentialsProvider;
import software.amazon.awssdk.auth.credentials.internal.CredentialSourceType;
import software.amazon.awssdk.auth.credentials.internal.WebIdentityCredentialsUtils;
import software.amazon.awssdk.auth.credentials.internal.WebIdentityTokenCredentialProperties;
import software.amazon.awssdk.core.internal.util.ClassLoaderHelper;
import software.amazon.awssdk.core.useragent.BusinessMetricFeatureId;
import software.amazon.awssdk.profiles.Profile;
import software.amazon.awssdk.profiles.ProfileFile;
import software.amazon.awssdk.profiles.internal.ProfileSection;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public final class ProfileCredentialsUtils {
    private static final String STS_PROFILE_CREDENTIALS_PROVIDER_FACTORY = "software.amazon.awssdk.services.sts.internal.StsProfileCredentialsProviderFactory";
    private static final String SSO_PROFILE_CREDENTIALS_PROVIDER_FACTORY = "software.amazon.awssdk.services.sso.auth.SsoProfileCredentialsProviderFactory";
    private final ProfileFile profileFile;
    private final Profile profile;
    private final String name;
    private final Map<String, String> properties;
    private final Function<String, Optional<Profile>> credentialsSourceResolver;

    public ProfileCredentialsUtils(ProfileFile profileFile, Profile profile, Function<String, Optional<Profile>> credentialsSourceResolver) {
        this.profileFile = (ProfileFile)Validate.paramNotNull((Object)profileFile, (String)"profileFile");
        this.profile = (Profile)Validate.paramNotNull((Object)profile, (String)"profile");
        this.name = profile.name();
        this.properties = profile.properties();
        this.credentialsSourceResolver = credentialsSourceResolver;
    }

    public Optional<AwsCredentialsProvider> credentialsProvider() {
        return this.credentialsProvider(new HashSet<String>());
    }

    private Optional<AwsCredentialsProvider> credentialsProvider(Set<String> children) {
        return this.credentialsProviderWithFeatureID(children).map(CredentialsWithFeatureId::provider);
    }

    private Optional<CredentialsWithFeatureId> credentialsProviderWithFeatureID(Set<String> children) {
        if (this.properties.containsKey("role_arn") && this.properties.containsKey("web_identity_token_file")) {
            return Optional.of(this.roleAndWebIdentityTokenProfileCredentialsProvider());
        }
        if (this.properties.containsKey("sso_role_name") || this.properties.containsKey("sso_account_id") || this.properties.containsKey("sso_region") || this.properties.containsKey("sso_start_url") || this.properties.containsKey(ProfileSection.SSO_SESSION.getPropertyKeyName())) {
            return Optional.of(this.ssoProfileCredentialsProvider());
        }
        if (this.properties.containsKey("role_arn")) {
            boolean hasSourceProfile = this.properties.containsKey("source_profile");
            boolean hasCredentialSource = this.properties.containsKey("credential_source");
            Validate.validState((!hasSourceProfile || !hasCredentialSource ? 1 : 0) != 0, (String)"Invalid profile file: profile has both %s and %s.", (Object[])new Object[]{"source_profile", "credential_source"});
            if (hasSourceProfile) {
                return Optional.of(this.roleAndSourceProfileBasedProfileCredentialsProvider(children));
            }
            if (hasCredentialSource) {
                return Optional.of(this.roleAndCredentialSourceBasedProfileCredentialsProvider());
            }
        }
        if (this.properties.containsKey("credential_process")) {
            return Optional.of(this.credentialProcessCredentialsProvider());
        }
        if (this.properties.containsKey("aws_session_token")) {
            return Optional.of(this.sessionProfileCredentialsProvider());
        }
        if (this.properties.containsKey("aws_access_key_id")) {
            return Optional.of(this.basicProfileCredentialsProvider());
        }
        return Optional.empty();
    }

    private CredentialsWithFeatureId basicProfileCredentialsProvider() {
        this.requireProperties("aws_access_key_id", "aws_secret_access_key");
        String featureId = BusinessMetricFeatureId.CREDENTIALS_PROFILE.value();
        AwsBasicCredentials credentials = AwsBasicCredentials.builder().accessKeyId(this.properties.get("aws_access_key_id")).secretAccessKey(this.properties.get("aws_secret_access_key")).accountId(this.properties.get("aws_account_id")).providerName(featureId).build();
        return new CredentialsWithFeatureId(StaticCredentialsProvider.create(credentials), featureId);
    }

    private CredentialsWithFeatureId sessionProfileCredentialsProvider() {
        this.requireProperties("aws_access_key_id", "aws_secret_access_key", "aws_session_token");
        String featureId = BusinessMetricFeatureId.CREDENTIALS_PROFILE.value();
        AwsSessionCredentials credentials = AwsSessionCredentials.builder().accessKeyId(this.properties.get("aws_access_key_id")).secretAccessKey(this.properties.get("aws_secret_access_key")).sessionToken(this.properties.get("aws_session_token")).accountId(this.properties.get("aws_account_id")).providerName(featureId).build();
        return new CredentialsWithFeatureId(StaticCredentialsProvider.create(credentials), featureId);
    }

    private CredentialsWithFeatureId credentialProcessCredentialsProvider() {
        this.requireProperties("credential_process");
        String featureId = BusinessMetricFeatureId.CREDENTIALS_PROFILE_PROCESS.value();
        ProcessCredentialsProvider provider = ProcessCredentialsProvider.builder().command(this.properties.get("credential_process")).staticAccountId(this.properties.get("aws_account_id")).sourceChain(featureId).build();
        return new CredentialsWithFeatureId(provider, featureId);
    }

    private CredentialsWithFeatureId ssoProfileCredentialsProvider() {
        this.validateRequiredPropertiesForSsoCredentialsProvider();
        boolean isLegacy = this.isLegacySsoConfiguration();
        String featureId = isLegacy ? BusinessMetricFeatureId.CREDENTIALS_PROFILE_SSO_LEGACY.value() : BusinessMetricFeatureId.CREDENTIALS_PROFILE_SSO.value();
        AwsCredentialsProvider provider = this.ssoCredentialsProviderFactory().create(ProfileProviderCredentialsContext.builder().profile(this.profile).profileFile(this.profileFile).sourceChain(featureId).build());
        return new CredentialsWithFeatureId(provider, featureId);
    }

    private void validateRequiredPropertiesForSsoCredentialsProvider() {
        this.requireProperties("sso_account_id", "sso_role_name");
        if (!this.properties.containsKey(ProfileSection.SSO_SESSION.getPropertyKeyName())) {
            this.requireProperties("sso_region", "sso_start_url");
        }
    }

    private boolean isLegacySsoConfiguration() {
        return !this.properties.containsKey(ProfileSection.SSO_SESSION.getPropertyKeyName());
    }

    private CredentialsWithFeatureId roleAndWebIdentityTokenProfileCredentialsProvider() {
        this.requireProperties("role_arn", "web_identity_token_file");
        String featureId = BusinessMetricFeatureId.CREDENTIALS_PROFILE_STS_WEB_ID_TOKEN.value();
        String roleArn = this.properties.get("role_arn");
        String roleSessionName = this.properties.get("role_session_name");
        Path webIdentityTokenFile = Paths.get(this.properties.get("web_identity_token_file"), new String[0]);
        WebIdentityTokenCredentialProperties credentialProperties = WebIdentityTokenCredentialProperties.builder().roleArn(roleArn).roleSessionName(roleSessionName).webIdentityTokenFile(webIdentityTokenFile).sourceChain(featureId).build();
        return new CredentialsWithFeatureId(WebIdentityCredentialsUtils.factory().create(credentialProperties), featureId);
    }

    private CredentialsWithFeatureId roleAndSourceProfileBasedProfileCredentialsProvider(Set<String> children) {
        this.requireProperties("source_profile");
        Validate.validState((!children.contains(this.name) ? 1 : 0) != 0, (String)"Invalid profile file: Circular relationship detected with profiles %s.", (Object[])new Object[]{children});
        Validate.validState((this.credentialsSourceResolver != null ? 1 : 0) != 0, (String)"The profile '%s' must be configured with a source profile in order to use assumed roles.", (Object[])new Object[]{this.name});
        children.add(this.name);
        Optional sourceResult = this.credentialsSourceResolver.apply(this.properties.get("source_profile")).flatMap(p -> new ProfileCredentialsUtils(this.profileFile, (Profile)p, this.credentialsSourceResolver).credentialsProviderWithFeatureID(children));
        if (!sourceResult.isPresent()) {
            throw this.noSourceCredentialsException();
        }
        CredentialsWithFeatureId source = (CredentialsWithFeatureId)sourceResult.get();
        String profileMetric = BusinessMetricFeatureId.CREDENTIALS_PROFILE_SOURCE_PROFILE.value();
        String combinedMetrics = profileMetric + "," + source.featureId();
        AwsCredentialsProvider stsProvider = this.createStsCredentialsProviderWithFeatureID(source.provider(), combinedMetrics);
        return new CredentialsWithFeatureId(stsProvider, combinedMetrics);
    }

    private CredentialsWithFeatureId roleAndCredentialSourceBasedProfileCredentialsProvider() {
        this.requireProperties("credential_source");
        CredentialSourceType credentialSource = CredentialSourceType.parse(this.properties.get("credential_source"));
        String profileMetric = BusinessMetricFeatureId.CREDENTIALS_PROFILE_NAMED_PROVIDER.value();
        CredentialsWithFeatureId sourceResult = this.credentialSourceCredentialProvider(credentialSource);
        String combinedMetrics = profileMetric + "," + sourceResult.featureId();
        AwsCredentialsProvider stsProvider = this.createStsCredentialsProviderWithFeatureID(sourceResult.provider(), combinedMetrics);
        return new CredentialsWithFeatureId(stsProvider, combinedMetrics);
    }

    private AwsCredentialsProvider createStsCredentialsProviderWithFeatureID(AwsCredentialsProvider sourceProvider, String combinedMetrics) {
        ChildProfileCredentialsProviderFactory.ChildProfileCredentialsRequest request = ChildProfileCredentialsProviderFactory.ChildProfileCredentialsRequest.builder().sourceCredentialsProvider(sourceProvider).profile(this.profile).sourceChain(combinedMetrics).build();
        return this.stsCredentialsProviderFactory().create(request);
    }

    private CredentialsWithFeatureId credentialSourceCredentialProvider(CredentialSourceType credentialSource) {
        switch (credentialSource) {
            case ECS_CONTAINER: {
                return new CredentialsWithFeatureId((AwsCredentialsProvider)ContainerCredentialsProvider.builder().build(), BusinessMetricFeatureId.CREDENTIALS_HTTP.value());
            }
            case EC2_INSTANCE_METADATA: {
                return new CredentialsWithFeatureId(InstanceProfileCredentialsProvider.builder().profileFile(this.profileFile).profileName(this.name).build(), BusinessMetricFeatureId.CREDENTIALS_IMDS.value());
            }
            case ENVIRONMENT: {
                return new CredentialsWithFeatureId(AwsCredentialsProviderChain.builder().addCredentialsProvider(SystemPropertyCredentialsProvider.create()).addCredentialsProvider(EnvironmentVariableCredentialsProvider.create()).build(), BusinessMetricFeatureId.CREDENTIALS_ENV_VARS.value());
            }
        }
        throw this.noSourceCredentialsException();
    }

    private void requireProperties(String ... requiredProperties) {
        Arrays.stream(requiredProperties).forEach(p -> Validate.isTrue((boolean)this.properties.containsKey(p), (String)"Profile property '%s' was not configured for '%s'.", (Object[])new Object[]{p, this.name}));
    }

    private IllegalStateException noSourceCredentialsException() {
        String error = String.format("The source profile of '%s' was configured to be '%s', but that source profile has no credentials configured.", this.name, this.properties.get("source_profile"));
        return new IllegalStateException(error);
    }

    private ChildProfileCredentialsProviderFactory stsCredentialsProviderFactory() {
        try {
            Class stsCredentialsProviderFactory = ClassLoaderHelper.loadClass((String)STS_PROFILE_CREDENTIALS_PROVIDER_FACTORY, (Class[])new Class[]{this.getClass()});
            return (ChildProfileCredentialsProviderFactory)stsCredentialsProviderFactory.getConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("To use assumed roles in the '" + this.name + "' profile, the 'sts' service module must be on the class path.", e);
        }
        catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            throw new IllegalStateException("Failed to create the '" + this.name + "' profile credentials provider.", e);
        }
    }

    private ProfileCredentialsProviderFactory ssoCredentialsProviderFactory() {
        try {
            Class ssoProfileCredentialsProviderFactory = ClassLoaderHelper.loadClass((String)SSO_PROFILE_CREDENTIALS_PROVIDER_FACTORY, (Class[])new Class[]{this.getClass()});
            return (ProfileCredentialsProviderFactory)ssoProfileCredentialsProviderFactory.getConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("To use Sso related properties in the '" + this.name + "' profile, the 'sso' service module must be on the class path.", e);
        }
        catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            throw new IllegalStateException("Failed to create the '" + this.name + "' profile credentials provider.", e);
        }
    }

    private static final class CredentialsWithFeatureId {
        private final AwsCredentialsProvider provider;
        private final String featureId;

        CredentialsWithFeatureId(AwsCredentialsProvider provider, String featureId) {
            this.provider = provider;
            this.featureId = featureId;
        }

        AwsCredentialsProvider provider() {
            return this.provider;
        }

        String featureId() {
            return this.featureId;
        }
    }
}

