/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationdiscovery.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A metadata object that represents the deletion task being executed.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchDeleteConfigurationTask implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchDeleteConfigurationTask.Builder, BatchDeleteConfigurationTask> {
    private static final SdkField<String> TASK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("taskId")
            .getter(getter(BatchDeleteConfigurationTask::taskId)).setter(setter(Builder::taskId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(BatchDeleteConfigurationTask::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(BatchDeleteConfigurationTask::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(BatchDeleteConfigurationTask::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> CONFIGURATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("configurationType").getter(getter(BatchDeleteConfigurationTask::configurationTypeAsString))
            .setter(setter(Builder::configurationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationType").build()).build();

    private static final SdkField<List<String>> REQUESTED_CONFIGURATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("requestedConfigurations")
            .getter(getter(BatchDeleteConfigurationTask::requestedConfigurations))
            .setter(setter(Builder::requestedConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestedConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DELETED_CONFIGURATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("deletedConfigurations")
            .getter(getter(BatchDeleteConfigurationTask::deletedConfigurations))
            .setter(setter(Builder::deletedConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deletedConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<FailedConfiguration>> FAILED_CONFIGURATIONS_FIELD = SdkField
            .<List<FailedConfiguration>> builder(MarshallingType.LIST)
            .memberName("failedConfigurations")
            .getter(getter(BatchDeleteConfigurationTask::failedConfigurations))
            .setter(setter(Builder::failedConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failedConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FailedConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(FailedConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DeletionWarning>> DELETION_WARNINGS_FIELD = SdkField
            .<List<DeletionWarning>> builder(MarshallingType.LIST)
            .memberName("deletionWarnings")
            .getter(getter(BatchDeleteConfigurationTask::deletionWarnings))
            .setter(setter(Builder::deletionWarnings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deletionWarnings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DeletionWarning> builder(MarshallingType.SDK_POJO)
                                            .constructor(DeletionWarning::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TASK_ID_FIELD, STATUS_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, CONFIGURATION_TYPE_FIELD, REQUESTED_CONFIGURATIONS_FIELD,
            DELETED_CONFIGURATIONS_FIELD, FAILED_CONFIGURATIONS_FIELD, DELETION_WARNINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String taskId;

    private final String status;

    private final Instant startTime;

    private final Instant endTime;

    private final String configurationType;

    private final List<String> requestedConfigurations;

    private final List<String> deletedConfigurations;

    private final List<FailedConfiguration> failedConfigurations;

    private final List<DeletionWarning> deletionWarnings;

    private BatchDeleteConfigurationTask(BuilderImpl builder) {
        this.taskId = builder.taskId;
        this.status = builder.status;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.configurationType = builder.configurationType;
        this.requestedConfigurations = builder.requestedConfigurations;
        this.deletedConfigurations = builder.deletedConfigurations;
        this.failedConfigurations = builder.failedConfigurations;
        this.deletionWarnings = builder.deletionWarnings;
    }

    /**
     * <p>
     * The deletion task's unique identifier.
     * </p>
     * 
     * @return The deletion task's unique identifier.
     */
    public final String taskId() {
        return taskId;
    }

    /**
     * <p>
     * The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING | DELETING |
     * COMPLETED | FAILED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BatchDeleteConfigurationTaskStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING | DELETING
     *         | COMPLETED | FAILED.
     * @see BatchDeleteConfigurationTaskStatus
     */
    public final BatchDeleteConfigurationTaskStatus status() {
        return BatchDeleteConfigurationTaskStatus.fromValue(status);
    }

    /**
     * <p>
     * The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING | DELETING |
     * COMPLETED | FAILED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BatchDeleteConfigurationTaskStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING | DELETING
     *         | COMPLETED | FAILED.
     * @see BatchDeleteConfigurationTaskStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * An epoch seconds timestamp (UTC) of when the deletion task was started.
     * </p>
     * 
     * @return An epoch seconds timestamp (UTC) of when the deletion task was started.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * An epoch seconds timestamp (UTC) of when the deletion task was completed or failed.
     * </p>
     * 
     * @return An epoch seconds timestamp (UTC) of when the deletion task was completed or failed.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The type of configuration item to delete. Supported types are: SERVER.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #configurationType}
     * will return {@link DeletionConfigurationItemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #configurationTypeAsString}.
     * </p>
     * 
     * @return The type of configuration item to delete. Supported types are: SERVER.
     * @see DeletionConfigurationItemType
     */
    public final DeletionConfigurationItemType configurationType() {
        return DeletionConfigurationItemType.fromValue(configurationType);
    }

    /**
     * <p>
     * The type of configuration item to delete. Supported types are: SERVER.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #configurationType}
     * will return {@link DeletionConfigurationItemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #configurationTypeAsString}.
     * </p>
     * 
     * @return The type of configuration item to delete. Supported types are: SERVER.
     * @see DeletionConfigurationItemType
     */
    public final String configurationTypeAsString() {
        return configurationType;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestedConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRequestedConfigurations() {
        return requestedConfigurations != null && !(requestedConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of configuration IDs that were originally requested to be deleted by the deletion task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestedConfigurations} method.
     * </p>
     * 
     * @return The list of configuration IDs that were originally requested to be deleted by the deletion task.
     */
    public final List<String> requestedConfigurations() {
        return requestedConfigurations;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeletedConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDeletedConfigurations() {
        return deletedConfigurations != null && !(deletedConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of configuration IDs that were successfully deleted by the deletion task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeletedConfigurations} method.
     * </p>
     * 
     * @return The list of configuration IDs that were successfully deleted by the deletion task.
     */
    public final List<String> deletedConfigurations() {
        return deletedConfigurations;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailedConfigurations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailedConfigurations() {
        return failedConfigurations != null && !(failedConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of configuration IDs that failed to delete during the deletion task, each paired with an error message.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailedConfigurations} method.
     * </p>
     * 
     * @return A list of configuration IDs that failed to delete during the deletion task, each paired with an error
     *         message.
     */
    public final List<FailedConfiguration> failedConfigurations() {
        return failedConfigurations;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeletionWarnings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeletionWarnings() {
        return deletionWarnings != null && !(deletionWarnings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of configuration IDs that produced warnings regarding their deletion, paired with a warning message.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeletionWarnings} method.
     * </p>
     * 
     * @return A list of configuration IDs that produced warnings regarding their deletion, paired with a warning
     *         message.
     */
    public final List<DeletionWarning> deletionWarnings() {
        return deletionWarnings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(taskId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(configurationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestedConfigurations() ? requestedConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDeletedConfigurations() ? deletedConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFailedConfigurations() ? failedConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDeletionWarnings() ? deletionWarnings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchDeleteConfigurationTask)) {
            return false;
        }
        BatchDeleteConfigurationTask other = (BatchDeleteConfigurationTask) obj;
        return Objects.equals(taskId(), other.taskId()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(configurationTypeAsString(), other.configurationTypeAsString())
                && hasRequestedConfigurations() == other.hasRequestedConfigurations()
                && Objects.equals(requestedConfigurations(), other.requestedConfigurations())
                && hasDeletedConfigurations() == other.hasDeletedConfigurations()
                && Objects.equals(deletedConfigurations(), other.deletedConfigurations())
                && hasFailedConfigurations() == other.hasFailedConfigurations()
                && Objects.equals(failedConfigurations(), other.failedConfigurations())
                && hasDeletionWarnings() == other.hasDeletionWarnings()
                && Objects.equals(deletionWarnings(), other.deletionWarnings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchDeleteConfigurationTask").add("TaskId", taskId()).add("Status", statusAsString())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("ConfigurationType", configurationTypeAsString())
                .add("RequestedConfigurations", hasRequestedConfigurations() ? requestedConfigurations() : null)
                .add("DeletedConfigurations", hasDeletedConfigurations() ? deletedConfigurations() : null)
                .add("FailedConfigurations", hasFailedConfigurations() ? failedConfigurations() : null)
                .add("DeletionWarnings", hasDeletionWarnings() ? deletionWarnings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "taskId":
            return Optional.ofNullable(clazz.cast(taskId()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "configurationType":
            return Optional.ofNullable(clazz.cast(configurationTypeAsString()));
        case "requestedConfigurations":
            return Optional.ofNullable(clazz.cast(requestedConfigurations()));
        case "deletedConfigurations":
            return Optional.ofNullable(clazz.cast(deletedConfigurations()));
        case "failedConfigurations":
            return Optional.ofNullable(clazz.cast(failedConfigurations()));
        case "deletionWarnings":
            return Optional.ofNullable(clazz.cast(deletionWarnings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("taskId", TASK_ID_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("endTime", END_TIME_FIELD);
        map.put("configurationType", CONFIGURATION_TYPE_FIELD);
        map.put("requestedConfigurations", REQUESTED_CONFIGURATIONS_FIELD);
        map.put("deletedConfigurations", DELETED_CONFIGURATIONS_FIELD);
        map.put("failedConfigurations", FAILED_CONFIGURATIONS_FIELD);
        map.put("deletionWarnings", DELETION_WARNINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BatchDeleteConfigurationTask, T> g) {
        return obj -> g.apply((BatchDeleteConfigurationTask) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchDeleteConfigurationTask> {
        /**
         * <p>
         * The deletion task's unique identifier.
         * </p>
         * 
         * @param taskId
         *        The deletion task's unique identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskId(String taskId);

        /**
         * <p>
         * The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING | DELETING |
         * COMPLETED | FAILED.
         * </p>
         * 
         * @param status
         *        The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING |
         *        DELETING | COMPLETED | FAILED.
         * @see BatchDeleteConfigurationTaskStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BatchDeleteConfigurationTaskStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING | DELETING |
         * COMPLETED | FAILED.
         * </p>
         * 
         * @param status
         *        The current execution status of the deletion task. Valid status are: INITIALIZING | VALIDATING |
         *        DELETING | COMPLETED | FAILED.
         * @see BatchDeleteConfigurationTaskStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BatchDeleteConfigurationTaskStatus
         */
        Builder status(BatchDeleteConfigurationTaskStatus status);

        /**
         * <p>
         * An epoch seconds timestamp (UTC) of when the deletion task was started.
         * </p>
         * 
         * @param startTime
         *        An epoch seconds timestamp (UTC) of when the deletion task was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * An epoch seconds timestamp (UTC) of when the deletion task was completed or failed.
         * </p>
         * 
         * @param endTime
         *        An epoch seconds timestamp (UTC) of when the deletion task was completed or failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The type of configuration item to delete. Supported types are: SERVER.
         * </p>
         * 
         * @param configurationType
         *        The type of configuration item to delete. Supported types are: SERVER.
         * @see DeletionConfigurationItemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeletionConfigurationItemType
         */
        Builder configurationType(String configurationType);

        /**
         * <p>
         * The type of configuration item to delete. Supported types are: SERVER.
         * </p>
         * 
         * @param configurationType
         *        The type of configuration item to delete. Supported types are: SERVER.
         * @see DeletionConfigurationItemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeletionConfigurationItemType
         */
        Builder configurationType(DeletionConfigurationItemType configurationType);

        /**
         * <p>
         * The list of configuration IDs that were originally requested to be deleted by the deletion task.
         * </p>
         * 
         * @param requestedConfigurations
         *        The list of configuration IDs that were originally requested to be deleted by the deletion task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedConfigurations(Collection<String> requestedConfigurations);

        /**
         * <p>
         * The list of configuration IDs that were originally requested to be deleted by the deletion task.
         * </p>
         * 
         * @param requestedConfigurations
         *        The list of configuration IDs that were originally requested to be deleted by the deletion task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedConfigurations(String... requestedConfigurations);

        /**
         * <p>
         * The list of configuration IDs that were successfully deleted by the deletion task.
         * </p>
         * 
         * @param deletedConfigurations
         *        The list of configuration IDs that were successfully deleted by the deletion task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletedConfigurations(Collection<String> deletedConfigurations);

        /**
         * <p>
         * The list of configuration IDs that were successfully deleted by the deletion task.
         * </p>
         * 
         * @param deletedConfigurations
         *        The list of configuration IDs that were successfully deleted by the deletion task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletedConfigurations(String... deletedConfigurations);

        /**
         * <p>
         * A list of configuration IDs that failed to delete during the deletion task, each paired with an error
         * message.
         * </p>
         * 
         * @param failedConfigurations
         *        A list of configuration IDs that failed to delete during the deletion task, each paired with an error
         *        message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedConfigurations(Collection<FailedConfiguration> failedConfigurations);

        /**
         * <p>
         * A list of configuration IDs that failed to delete during the deletion task, each paired with an error
         * message.
         * </p>
         * 
         * @param failedConfigurations
         *        A list of configuration IDs that failed to delete during the deletion task, each paired with an error
         *        message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedConfigurations(FailedConfiguration... failedConfigurations);

        /**
         * <p>
         * A list of configuration IDs that failed to delete during the deletion task, each paired with an error
         * message.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationdiscovery.model.FailedConfiguration.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.applicationdiscovery.model.FailedConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationdiscovery.model.FailedConfiguration.Builder#build()} is
         * called immediately and its result is passed to {@link #failedConfigurations(List<FailedConfiguration>)}.
         * 
         * @param failedConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationdiscovery.model.FailedConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failedConfigurations(java.util.Collection<FailedConfiguration>)
         */
        Builder failedConfigurations(Consumer<FailedConfiguration.Builder>... failedConfigurations);

        /**
         * <p>
         * A list of configuration IDs that produced warnings regarding their deletion, paired with a warning message.
         * </p>
         * 
         * @param deletionWarnings
         *        A list of configuration IDs that produced warnings regarding their deletion, paired with a warning
         *        message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionWarnings(Collection<DeletionWarning> deletionWarnings);

        /**
         * <p>
         * A list of configuration IDs that produced warnings regarding their deletion, paired with a warning message.
         * </p>
         * 
         * @param deletionWarnings
         *        A list of configuration IDs that produced warnings regarding their deletion, paired with a warning
         *        message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionWarnings(DeletionWarning... deletionWarnings);

        /**
         * <p>
         * A list of configuration IDs that produced warnings regarding their deletion, paired with a warning message.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationdiscovery.model.DeletionWarning.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.applicationdiscovery.model.DeletionWarning#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationdiscovery.model.DeletionWarning.Builder#build()} is called
         * immediately and its result is passed to {@link #deletionWarnings(List<DeletionWarning>)}.
         * 
         * @param deletionWarnings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationdiscovery.model.DeletionWarning.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deletionWarnings(java.util.Collection<DeletionWarning>)
         */
        Builder deletionWarnings(Consumer<DeletionWarning.Builder>... deletionWarnings);
    }

    static final class BuilderImpl implements Builder {
        private String taskId;

        private String status;

        private Instant startTime;

        private Instant endTime;

        private String configurationType;

        private List<String> requestedConfigurations = DefaultSdkAutoConstructList.getInstance();

        private List<String> deletedConfigurations = DefaultSdkAutoConstructList.getInstance();

        private List<FailedConfiguration> failedConfigurations = DefaultSdkAutoConstructList.getInstance();

        private List<DeletionWarning> deletionWarnings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BatchDeleteConfigurationTask model) {
            taskId(model.taskId);
            status(model.status);
            startTime(model.startTime);
            endTime(model.endTime);
            configurationType(model.configurationType);
            requestedConfigurations(model.requestedConfigurations);
            deletedConfigurations(model.deletedConfigurations);
            failedConfigurations(model.failedConfigurations);
            deletionWarnings(model.deletionWarnings);
        }

        public final String getTaskId() {
            return taskId;
        }

        public final void setTaskId(String taskId) {
            this.taskId = taskId;
        }

        @Override
        public final Builder taskId(String taskId) {
            this.taskId = taskId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(BatchDeleteConfigurationTaskStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getConfigurationType() {
            return configurationType;
        }

        public final void setConfigurationType(String configurationType) {
            this.configurationType = configurationType;
        }

        @Override
        public final Builder configurationType(String configurationType) {
            this.configurationType = configurationType;
            return this;
        }

        @Override
        public final Builder configurationType(DeletionConfigurationItemType configurationType) {
            this.configurationType(configurationType == null ? null : configurationType.toString());
            return this;
        }

        public final Collection<String> getRequestedConfigurations() {
            if (requestedConfigurations instanceof SdkAutoConstructList) {
                return null;
            }
            return requestedConfigurations;
        }

        public final void setRequestedConfigurations(Collection<String> requestedConfigurations) {
            this.requestedConfigurations = ConfigurationIdListCopier.copy(requestedConfigurations);
        }

        @Override
        public final Builder requestedConfigurations(Collection<String> requestedConfigurations) {
            this.requestedConfigurations = ConfigurationIdListCopier.copy(requestedConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requestedConfigurations(String... requestedConfigurations) {
            requestedConfigurations(Arrays.asList(requestedConfigurations));
            return this;
        }

        public final Collection<String> getDeletedConfigurations() {
            if (deletedConfigurations instanceof SdkAutoConstructList) {
                return null;
            }
            return deletedConfigurations;
        }

        public final void setDeletedConfigurations(Collection<String> deletedConfigurations) {
            this.deletedConfigurations = ConfigurationIdListCopier.copy(deletedConfigurations);
        }

        @Override
        public final Builder deletedConfigurations(Collection<String> deletedConfigurations) {
            this.deletedConfigurations = ConfigurationIdListCopier.copy(deletedConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deletedConfigurations(String... deletedConfigurations) {
            deletedConfigurations(Arrays.asList(deletedConfigurations));
            return this;
        }

        public final List<FailedConfiguration.Builder> getFailedConfigurations() {
            List<FailedConfiguration.Builder> result = FailedConfigurationListCopier.copyToBuilder(this.failedConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFailedConfigurations(Collection<FailedConfiguration.BuilderImpl> failedConfigurations) {
            this.failedConfigurations = FailedConfigurationListCopier.copyFromBuilder(failedConfigurations);
        }

        @Override
        public final Builder failedConfigurations(Collection<FailedConfiguration> failedConfigurations) {
            this.failedConfigurations = FailedConfigurationListCopier.copy(failedConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failedConfigurations(FailedConfiguration... failedConfigurations) {
            failedConfigurations(Arrays.asList(failedConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failedConfigurations(Consumer<FailedConfiguration.Builder>... failedConfigurations) {
            failedConfigurations(Stream.of(failedConfigurations).map(c -> FailedConfiguration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<DeletionWarning.Builder> getDeletionWarnings() {
            List<DeletionWarning.Builder> result = DeletionWarningsListCopier.copyToBuilder(this.deletionWarnings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDeletionWarnings(Collection<DeletionWarning.BuilderImpl> deletionWarnings) {
            this.deletionWarnings = DeletionWarningsListCopier.copyFromBuilder(deletionWarnings);
        }

        @Override
        public final Builder deletionWarnings(Collection<DeletionWarning> deletionWarnings) {
            this.deletionWarnings = DeletionWarningsListCopier.copy(deletionWarnings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deletionWarnings(DeletionWarning... deletionWarnings) {
            deletionWarnings(Arrays.asList(deletionWarnings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deletionWarnings(Consumer<DeletionWarning.Builder>... deletionWarnings) {
            deletionWarnings(Stream.of(deletionWarnings).map(c -> DeletionWarning.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public BatchDeleteConfigurationTask build() {
            return new BatchDeleteConfigurationTask(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
