/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.rendering.block;

import java.util.List;
import java.util.Map;

import org.xwiki.rendering.listener.Listener;

/**
 * A special block that Macro Blocks generate when they are executed so that it's possible to reconstruct the initial
 * syntax even after Macros have been executed. For example this is important in a WYSWIYG editor where you want to show
 * the Macro's rendered result and also let users modify the macro content.
 * 
 * @version $Id$
 * @since 1.5M2
 */
public class MacroMarkerBlock extends AbstractBlock
{
    /**
     * The macro name that we are preserving.
     */
    private String id;

    /**
     * The macro content that we are preserving.
     */
    private String content;

    /**
     * The macro is located in a inline content (like paragraph, etc.).
     */
    private boolean isInline;

    /**
     * @param id the name of the macro
     * @param parameters the parameters of the macro
     * @param childBlocks the list of children blocks generated by the macro
     * @param isInline indicate if the macro is located in a inline content (like paragraph, etc.)
     */
    public MacroMarkerBlock(String id, Map<String, String> parameters, List<Block> childBlocks, boolean isInline)
    {
        this(id, parameters, null, childBlocks, isInline);
    }

    /**
     * @param id the name of the macro
     * @param parameters the parameters of the macro
     * @param content the content of the macro. Null if the macro does not have content
     * @param childBlocks the list of children blocks generated by the macro
     * @param isInline indicate if the macro is located in a inline content (like paragraph, etc.)
     */
    public MacroMarkerBlock(String id, Map<String, String> parameters, String content, List<Block> childBlocks,
        boolean isInline)
    {
        super(childBlocks, parameters);

        this.id = id;
        this.content = content;
        this.isInline = isInline;
    }

    /**
     * @return the macro name.
     * @deprecated since 2.4M1 use {@link #getId()} instead
     */
    @Deprecated
    public String getName()
    {
        return getId();
    }

    /**
     * @return the macro identifier.
     * @since 2.4M1
     */
    public String getId()
    {
        return this.id;
    }

    /**
     * @return the macro content.
     */
    public String getContent()
    {
        return this.content;
    }

    /**
     * @return if true the macro is located in a inline content (like paragraph, etc.).
     */
    public boolean isInline()
    {
        return this.isInline;
    }

    @Override
    public void before(Listener listener)
    {
        listener.beginMacroMarker(getId(), getParameters(), getContent(), isInline());
    }

    @Override
    public void after(Listener listener)
    {
        listener.endMacroMarker(getId(), getParameters(), getContent(), isInline());
    }
}
