package com.alibaba.weex.plugin.loader;

import android.content.Context;
import android.content.res.AssetManager;
import android.text.TextUtils;
import android.util.Log;

import com.alibaba.weex.plugin.loader.utils.WeexSDKUtil;
import com.taobao.weex.WXSDKEngine;
import com.taobao.weex.WXSDKManager;
import com.taobao.weex.common.WXException;
import com.taobao.weex.common.WXModule;
import com.taobao.weex.dom.WXDomObject;
import com.taobao.weex.ui.ComponentCreator;
import com.taobao.weex.ui.SimpleComponentHolder;
import com.taobao.weex.ui.component.WXComponent;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by moxun on 17/3/14.
 */

public class WeexPluginContainer {

    public static final String TAG = "WeexPluginContainer";

    public static void loadAll(Context context, ClassLoader classLoader) {
        try {
            loadAdapters(context, classLoader);
            loadDomObjects(context, classLoader);
            loadComponents(context, classLoader);
            loadModules(context, classLoader);
        } catch (Throwable throwable) {
            Log.w(TAG, "Unexpected throwable in loadAll", throwable);
        }
    }

    public static void loadAll(final Context context) {
        loadAll(context, null);
    }

    public static void loadModules(final Context context) {
        loadModules(context, null);
    }

    public static void loadModules(final Context context, final ClassLoader classLoader) {
        try {
            for (JSONObject spec : getPluginSpecs(context, "weex_plugin/module")) {

                String moduleName = spec.optString("name", null);
                String clazz = spec.optString("class", null);
                boolean canOverride = spec.optBoolean("canOverrideExisting", true);
                boolean globalRegistration = spec.optBoolean("globalRegistration", false);
                boolean lazyLoad = spec.optBoolean("lazyLoad", false);
                JSONArray functions = null;
                if (lazyLoad) {
                    functions = spec.optJSONArray("functions");
                }

                if (moduleName == null || clazz == null) {
                    Log.w(TAG, "Module name or module class is null");
                    continue;
                }

                if (!canOverride && WeexSDKUtil.containsModule(moduleName, globalRegistration)) {
                    Log.w(TAG, "Ignore duplicated module: " + moduleName);
                    continue;
                }

                try {
                    Class<? extends WXModule> moduleClass = (Class<? extends WXModule>)loadClass(clazz, classLoader);
                    if (moduleClass != null) {
                        if (!WXModule.class.isAssignableFrom(moduleClass)) {
                            Log.e(TAG, "Class " + clazz + " is not a subclass of WXModule");
                            continue;
                        }
                        WXSDKEngine.registerModule(moduleName, moduleClass, globalRegistration);
                        Log.d(TAG, "Module plugin ["+moduleName+"][clazz:"+moduleClass+"][globalRegistration:"+globalRegistration+"][lazyLoad:"+lazyLoad+"][canOverride:"+canOverride+"]");
                    }
//                    WXSDKEngine.registerModule(moduleName, moduleClass);
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                } catch (WXException e) {
                    e.printStackTrace();
                }
            }
        } catch (Throwable throwable) {
            Log.w(TAG, "Unexpected throwable in loadModules", throwable);
        }
    }

    public static void loadComponents(final Context context) {
        loadModules(context, null);
    }

    public static void loadComponents(final Context context, final ClassLoader classLoader) {
        try {

            for (JSONObject spec : getPluginSpecs(context, "weex_plugin/component")) {

                List<String> names = new ArrayList<String>();
                JSONArray namesTemp = spec.optJSONArray("names");
                for (int i = 0; i < namesTemp.length(); i++) {
                    String name = namesTemp.optString(i);
                    if (!TextUtils.isEmpty(name)) {
                        names.add(name);
                    }
                }

                String[] namesArray = names.toArray(new String[names.size()]);

                String clazz = spec.optString("class");
                boolean appendTree = spec.optBoolean("appendTree", false);
                boolean usingHolder = spec.optBoolean("usingHolder", false);
                boolean canOverride = spec.optBoolean("canOverrideExisting", true);

                String creator = null;
                if (usingHolder) {
                    creator = spec.optString("creator", null);
                }
                if (names.size() == 0 || TextUtils.isEmpty(clazz)) {
                    Log.w(TAG, "Component names or component class is null");
                    continue;
                }

                if (!canOverride && WeexSDKUtil.containsComponent(names)) {
                    Log.w(TAG, "Ignore duplicated component: " + names.toString());
                    continue;
                }

                try {
                    Class<? extends WXComponent<?>> componentClass = (Class<? extends WXComponent<?>>)loadClass(clazz, classLoader);
                    if (componentClass != null) {
                        if (!WXComponent.class.isAssignableFrom(componentClass)) {
                            Log.e(TAG, "Class " + clazz + " is not a subclass of WXComponent");
                            continue;
                        }

                        if (usingHolder) {
                            if (creator == null || !"NULL".equals(creator)) {
                                WXSDKEngine.registerComponent(new SimpleComponentHolder(componentClass), appendTree, namesArray);
                            } else {
                                Class creatorClass = loadClass(creator, classLoader);
                                if (creatorClass == null) {
                                    WXSDKEngine.registerComponent(new SimpleComponentHolder(componentClass), appendTree, namesArray);
                                } else {
                                    ComponentCreator componentCreator = (ComponentCreator) creatorClass.newInstance();
                                    WXSDKEngine.registerComponent(new SimpleComponentHolder(componentClass, componentCreator), appendTree, namesArray);
                                }
                            }
                        } else {
                            WXSDKEngine.registerComponent(componentClass, appendTree, namesArray);
                        }

//                        WXSDKEngine.registerComponent(namesArray[0], componentClass);

                        Log.d(TAG, "Component plugin ["+names+"][clazz:"+componentClass+"][appendTree:"+appendTree+"][usingHolder:"+usingHolder+"][canOverride:"+canOverride+"]["+creator+"]");
                    }
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                } catch (WXException e) {
                    e.printStackTrace();
                } catch (InstantiationException e) {
                    e.printStackTrace();
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        } catch (Throwable throwable) {
            Log.w(TAG, "Unexpected throwable in loadComponents", throwable);
        }
    }

    public static void loadAdapters(final Context context) {
        loadAdapters(context, null);
    }

    public static void loadAdapters(final Context context, final ClassLoader classLoader) {
        try {
            for (JSONObject spec : getPluginSpecs(context, "weex_plugin/adapter")) {
                String type = spec.optString("type");
                String clazz = spec.optString("class");
                boolean canOverride = spec.optBoolean("canOverrideExisting", true);

                if (!WeexSDKUtil.isValidAdapterType(type)) {
                    Log.e(TAG, "Invalid adapter type: " + type);
                    continue;
                }

                if (!canOverride && WeexSDKUtil.containsAdapter(type)) {
                    Log.w(TAG, "Ignore duplicated adapter: " + clazz);
                    continue;
                }

                try {
                    Class adapterClass = loadClass(clazz, classLoader);
                    if (adapterClass != null) {
                        try {
                            Object adapter = adapterClass.newInstance();
                            String fieldName = WeexSDKUtil.getAdapterField(type);

                            Field field = WXSDKManager.class.getDeclaredField(fieldName);
                            field.setAccessible(true);
                            field.set(WXSDKManager.getInstance(), adapter);

                            Log.d(TAG, "Adapter plugin " + clazz + " registered.");
                        } catch (InstantiationException e) {
                            e.printStackTrace();
                        } catch (IllegalAccessException e) {
                            e.printStackTrace();
                        } catch (NoSuchFieldException e) {
                            e.printStackTrace();
                        }
                    }
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                }
            }
        } catch (Throwable throwable) {
            Log.w(TAG, "Unexpected throwable in loadAdapters", throwable);
        }
    }

    public static void loadDomObjects(final Context context) {
        loadDomObjects(context, null);
    }

    public static void loadDomObjects(final Context context, final ClassLoader classLoader) {
        try {
            for (JSONObject spec : getPluginSpecs(context, "weex_plugin/domObject")) {
                String clazz = spec.optString("class");
                String type = spec.optString("type");
                boolean canOverride = spec.optBoolean("canOverrideExisting", false);
                //ignore canOverride temporary

                if (!TextUtils.isEmpty(clazz)) {
                    try {
                        Class<? extends WXDomObject> domClass = (Class<? extends WXDomObject>)loadClass(clazz, classLoader);
                        if (domClass != null) {
                            if (!WXDomObject.class.isAssignableFrom(domClass)) {
                                Log.e(TAG, "Class " + clazz + " is not a subclass of WXDomObject");
                                continue;
                            }

                            if (WXDomObject.class.isAssignableFrom(domClass)) {
                                WXSDKEngine.registerDomObject(type, domClass);
                                Log.d(TAG, "Dom object plugin " + type + " registered.");
                            }
                        }
                    } catch (ClassNotFoundException e) {
                        e.printStackTrace();
                    } catch (WXException e) {
                        e.printStackTrace();
                    }
                }
            }
        } catch (Throwable throwable) {
            Log.w(TAG, "Unexpected throwable in loadDomObjects", throwable);
        }
    }

    private static Class<?> loadClass(String className, ClassLoader classLoader) throws ClassNotFoundException {
        if (classLoader == null) {
            classLoader = DEFAULT_CLASS_LOADER;
        }
        return classLoader.loadClass(className);
    }
    private final static ClassLoader DEFAULT_CLASS_LOADER = WeexPluginContainer.class.getClassLoader();
    private static List<JSONObject> getPluginSpecs(Context context, String pathOnAssets) {
        AssetManager assetManager = context.getAssets();
        List<JSONObject> specs = new ArrayList<JSONObject>();
        try {
            String[] files = assetManager.list(pathOnAssets);
            if (files != null && files.length > 0) {
                for (String fileName : files) {
                    if (fileName.endsWith(".json")) {
                        Log.d(TAG, "Parse plugin meta info >>> " + fileName);
                        InputStream inputStream = assetManager.open(pathOnAssets + "/" + fileName);
                        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
                        StringBuffer stringBuffer = new StringBuffer();
                        String line;
                        while ((line = reader.readLine()) != null) {
                            stringBuffer.append(line);
                        }
                        JSONArray spec = new JSONArray(stringBuffer.toString());
                        for (int i = 0; i < spec.length(); i++) {
                            JSONObject element = spec.optJSONObject(i);
                            if (element != null) {
                                specs.add(element);
                            }
                        }
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return specs;
    }
}
