/**
@license
Copyright (c) 2015 The Polymer Project Authors. All rights reserved.
This code may only be used under the BSD style license found at
http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
part of the polymer project is also subject to an additional IP rights grant
found at http://polymer.github.io/PATENTS.txt
*/
import '@polymer/app-route/app-location.js';
import '@polymer/app-route/app-route.js';
import '@polymer/iron-flex-layout/iron-flex-layout.js';
import '@polymer/iron-media-query/iron-media-query.js';
import '@polymer/paper-icon-button/paper-icon-button.js';
import '@polymer/paper-tabs/paper-tabs.js';
import '@polymer/iron-icons/iron-icons.js';
import '@polymer/iron-pages/iron-pages.js';
import '@polymer/app-layout/app-layout.js';
import '@polymer/app-layout/app-scroll-effects/app-scroll-effects.js';
import './shrine-list.js';
import './shrine-detail.js';

import {scroll} from '@polymer/app-layout/helpers/helpers.js';
import {Polymer} from '@polymer/polymer/lib/legacy/polymer-fn.js';
import {html} from '@polymer/polymer/lib/utils/html-tag.js';

Polymer({
  _template: html`
    <style>

      :host {
        -webkit-tap-highlight-color: rgba(0, 0, 0, 0);
      }

      app-header {
        background-color: rgba(255, 255, 255, 0.95);
        --app-header-shadow: {
          box-shadow: inset 0px 5px 6px -3px rgba(0, 0, 0, 0.2);
          height: 10px;
          bottom: -10px;
        };
      }

      app-header:not([shadow]) {
        border-bottom: 1px solid #ddd;
      }

      app-drawer {
        --app-drawer-content-container: {
          background-color: #f3f3f3;
        };
      }

      #mainToolbar {
        height: 110px;
      }

      paper-icon-button {
        color: #000;
        --paper-icon-button-ink-color: #31f0ef;
      }

      .navItem {
        @apply --layout-self-start;
        margin-top: 13px;
        width: 88px;
      }

      .spacer {
        @apply --layout;
        @apply --layout-flex-auto;
        @apply --layout-center-center;
      }

      .logo {
        width: 120px;
        height: 40px;
        background-size: 100% 100%;
      }

      .leftItem [icon] {
        display: none;
      }


      shrine-detail {
        height: calc(100vh - 65px);
      }

      :host([page=detail]) #mainToolbar {
        height: 64px;
      }

      :host([page=detail]) paper-tabs {
        display: none;
      }

      :host([page=detail]) .leftItem [icon=arrow-back] {
        display: block;
      }

      :host([page=detail]) .leftItem [icon=menu] {
        display: none;
      }

      paper-tabs {
        --paper-tabs-selection-bar-color: #31f0ef;
        margin-left: -55px;
        color: #31f0ef;
        font-size: 13px;
      }

      paper-tab {
        @apply --layout-flex-none;
        padding: 0;
        --paper-tab-ink: #31f0ef;
      }

      paper-tab a {
        @apply --layout-horizontal;
        @apply --layout-center-center;
        text-decoration: none;
        text-transform: uppercase;
        color: #7c7c7c;
        font-weight: 500;
        padding: 0 20px;
        height: 100%;
      }

      .list {
        margin-left: 20px;
        margin-right: 20px;
      }

      .list a {
        display: block;
        line-height: 40px;
        text-decoration: none;
        text-transform: uppercase;
        color: #7c7c7c;
      }

      .list a.active {
        color: black;
        font-weight: bold;
      }

      .profileBar {
        margin-bottom: 40px;
      }

      .profilePic {
        border-radius: 50% 50%;
      }

      .profileName {
        @apply --layout-flex;
        margin-left: 20px;
        font-size: 14px;
        font-weight: 500;
        color: #0a3142;
      }

      [icon=settings] {
        width: 35px;
        height: 35px;
        color: #0a3142;
      }

      /**
       * Desktop small, tablet
       */
      @media (max-width: 1200px) {
        .logo {
          width: 89px;
          height: 17px;
        }

        #mainToolbar {
          height: 64px;
        }

        shrine-detail {
          height: auto;
        }
      }

      /**
       * Phone
       */
      @media (max-width: 400px) {
        .leftItem [icon=menu] {
          display: block;
        }

        paper-tabs {
          display: none;
        }
      }

    </style>

    <app-location route="{{route}}" use-hash-as-path></app-location>
    <app-route route="{{route}}" pattern=":section" data="{{sectionData}}" tail="{{subRoute}}"></app-route>
    <app-route route="{{subRoute}}" pattern="/:id" data="{{idData}}" active="{{onDetailPage}}"></app-route>

    <iron-media-query query="max-width: 400px" query-matches="{{smallScreen}}"></iron-media-query>

    <app-drawer-layout drawer-width="288px" force-narrow>
      <!-- navigation drawer for small screen sizes -->
      <app-drawer slot="drawer" id="drawer" swipe-open="[[smallScreen]]">
        <template is="dom-if" if="[[smallScreen]]">
          <app-toolbar class="profileBar">
            <img class="profilePic" src="//app-layout-assets.appspot.com/assets/shrine/profile_pic.jpg" width="30" height="30">
            <div class="profileName">Stella</div>
            <paper-icon-button icon="settings" aria-label="Settings"></paper-icon-button>
          </app-toolbar>
          <div class="list">
            <template is="dom-repeat" items="[[sections]]" as="section" initial-count="1">
              <a href="#[[section]]" class\$="[[_getSectionClass(index, selectedTab)]]">[[section]]</a>
            </template>
          </div>
        </template>
      </app-drawer>

      <app-header-layout id="headerLayout">
        <!-- main header -->
        <app-header slot="header" effects="waterfall" fixed="[[smallScreen]]" condenses="[[!smallScreen]]" reveals="[[!smallScreen]]">
          <app-toolbar id="mainToolbar">
            <div class="navItem leftItem">
              <paper-icon-button icon="menu" drawer-toggle alt="Toogle navigation menu"></paper-icon-button>
              <a href="#[[sectionData.section]]" tabindex="-1"><paper-icon-button icon="arrow-back" alt="Back to the home"></paper-icon-button></a>
            </div>
            <div class="spacer">
              <img src="//app-layout-assets.appspot.com/assets/shrine/shrine_logo.svg" class="logo" alt="SHRINE">
            </div>
            <div class="navItem">
              <paper-icon-button icon="shopping-cart" aria-label="Shopping cart"></paper-icon-button>
              <paper-icon-button icon="more-vert" aria-label="More options"></paper-icon-button>
            </div>
          </app-toolbar>
          <template is="dom-if" if="[[_shouldShowTabs(onDetailPage, smallScreen)]]">
            <paper-tabs selected="[[selectedTab]]" scrollable sticky\$="[[_shouldShowTabs(onDetailPage, smallScreen)]]">
              <template is="dom-repeat" items="[[sections]]" as="section">
                <paper-tab>
                  <a href="#[[section]]">[[section]]</a>
                </paper-tab>
              </template>
            </paper-tabs>
          </template>
        </app-header>

        <iron-pages selected="[[page]]" attr-for-selected="name">
          <!-- detail of item -->
          <template is="dom-if" if="[[_equal(page, 'detail')]]">
            <shrine-detail name="detail" section="[[sectionData.section]]" related-items="[[items]]" item="[[_getDetailItem(items, idData.id)]]">
            </shrine-detail>
          </template>
          <!-- list of items -->
          <template is="dom-if" if="[[_equal(page, 'list')]]">
            <shrine-list name="list" section="[[sectionData.section]]" items="[[_getItemsCopy(items, sectionData.section)]]" featured-item="[[_getFeaturedItem(featuredItems, sectionData.section)]]">
            </shrine-list>
          </template>
        </iron-pages>

      </app-header-layout>
    </app-drawer-layout>
`,

  is: 'shrine-app',

  properties: {

    sections: {
      type: Array,
      value: function() {
        return [
          'feature',
          'latest',
          'fashion',
          'furniture',
          'beauty',
          'food',
          'travel'
        ];
      }
    },

    selectedTab: {
      type: Number,
      computed: '_computeSelectedTab(sections, sectionData.section)'
    },

    items: {type: Array},

    featuredItems: {type: Array},

    page: {
      type: String,
      computed: '_computePage(onDetailPage)',
      reflectToAttribute: true
    },

    route: Object,

    subRoute: Object,

    sectionData: Object,

    idData: Object,

    onDetailPage: Boolean
  },

  observers: ['_hashDidChange(route.path, items, featuredItems)'],

  attached: function() {
    this.async(function() {
      if (!this.route.path) {
        this.set('route.path', 'feature');
      }
    });
  },

  _computeSelectedTab: function(sections, section) {
    return sections.indexOf(section);
  },

  _getItemsCopy: function(items) {
    return items ? items.slice() : [];
  },

  _getFeaturedItem: function(featuredItems, section) {
    if (featuredItems && section) {
      return featuredItems
          .filter(function(item) {
            return item.category.toLowerCase() === section;
          })
          .pop();
    }
    return '';
  },

  _getDetailItem: function(items, id) {
    if (items) {
      return items[id];
    }
  },

  _computePage: function(onDetailPage) {
    return onDetailPage ? 'detail' : 'list';
  },

  _hashDidChange: function() {
    scroll({top: 0, behavior: 'silent'});
    this.$.headerLayout.resetLayout();
    this.$.drawer.close();
  },

  _equal: function(a, b) {
    return a === b;
  },

  _getSectionClass: function(index, selectedTab) {
    return index === selectedTab ? 'active' : '';
  },

  _shouldShowTabs: function(onDetailPage, smallScreen) {
    return !onDetailPage && !smallScreen;
  }
});
