/*
 * All content copyright (c) Terracotta, Inc., except as may otherwise be noted in a separate copyright notice. All
 * rights reserved.
 */
package org.terracotta.cache.serialization;

import org.terracotta.cache.CacheConfigFactory;
import org.terracotta.cache.impl.TimeSource;
import org.terracotta.cache.value.AbstractTimestampedValue;

import com.tc.object.bytecode.TransparentAccess;

import java.io.IOException;
import java.lang.reflect.Field;

/**
 * Wrapper class around the serialized cache entries. This handles the serialization and de-serialization of the cache
 * entries, and also the local caching (in a transient reference) of the de-serialized value.
 */
public class SerializedEntry<T> extends AbstractTimestampedValue<byte[]> {

  /**
   * <pre>
   * ********************************************************************************************
   * IF YOU'RE CHANGING ANYTHING ABOUT THE FIELDS IN THIS CLASS (name, type, add or remove, etc)
   * YOU MUST UPDATE BOTH THE APPLICATOR AND SERVER STATE CLASSES ACCORDINGLY!
   * ********************************************************************************************
   * </pre>
   */
  private final byte[] value;
  private final int    createTime;

  private transient T  cached;

  public SerializedEntry(final T deserialized, final byte[] serialized, final TimeSource timeSource) {
    this(deserialized, serialized, timeSource.now());
  }

  public SerializedEntry(final T deserialized, final byte[] serialized, final int createTime) {
    this(deserialized, serialized, createTime, createTime);
  }

  public SerializedEntry(final T deserialized, final byte[] serialized, final int createTime, final int lastAccessedTime) {
    this.value = serialized;
    this.createTime = createTime;
    this.setLastAccessedTimeInternal(lastAccessedTime);
    this.cached = deserialized;
  }

  public T getDeserializedValueCopy(final SerializationStrategy<T> strategy) throws IOException, ClassNotFoundException {
    return strategy.deserialize(getValue());
  }

  public T getDeserializedValueCopy(final SerializationStrategy<T> strategy, ClassLoader classLoader)
      throws IOException, ClassNotFoundException {
    return strategy.deserialize(getValue(), classLoader);
  }

  public synchronized T getDeserializedValue(final SerializationStrategy<T> strategy) throws IOException,
      ClassNotFoundException {
    if (cached == null) {
      cached = strategy.deserialize(getValue());
      nullByteArray();
    }

    return cached;
  }

  public synchronized T getDeserializedValue(final SerializationStrategy<T> strategy, ClassLoader classLoader)
      throws IOException, ClassNotFoundException {
    if (cached == null) {
      cached = strategy.deserialize(getValue(), classLoader);
      nullByteArray();
    }

    return cached;
  }

  public void nullByteArray() {
    if (!CacheConfigFactory.DSO_ACTIVE) return;
    // This bit of trickery will simply null the reference to the byte[] but without telling terracotta about it. Any
    // attempts to read the reference will just give back null (ie. it will not be resolved/faulted ever again)
    ((TransparentAccess) this).__tc_setfield(VALUE_FIELD, null);

    if (this.value != null) { throw new AssertionError(); }
  }

  @Override
  public int getCreateTime() {
    return this.createTime;
  }

  @Override
  public byte[] getValue() {
    return this.value;
  }

  private static final String VALUE_FIELD;

  static {
    // make sure there is a field named "value" since nullByteArray() depends on that name
    Class c = SerializedEntry.class;

    try {
      Field valueField = c.getDeclaredField("value");
      VALUE_FIELD = c.getName() + "." + valueField.getName();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

}
