/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.impl;

import org.terracotta.cache.CacheConfig;
import org.terracotta.cache.CacheConfigFactory;
import org.terracotta.cache.DistributedCache;
import org.terracotta.cache.evictor.CapacityEvictionPolicyData;
import org.terracotta.cache.evictor.LFUCapacityEvictionPolicyData;
import org.terracotta.cache.evictor.CapacityEvictionPolicyData.Factory;
import org.terracotta.cache.logging.ConfigChangeListener;
import org.terracotta.cache.logging.TCLoggerConfigChangeListener;

import java.util.ArrayList;
import java.util.List;

/**
 * Initialized with default configuration values. The default values are:
 * <ul>
 * <li>maxTTISeconds = 0 (off)</li>
 * <li>maxTTLSeconds = 0 (off)</li>
 * <li>orphanEvictionEnabled = true</li>
 * <li>orphanEvictionPeriod = 4</li>
 * <li>loggingEnabled = false</li>
 * <li>targetMaxInMemoryCount = 0</li>
 * <li>targetMaxTotalCount = 0</li>
 * </ul>
 */
public class MutableConfig implements CacheConfig {
  private static final boolean                 DSO_ACTIVE                        = Boolean.getBoolean("tc.active");

  //
  // Basic
  //
  // name of the cache instance
  private String                               name                              = "Distributed Cache";

  //
  // Expiration
  //
  // max Time To Live till expire
  private int                                  maxTTISeconds                     = 0;

  // max Time To Idle till expire
  private int                                  maxTTLSeconds                     = 0;

  //
  // Orphan eviction
  //
  // detect and evict orphaned elements
  private boolean                              orphanEvictionEnabled             = true;

  // # of times to run local eviction before doing orphan
  private int                                  orphanEvictionPeriod              = 4;

  //
  // Logging
  //
  // Basic cache logging
  private boolean                              loggingEnabled                    = false;

  private int                                  targetMaxInMemoryCount            = 0;
  private int                                  targetMaxTotalCount               = 0;

  private transient List<ConfigChangeListener> configListeners                   = new ArrayList<ConfigChangeListener>();

  private CapacityEvictionPolicyData.Factory   capacityEvictionPolicyDataFactory = new LFUCapacityEvictionPolicyData.Factory();

  private volatile transient CacheConfig       snapshot;

  public void initializeOnLoad() {
    configListeners = new ArrayList<ConfigChangeListener>();
    refresh();
    registerTCLoggerIfActive();
  }

  public MutableConfig() {
    refresh();
    registerTCLoggerIfActive();
  }

  public MutableConfig(final String name, final boolean loggingEnabled, final int maxTTISeconds,
                       final int maxTTLSeconds, final boolean orphanEvictionEnabled, final int orphanEvictionPeriod,
                       final int targetMaxInMemoryCount, final int targetMaxTotalCount,
                       final CapacityEvictionPolicyData.Factory capacityEvictionPolicyDataFactory) {
    this.name = name;
    this.loggingEnabled = loggingEnabled;
    this.maxTTISeconds = maxTTISeconds;
    this.maxTTLSeconds = maxTTLSeconds;
    this.orphanEvictionEnabled = orphanEvictionEnabled;
    this.orphanEvictionPeriod = orphanEvictionPeriod;
    this.targetMaxInMemoryCount = targetMaxInMemoryCount;
    this.targetMaxTotalCount = targetMaxTotalCount;
    this.capacityEvictionPolicyDataFactory = capacityEvictionPolicyDataFactory;
    refresh();
    registerTCLoggerIfActive();
  }

  private void registerTCLoggerIfActive() {
    if (DSO_ACTIVE) {
      addConfigChangeListener(new TCLoggerConfigChangeListener());
    }
  }

  public String getName() {
    return snapshot.getName();
  }

  public int getMaxTTISeconds() {
    return snapshot.getMaxTTISeconds();
  }

  public int getMaxTTLSeconds() {
    return snapshot.getMaxTTLSeconds();
  }

  public boolean isOrphanEvictionEnabled() {
    return snapshot.isOrphanEvictionEnabled();
  }

  public int getOrphanEvictionPeriod() {
    return snapshot.getOrphanEvictionPeriod();
  }

  public boolean isLoggingEnabled() {
    return snapshot.isLoggingEnabled();
  }

  public int getTargetMaxInMemoryCount() {
    return snapshot.getTargetMaxInMemoryCount();
  }

  public int getTargetMaxTotalCount() {
    return snapshot.getTargetMaxTotalCount();
  }

  public Factory getCapacityEvictionPolicyDataFactory() {
    return snapshot.getCapacityEvictionPolicyDataFactory();
  }

  public MutableConfig setMaxTTISeconds(int maxTTISeconds) {
    if (maxTTISeconds < 0) throw new IllegalArgumentException("Max TTI must be >= 0");

    // write <autolock>'d
    synchronized (this) {
      this.maxTTISeconds = maxTTISeconds;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setMaxTTLSeconds(int maxTTLSeconds) {
    if (maxTTLSeconds < 0) throw new IllegalArgumentException("Max TTL must be >= 0");

    // write <autolock>'d
    synchronized (this) {
      this.maxTTLSeconds = maxTTLSeconds;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setOrphanEvictionEnabled(boolean orphanEvictionEnabled) {

    // write <autolock>'d
    synchronized (this) {
      this.orphanEvictionEnabled = orphanEvictionEnabled;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setOrphanEvictionPeriod(int orphanEvictionPeriod) {

    // write <autolock>'d
    synchronized (this) {
      this.orphanEvictionPeriod = orphanEvictionPeriod;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setLoggingEnabled(boolean loggingEnabled) {

    // write <autolock>'d
    synchronized (this) {
      this.loggingEnabled = loggingEnabled;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setTargetMaxInMemoryCount(int targetMaxInMemoryCount) {

    // write <autolock>'d
    synchronized (this) {
      this.targetMaxInMemoryCount = targetMaxInMemoryCount;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setTargetMaxTotalCount(int targetMaxTotalCount) {

    // write <autolock>'d
    synchronized (this) {
      this.targetMaxTotalCount = targetMaxTotalCount;
      notifyAll();
    }
    refresh();

    return this;
  }

  public MutableConfig setName(String name) {
    if (name == null) throw new NullPointerException("Name cannot be null");

    // write <autolock>'d
    synchronized (this) {
      this.name = name;
      notifyAll();
    }
    refresh();

    return this;
  }

  public CacheConfig setCapacityEvictionPolicyDataFactory(Factory factory) {
    if (factory == null) throw new NullPointerException("capacityEvictionPolicyDataFactory cannot be null");

    // write <autolock>'d
    synchronized (this) {
      this.capacityEvictionPolicyDataFactory = factory;
      notifyAll();
    }
    refresh();
    return this;
  }

  public void waitForChange(long maxWait) {
    // write <autolock>'d
    synchronized (this) {
      if (refresh()) {
        return;
      } else {
        try {
          wait(maxWait);
        } catch (InterruptedException e) {
          // no problem
        } finally {
          refresh();
        }
      }
    }
  }

  public boolean refresh() {
    // read <autolock>'d
    synchronized (this) {
      CacheConfig oldSnapshot = snapshot;
      // don't use the ImmutableConfig(Config) constructor; it would read the snapshot...
      snapshot = new ImmutableConfig(name, loggingEnabled, maxTTISeconds, maxTTLSeconds, orphanEvictionEnabled,
                                     orphanEvictionPeriod, targetMaxInMemoryCount, targetMaxTotalCount,
                                     capacityEvictionPolicyDataFactory);

      if (oldSnapshot != null) {
        fireChangeEventIfNeeded("name", oldSnapshot.getName(), name);
        fireChangeEventIfNeeded("loggingEnabled", oldSnapshot.isLoggingEnabled(), loggingEnabled);
        fireChangeEventIfNeeded("maxTTISeconds", oldSnapshot.getMaxTTISeconds(), maxTTISeconds);
        fireChangeEventIfNeeded("maxTTLSeconds", oldSnapshot.getMaxTTLSeconds(), maxTTLSeconds);
        fireChangeEventIfNeeded("orphanEvictionEnabled", oldSnapshot.isOrphanEvictionEnabled(), orphanEvictionEnabled);
        fireChangeEventIfNeeded("orphanEvictionPeriod", oldSnapshot.getOrphanEvictionPeriod(), orphanEvictionPeriod);
        fireChangeEventIfNeeded("targetMaxInMemoryCount", oldSnapshot.getTargetMaxInMemoryCount(),
                                targetMaxInMemoryCount);
        fireChangeEventIfNeeded("targetMaxTotalCount", oldSnapshot.getTargetMaxTotalCount(), targetMaxTotalCount);
        fireChangeEventIfNeeded("capacityEvictionPolicyDataFactory",
                                oldSnapshot.getCapacityEvictionPolicyDataFactory(), capacityEvictionPolicyDataFactory);
      }

      return !snapshot.equals(oldSnapshot);
    }
  }

  /**
   * Construct a new cache instance based on the configuration changes specified in this instance of the builder so far.
   * If Terracotta is not being used, a non-clustered implementation will be returned instead.
   * 
   * @param <K> Key type
   * @param <V> Value type
   * @return Map constructed by this method based on previous parameters and defaults
   */
  public <K, V> DistributedCache<K, V> newCache() {
    MutableConfig copyConfig = new MutableConfig(name, loggingEnabled, maxTTISeconds, maxTTLSeconds,
                                                 orphanEvictionEnabled, orphanEvictionPeriod, targetMaxInMemoryCount,
                                                 targetMaxTotalCount, capacityEvictionPolicyDataFactory);

    if (CacheConfigFactory.DSO_ACTIVE) {
      return new DistributedCacheImpl<K, V>(copyConfig);
    } else {
      return new LocalCache<K, V>(copyConfig);
    }
  }

  @Override
  public boolean equals(Object o) {
    if (!(o instanceof MutableConfig)) { return false; }

    MutableConfig config = (MutableConfig) o;

    return getName().equals(config.getName()) && (getMaxTTISeconds() == config.getMaxTTISeconds())
           && (getMaxTTLSeconds() == config.getMaxTTLSeconds())
           && (isOrphanEvictionEnabled() == config.isOrphanEvictionEnabled())
           && (getOrphanEvictionPeriod() == config.getOrphanEvictionPeriod())
           && (isLoggingEnabled() == config.isLoggingEnabled())
           && (getTargetMaxInMemoryCount() == config.getTargetMaxInMemoryCount())
           && (getTargetMaxTotalCount() == config.getTargetMaxTotalCount());
  }

  @Override
  public int hashCode() {
    return this.getName().hashCode();
  }

  private void fireConfigChangeEvent(String configName, Object oldValue, Object newValue) {
    for (ConfigChangeListener ccl : configListeners.toArray(new ConfigChangeListener[0])) {
      ccl.configChanged(name, configName, oldValue, newValue);
    }
  }

  private void fireChangeEventIfNeeded(String configName, Object oldValue, Object newValue) {
    if (oldValue != null && !oldValue.equals(newValue)) {
      fireConfigChangeEvent(configName, oldValue, newValue);
    }
  }

  /**
   * Register listener for changes in the config
   */
  public synchronized void addConfigChangeListener(ConfigChangeListener ccl) {
    configListeners.add(ccl);
  }

  /**
   * Remove a config listener
   */
  public synchronized void removeConfigChangeListener(ConfigChangeListener ccl) {
    configListeners.remove(ccl);
  }
}
