/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.evictor;

import org.terracotta.cache.CacheConfig;
import org.terracotta.collections.ConcurrentDistributedMap;
import org.terracotta.collections.MapSizeListener;

/**
 * @author abhi.sanoujam
 */
public class TargetCapacityMapSizeListener<K> implements MapSizeListener {
  /**
   * Fraction by which we allow the cache to exceed capacity (local or total) before we start becoming more aggressive
   * on eviction.
   */
  private static final float             OVERSHOOT_RATIO    = 0.01f;
  /**
   * Minimum value for OVERSHOOT_RATIO * overshoot
   */
  private static final int               OVERSHOOT_MINIMUM  = 10;
  /**
   * Number of entries to evict for each new entry added when being aggressive.
   */
  private static final int               MAX_EVICTION_RATIO = 2;

  private final ConcurrentDistributedMap map;
  private final CacheConfig              config;
  private final TargetCapacityEvictor    targetCapacityEvictor;

  private volatile int                   localEvictionRatio = 1;
  private volatile int                   totalEvictionRatio = 1;

  public TargetCapacityMapSizeListener(ConcurrentDistributedMap map, CacheConfig config) {
    this.map = map;
    this.config = config;
    this.targetCapacityEvictor = new TargetCapacityEvictor(map);
  }

  public void localSizeChanged(int delta) {
    if ((delta <= 0) || (config.getTargetMaxInMemoryCount() <= 0)) { return; }

    int overshoot = map.localSize() - config.getTargetMaxInMemoryCount();

    if (overshoot <= 0) {
      localEvictionRatio = 1;
      return;
    } else if (overshoot > Math.max(OVERSHOOT_MINIMUM, OVERSHOOT_RATIO * config.getTargetMaxInMemoryCount())) {
      localEvictionRatio = MAX_EVICTION_RATIO;
    }

    int maxEvict = Math.min(delta * localEvictionRatio, overshoot);
    targetCapacityEvictor.evictLocalElements(maxEvict);
  }

  public void sizeChanged(int delta) {
    if ((delta <= 0) || (config.getTargetMaxTotalCount() <= 0)) { return; }

    int overshoot = map.size() - config.getTargetMaxTotalCount();

    if (overshoot <= 0) {
      totalEvictionRatio = 1;
      return;
    } else if (overshoot > Math.max(OVERSHOOT_MINIMUM, OVERSHOOT_RATIO * config.getTargetMaxTotalCount())) {
      totalEvictionRatio = MAX_EVICTION_RATIO;
    }

    int maxEvict = Math.min(delta * totalEvictionRatio, overshoot);
    targetCapacityEvictor.evictOrphanElements(maxEvict);
  }

}
