/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.evictor;

/**
 * <p>
 * The evictor watches the {@link Evictable} and is run in a loop, periodically running an eviction of all keys existent
 * locally in this node.
 * </p>
 */
public class Evictor<K> {

  // The evictable store
  private final Evictable<K>     store;

  // Eviction lifecycle listener
  private final EvictionListener evictionListener;

  /**
   * Construct evictor with the evictable store to evict on. No EvictionListener is specified so the
   * {@link NullEvictionListener} is used - this listener does not do orphan eviction.
   * 
   * @param store The evictable collection
   */
  public Evictor(Evictable<K> store) {
    this(store, new NullEvictionListener());
  }

  /**
   * Construct evictor with the evictable store to evict on. Allows specification of an EvictionListener that can handle
   * orphan eviction or potentially other tasks.
   * 
   * @param store The evictable store
   * @param evictionListener The eviction listener
   */
  public Evictor(Evictable<K> store, EvictionListener evictionListener) {
    this.store = store;
    this.evictionListener = evictionListener;
  }

  /**
   * Run evictor run!
   * 
   * @throws RuntimeException if an error occurs
   */
  public void run() {
    evictionListener.startLocalEviction();
    store.evictExpiredLocalElements();
    evictionListener.endLocalEviction();
  }

  /**
   * Stop running - call lifecycle event
   */
  public void postRun() {
    evictionListener.onShutdown();
  }

}
