/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache;

import org.terracotta.collections.LockableMap;

import java.util.concurrent.ConcurrentMap;

/**
 * <p>
 * DistributedCache is a distributed, Terracotta-backed, clustered cache with eviction. Object identity is maintained
 * across the cluster such that an object put into the cache is seen as the same object across all nodes and changes to
 * that object will be maintained across all nodes. Note that this implies that changes to keys or values held in the
 * DistributedCache must be done with appropriate locking on the key or value object to maintain proper Java memory
 * model semantics.
 * </p>
 *
 * @param <K> Key type
 * @param <V> Value type
 */
public interface DistributedCache<K, V> extends ConcurrentMap<K, V>, LockableMap<K> {

  /**
   * Put the key/value into the map, replacing the existing value if present. This method differs from the normal put()
   * method only in that it returns void instead of the prior value. This is important in the cluster as it avoids
   * faulting that value to return when it is rarely used.
   *
   * @param key Key
   * @param value Value
   */
  void putNoReturn(K key, V value);

  /**
   * Remove the key from the map, if it exists. This method differs from the normal remove() method only in that it
   * returns void instead of the prior value. This is important in the cluster as it avoids faulting that value to
   * return when it is rarely used.
   *
   * @param key Key
   */
  void removeNoReturn(Object key);

  /**
   * Get the value wrapped in a timestamped object. This value is used entirely for maintaining the creation and last
   * accessed time for TTI and TTL eviction.
   *
   * @param key The key to search for
   * @return The wrapped value or null if not found
   */
  TimestampedValue<V> getTimestampedValue(K key);

  /**
   * Get the value wrapped in a timestamped object without updating the last usage statistics.
   *
   * @param key The key to search for
   * @return The wrapped value or null if not found
   * @see #getTimestampedValue(Object)
   */
  TimestampedValue<V> getTimestampedValueQuiet(K key);

  /**
   * Remove the timestamped value for the given key. Previous values will be returned as null if they're expired.
   *
   * @param key The key to search for
   * @return The wrapped value or null if not found or expired
   */
  TimestampedValue<V> removeTimestampedValue(K key);

  /**
   * Get the current config for the cache (which may or may not be the config passed to the cache originally.
   *
   * @return The configuration for this cache
   */
  CacheConfig getConfig();

  /**
   * Get the number of elements local in the current node - this number should always be less than size(), and is likely
   * to be different.
   *
   * @return The local size, in range, such that 0 < localSize < size
   */
  int localSize();

  /**
   * Shut down the background eviction thread, if any.
   */
  void shutdown();

  /**
   * Retrieves a lock ID that's specific for this cache and key.
   */
  public String getLockIdForKey(final K key);
}
