/*
 * All content copyright Terracotta, Inc., unless otherwise indicated. All rights reserved.
 */
package org.terracotta.cache;

import org.terracotta.cache.evictor.CapacityEvictionPolicyData.Factory;

/**
 * <p>
 * Defines the configuration for a DistributedCache. Most parameters can be changed dynamically after the cache is
 * constructed. Exceptions will be noted in the javadoc for the property.
 * </p>
 * <p>
 * Config implementations are permitted to return out-of-date results from getters; that is, a change made to a
 * DynamicConfig may not immediately be visible. This permits clustered Configs to be accessed without locking. In order
 * to ensure that values are up to date, the user should periodically call {@link #refresh()}.
 * </p>
 */
public interface CacheConfig {

  /**
   * Get distributed cache name, used solely for logging, thread naming, and identification purposes. If no name is
   * specified, a generic name is used.
   * 
   * @return Distributed cache instance name
   */
  String getName();

  /**
   * Set distributed cache name, used solely for logging and thread naming purposes. The name of a cache cannot be
   * changed after construction.
   * 
   * @param name Distributed cache instance name
   * @return Current config instance for method chaining
   * @throws IllegalStateException If called after cache is constructed
   */
  CacheConfig setName(String name);

  /**
   * Max "time to idle" in seconds. When an element in the DistributedCache has been in the map but not been accessed
   * longer than the MaxTTI, it will expire and can be evicted. Passing 0 for the MaxTTI indicates items do not expire
   * due to lack of use.
   * 
   * @return Max TTI in seconds, must be >= 0
   */
  int getMaxTTISeconds();

  /**
   * <p>
   * Set max "time to idle" in seconds. If the Max TTI is dynamically changed from UNUSED (0) to some value, items in
   * the cache may expire immediately.
   * </p>
   * <p>
   * The background eviction thread sleep interval is based on this value and the Max TTL, so a side effect of changing
   * this value is to change the sleep interval of the eviction thread.
   * </p>
   * 
   * @param maxTTISeconds Max TTI in seconds, must be >= 0, 0 = no TTI
   * @return Current config instance for method chaining
   */
  CacheConfig setMaxTTISeconds(int maxTTISeconds);

  /**
   * Max "time to live" in seconds. When an element in the DistributedCache has been in the map longer than the MaxTTL,
   * it will expire and can be evicted. Passing 0 for the MaxTTL indicates items do not expire due to their length of
   * time in map.
   * 
   * @return Max TTL in seconds, must be >= 0
   */
  int getMaxTTLSeconds();

  /**
   * <p>
   * Set max "time to live" in seconds. If the Max TTL is dynamically changed from UNUSED (0) to some value, items in
   * the cache may expire immediately.
   * </p>
   * <p>
   * The background eviction thread sleep interval is based on this value and the Max TTL, so a side effect of changing
   * this value is to change the sleep interval of the eviction thread.
   * </p>
   * 
   * @param maxTTLSeconds Max TTL in seconds, must be >= 0, 0 = no TTL
   * @return Current config instance for method chaining
   */
  CacheConfig setMaxTTLSeconds(int maxTTLSeconds);

  /**
   * Return true if distributed cache logging is enabled
   * 
   * @return True for distributed cache logging
   */
  boolean isLoggingEnabled();

  /**
   * Set distributed cache logging on or off. This setting should only be used for development purposes and may have a
   * performance impact.
   * 
   * @param loggingEnabled Change logging to on (true) or off (false)
   * @return Current config instance for method chaining
   */
  CacheConfig setLoggingEnabled(boolean loggingEnabled);

  /**
   * Indicates whether orphan eviction is turned on. Normal local eviction will only detect whether items in the local
   * node have expired and need to be evicted. Orphan eviction is used to detect that items in the map, but not
   * currently local to ANY node, have expired and need to be evicted. This can happen either due to node failures or
   * due to object flushing.
   * 
   * @return True if orphan eviction is enabled
   */
  boolean isOrphanEvictionEnabled();

  /**
   * Turn orphan eviction on or off.
   * 
   * @see #isOrphanEvictionEnabled()
   * @return Current config instance for method chaining
   */
  CacheConfig setOrphanEvictionEnabled(boolean orphanEvictionEnabled);

  /**
   * Get frequency at which orphan eviction will occur. The number indicates the number of local eviction cycles that
   * should occur in between orphan eviction runs. Generally, you will not need to modify this setting.
   * 
   * @return Number of local eviction cycles to wait between orphan eviction. If orphan eviction is enabled, must be >
   *         0.
   */
  int getOrphanEvictionPeriod();

  /**
   * Set frequency at which orphan eviction will occur. The number indicates the number of local eviction cycles that
   * should occur in between orphan eviction runs. Generally, you will not need to modify this setting.
   * 
   * @return Current config instance for method chaining
   */
  CacheConfig setOrphanEvictionPeriod(int orphanEvictionPeriod);

  /**
   * <p>
   * Get the current target for maximum in-memory number of entries. If the target is crossed, the evictor will start
   * flushing values from the cache back to the Terracotta server array. "Flushing" is a Terracotta concept where an
   * object reference is removed from a JVM but can be reconstitued from the Terracotta server state on demand. Flushed
   * values are NOT evicted from the cache (although they are still eligible to be evicted for other reasons).
   * </p>
   * <p>
   * A value of 0 indicates that there is no in-memory limit.
   * </p>
   * 
   * @return Target max in-memory count, >= 0, 0 = no max
   */
  int getTargetMaxInMemoryCount();

  /**
   * <p>
   * Set the current target for maximum in-memory number of entries. If the target is crossed, the evictor will start
   * flushing values from the cache back to the Terracotta server array. "Flushing" is a Terracotta concept where an
   * object reference is removed from a JVM but can be reconstitued from the Terracotta server state on demand. Flushed
   * values are NOT evicted from the cache (although they are still eligible to be evicted for other reasons).
   * </p>
   * <p>
   * A value of 0 indicates that there is no in-memory limit.
   * </p>
   * 
   * @param targetMaxInMemoryCount Target max in-memory count, must be >= 0, 0 = no max
   * @return Current config instance for method chaining
   */
  CacheConfig setTargetMaxInMemoryCount(int targetMaxInMemoryCount);

  /**
   * <p>
   * Get the current target for maximum total number of entries. If the target is crossed, items will be evicted from
   * the cache. For eviction, a preference will be given to items that are not currently in use in any node - these are
   * likely not recently used and also can be removed without causing a Terracotta "fault" into memory.
   * </p>
   * <p>
   * A value of 0 indicates that there is no limit.
   * </p>
   * 
   * @return Target max total count, >= 0, 0 = no max
   */
  int getTargetMaxTotalCount();

  /**
   * <p>
   * Set the current target for maximum total number of entries. If the target is crossed, items will be evicted from
   * the cache. For eviction, a preference will be given to items that are not currently in use in any node - these are
   * likely not recently used and also can be removed without causing a Terracotta "fault" into memory.
   * </p>
   * <p>
   * A value of 0 indicates that there is no limit.
   * </p>
   * 
   * @param targetMaxTotalCount Target max total count, >= 0, 0 = no max
   * @return Current config instance for method chaining
   */
  CacheConfig setTargetMaxTotalCount(int targetMaxTotalCount);

  /**
   * Construct a cache instance based on this config's settings. Note that this config instance is NOT connected to the
   * new cache instance - it will have a copy of these settings.
   * 
   * @param <K> Key type
   * @param <V> Value type
   * @return Map constructed by this method based on previous parameters and defaults
   */
  <K, V> DistributedCache<K, V> newCache();

  /**
   * Take a new snapshot of the current state of the config. For immutable configs this will generally be a no-op. For
   * dynamic configs, calling refresh will ensure that values returned by getters reflect the most recent changes.
   * 
   * @return <code>true</code> if the config has changed due to the refresh
   */
  boolean refresh();

  /**
   * Causes the current thread to wait for this configuration object to change. If the config has not changed within
   * <code>maxWait</code> milliseconds then the method will return regardless.
   * 
   * @param maxWait maximum time to wait (in milliseconds)
   */
  void waitForChange(long maxWait);

  /**
   * Get the <tt>CapacityEvictionPolicyData.Factory</tt> to use for creating <tt>CapacityEvictionPolicyData</tt>
   */
  Factory getCapacityEvictionPolicyDataFactory();

  /**
   * Set the <tt>CapacityEvictionPolicyData.Factory</tt> to use for creating <tt>CapacityEvictionPolicyData</tt>
   */
  CacheConfig setCapacityEvictionPolicyDataFactory(Factory factory);
}
