/*
 * Copyright 2016-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.integration.kafka.dsl;

import java.util.regex.Pattern;

import org.apache.kafka.common.TopicPartition;

import org.springframework.integration.kafka.inbound.KafkaMessageDrivenChannelAdapter;
import org.springframework.integration.kafka.inbound.KafkaMessageSource.KafkaAckCallbackFactory;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.core.ProducerFactory;
import org.springframework.kafka.listener.AbstractMessageListenerContainer;
import org.springframework.kafka.listener.GenericMessageListenerContainer;
import org.springframework.kafka.listener.config.ContainerProperties;
import org.springframework.kafka.requestreply.ReplyingKafkaTemplate;
import org.springframework.kafka.support.TopicPartitionInitialOffset;

/**
 * Factory class for Apache Kafka components.
 *
 * @author Artem Bilan
 * @author Nasko Vasilev
 * @author Gary Russell
 *
 * @since 3.0
 */
public final class Kafka {

	/**
	 * Create an initial {@link KafkaProducerMessageHandlerSpec}.
	 * @param kafkaTemplate the {@link KafkaTemplate} to use
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @param <S> the {@link KafkaProducerMessageHandlerSpec} extension type.
	 * @return the KafkaProducerMessageHandlerSpec.
	 */
	public static <K, V, S extends KafkaProducerMessageHandlerSpec<K, V, S>> KafkaProducerMessageHandlerSpec<K, V, S> outboundChannelAdapter(
			KafkaTemplate<K, V> kafkaTemplate) {

		return new KafkaProducerMessageHandlerSpec<>(kafkaTemplate);
	}

	/**
	 * Create an initial {@link KafkaProducerMessageHandlerSpec} with ProducerFactory.
	 * @param producerFactory the {@link ProducerFactory} Java 8 Lambda.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaProducerMessageHandlerSpec.
	 * @see <a href="https://kafka.apache.org/documentation.html#producerconfigs">Kafka Producer Configs</a>
	 */
	public static <K, V> KafkaProducerMessageHandlerSpec.KafkaProducerMessageHandlerTemplateSpec<K, V> outboundChannelAdapter(
			ProducerFactory<K, V> producerFactory) {

		return new KafkaProducerMessageHandlerSpec.KafkaProducerMessageHandlerTemplateSpec<>(producerFactory);
	}

	/**
	 * Create an initial {@link KafkaInboundChannelAdapterSpec} with the consumer factory and
	 * topics.
	 * @param consumerFactory the consumer factory.
	 * @param topics the topic(s).
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the spec.
	 * @since 3.0.1
	 */
	public static <K, V> KafkaInboundChannelAdapterSpec<K, V> inboundChannelAdapter(
			ConsumerFactory<K, V> consumerFactory, String... topics) {

		return new KafkaInboundChannelAdapterSpec<>(consumerFactory, topics);
	}

	/**
	 * Create an initial {@link KafkaInboundChannelAdapterSpec} with the consumer factory and
	 * topics with a custom ack callback factory.
	 * @param consumerFactory the consumer factory.
	 * @param ackCallbackFactory the callback factory.
	 * @param topics the topic(s).
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the spec.
	 * @since 3.0.1
	 */
	public static <K, V> KafkaInboundChannelAdapterSpec<K, V> inboundChannelAdapter(
			ConsumerFactory<K, V> consumerFactory,
			KafkaAckCallbackFactory<K, V> ackCallbackFactory, String... topics) {

		return new KafkaInboundChannelAdapterSpec<>(consumerFactory, ackCallbackFactory, topics);
	}

	/**
	 * Create an initial {@link KafkaMessageDrivenChannelAdapterSpec}.
	 * @param listenerContainer the {@link AbstractMessageListenerContainer}.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @param <S> the {@link KafkaMessageDrivenChannelAdapterSpec} extension type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.
	 */
	public static <K, V, S extends KafkaMessageDrivenChannelAdapterSpec<K, V, S>> KafkaMessageDrivenChannelAdapterSpec<K, V, S> messageDrivenChannelAdapter(
			AbstractMessageListenerContainer<K, V> listenerContainer) {

		return messageDrivenChannelAdapter(listenerContainer, KafkaMessageDrivenChannelAdapter.ListenerMode.record);
	}

	/**
	 * Create an initial {@link KafkaMessageDrivenChannelAdapterSpec}.
	 * @param listenerContainer the {@link AbstractMessageListenerContainer}.
	 * @param listenerMode the {@link KafkaMessageDrivenChannelAdapter.ListenerMode}.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @param <A> the {@link KafkaMessageDrivenChannelAdapterSpec} extension type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.
	 */
	public static <K, V, A extends KafkaMessageDrivenChannelAdapterSpec<K, V, A>> KafkaMessageDrivenChannelAdapterSpec<K, V, A> messageDrivenChannelAdapter(
			AbstractMessageListenerContainer<K, V> listenerContainer,
			KafkaMessageDrivenChannelAdapter.ListenerMode listenerMode) {

		return new KafkaMessageDrivenChannelAdapterSpec<>(listenerContainer, listenerMode);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param containerProperties the {@link ContainerProperties} to use.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory, ContainerProperties containerProperties) {

		return messageDrivenChannelAdapter(consumerFactory, containerProperties,
				KafkaMessageDrivenChannelAdapter.ListenerMode.record);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param containerProperties the {@link ContainerProperties} to use.
	 * @param listenerMode the {@link KafkaMessageDrivenChannelAdapter.ListenerMode}.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory, ContainerProperties containerProperties,
			KafkaMessageDrivenChannelAdapter.ListenerMode listenerMode) {

		return messageDrivenChannelAdapter(
				new KafkaMessageListenerContainerSpec<>(consumerFactory,
						containerProperties), listenerMode);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param topicPartitions the {@link TopicPartition} vararg.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory,
			TopicPartitionInitialOffset... topicPartitions) {

		return messageDrivenChannelAdapter(consumerFactory, KafkaMessageDrivenChannelAdapter.ListenerMode.record,
				topicPartitions);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param listenerMode the {@link KafkaMessageDrivenChannelAdapter.ListenerMode}.
	 * @param topicPartitions the {@link TopicPartition} vararg.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory,
			KafkaMessageDrivenChannelAdapter.ListenerMode listenerMode,
			TopicPartitionInitialOffset... topicPartitions) {

		return messageDrivenChannelAdapter(
				new KafkaMessageListenerContainerSpec<>(consumerFactory,
						topicPartitions), listenerMode);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param topics the topics vararg.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory, String... topics) {

		return messageDrivenChannelAdapter(consumerFactory, KafkaMessageDrivenChannelAdapter.ListenerMode.record,
				topics);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param listenerMode the {@link KafkaMessageDrivenChannelAdapter.ListenerMode}.
	 * @param topics the topics vararg.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory, KafkaMessageDrivenChannelAdapter.ListenerMode listenerMode,
			String... topics) {

		return messageDrivenChannelAdapter(
				new KafkaMessageListenerContainerSpec<>(consumerFactory,
						topics), listenerMode);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param topicPattern the topicPattern vararg.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory, Pattern topicPattern) {

		return messageDrivenChannelAdapter(consumerFactory, KafkaMessageDrivenChannelAdapter.ListenerMode.record,
				topicPattern);
	}

	/**
	 * Create an initial
	 * {@link KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec}.
	 * @param consumerFactory the {@link ConsumerFactory}.
	 * @param listenerMode the {@link KafkaMessageDrivenChannelAdapter.ListenerMode}.
	 * @param topicPattern the topicPattern vararg.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type.
	 * @return the KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec.
	 */
	public static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			ConsumerFactory<K, V> consumerFactory,
			KafkaMessageDrivenChannelAdapter.ListenerMode listenerMode, Pattern topicPattern) {

		return messageDrivenChannelAdapter(
				new KafkaMessageListenerContainerSpec<>(consumerFactory,
						topicPattern),
				listenerMode);
	}

	/**
	 * Create an initial {@link KafkaProducerMessageHandlerSpec}.
	 * @param kafkaTemplate the {@link ReplyingKafkaTemplate} to use
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type (request).
	 * @param <R> the Kafka message value type (reply).
	 * @param <S> the {@link KafkaOutboundGatewaySpec} extension type.
	 * @return the KafkaGatewayMessageHandlerSpec.
	 * @since 3.0.2
	 */
	public static <K, V, R, S extends KafkaOutboundGatewaySpec<K, V, R, S>> KafkaOutboundGatewaySpec<K, V, R, S> outboundGateway(
			ReplyingKafkaTemplate<K, V, R> kafkaTemplate) {

		return new KafkaOutboundGatewaySpec<>(kafkaTemplate);
	}

	/**
	 * Create an initial {@link KafkaProducerMessageHandlerSpec} with ProducerFactory.
	 * @param producerFactory the {@link ProducerFactory} Java 8 Lambda.
	 * @param replyContainer a listener container for replies.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type (request).
	 * @param <R> the Kafka message value type (reply).
	 * @return the KafkaGatewayMessageHandlerSpec.
	 * @since 3.0.2
	 */
	public static <K, V, R> KafkaOutboundGatewaySpec.KafkaGatewayMessageHandlerTemplateSpec<K, V, R> outboundGateway(
			ProducerFactory<K, V> producerFactory, GenericMessageListenerContainer<K, R> replyContainer) {

		return new KafkaOutboundGatewaySpec.KafkaGatewayMessageHandlerTemplateSpec<>(producerFactory,
				replyContainer);
	}

	/**
	 * Create an initial {@link KafkaInboundGatewaySpec} with the provided container and
	 * template.
	 * @param container the container.
	 * @param template the template.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type (request).
	 * @param <R> the Kafka message value type (reply).
	 * @param <S> the {@link KafkaInboundGatewaySpec} extension type.
	 * @return the spec.
	 * @since 3.0.2
	 */
	public static <K, V, R, S extends KafkaInboundGatewaySpec<K, V, R, S>> KafkaInboundGatewaySpec<K, V, R, S> inboundGateway(
			AbstractMessageListenerContainer<K, V> container, KafkaTemplate<K, R> template) {

		return new KafkaInboundGatewaySpec<>(container, template);
	}

	/**
	 * Create an initial {@link KafkaInboundGatewaySpec} with the provided consumer factory,
	 * container properties and producer factory.
	 * @param consumerFactory the consumer factory.
	 * @param containerProperties the container properties.
	 * @param producerFactory the producer factory.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type (request).
	 * @param <R> the Kafka message value type (reply).
	 * @return the spec.
	 * @since 3.0.2
	 */
	public static <K, V, R> KafkaInboundGatewaySpec.KafkaInboundGatewayListenerContainerSpec<K, V, R> inboundGateway(
			ConsumerFactory<K, V> consumerFactory, ContainerProperties containerProperties,
			ProducerFactory<K, R> producerFactory) {

		return inboundGateway(
				new KafkaMessageListenerContainerSpec<>(consumerFactory, containerProperties),
				new KafkaTemplateSpec<>(producerFactory));
	}

	/**
	 * Create an initial {@link KafkaInboundGatewaySpec} with the provided container and
	 * template specs.
	 * @param containerSpec the container spec.
	 * @param templateSpec the template spec.
	 * @param <K> the Kafka message key type.
	 * @param <V> the Kafka message value type (request).
	 * @param <R> the Kafka message value type (reply).
	 * @return the spec.
	 * @since 3.0.2
	 */
	public static <K, V, R> KafkaInboundGatewaySpec.KafkaInboundGatewayListenerContainerSpec<K, V, R> inboundGateway(
			KafkaMessageListenerContainerSpec<K, V> containerSpec, KafkaTemplateSpec<K, R> templateSpec) {

		return new KafkaInboundGatewaySpec.KafkaInboundGatewayListenerContainerSpec<>(containerSpec, templateSpec);
	}

	private static <K, V>
	KafkaMessageDrivenChannelAdapterSpec.KafkaMessageDrivenChannelAdapterListenerContainerSpec<K, V> messageDrivenChannelAdapter(
			KafkaMessageListenerContainerSpec<K, V> spec, KafkaMessageDrivenChannelAdapter.ListenerMode listenerMode) {

		return new KafkaMessageDrivenChannelAdapterSpec
				.KafkaMessageDrivenChannelAdapterListenerContainerSpec<>(spec, listenerMode);
	}

	private Kafka() {
		super();
	}

}
