/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.integration.aws.lock;

import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.springframework.dao.CannotAcquireLockException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.integration.aws.lock.DynamoDbLockRepository;
import org.springframework.integration.support.locks.ExpirableLockRegistry;
import org.springframework.integration.support.locks.RenewableLockRegistry;
import org.springframework.util.Assert;
import software.amazon.awssdk.services.dynamodb.model.TransactionConflictException;

public class DynamoDbLockRegistry
implements ExpirableLockRegistry,
RenewableLockRegistry {
    private static final int DEFAULT_IDLE = 100;
    private final Map<String, DynamoDbLock> locks = new ConcurrentHashMap<String, DynamoDbLock>();
    private final DynamoDbLockRepository dynamoDbLockRepository;
    private Duration idleBetweenTries = Duration.ofMillis(100L);

    public DynamoDbLockRegistry(DynamoDbLockRepository dynamoDbLockRepository) {
        Assert.notNull((Object)dynamoDbLockRepository, (String)"'dynamoDbLockRepository' must not be null");
        this.dynamoDbLockRepository = dynamoDbLockRepository;
    }

    public void setIdleBetweenTries(Duration idleBetweenTries) {
        Assert.notNull((Object)idleBetweenTries, (String)"'idleBetweenTries' must not be null");
        this.idleBetweenTries = idleBetweenTries;
    }

    public Lock obtain(Object lockKey) {
        Assert.isInstanceOf(String.class, (Object)lockKey, (String)"'lockKey' must of String type");
        return this.locks.computeIfAbsent((String)lockKey, x$0 -> new DynamoDbLock((String)x$0));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void expireUnusedOlderThan(long age) {
        long now = System.currentTimeMillis();
        Map<String, DynamoDbLock> map = this.locks;
        synchronized (map) {
            this.locks.entrySet().removeIf(entry -> {
                DynamoDbLock lock = (DynamoDbLock)entry.getValue();
                return now - lock.lastUsed > age && !lock.isAcquiredInThisProcess();
            });
        }
    }

    public void renewLock(Object lockKey) {
        Assert.isInstanceOf(String.class, (Object)lockKey, (String)"'lockKey' must of String type");
        String lockId = (String)lockKey;
        DynamoDbLock dynamoDbLock = this.locks.get(lockId);
        if (dynamoDbLock == null) {
            throw new IllegalStateException("Could not found mutex at " + lockId);
        }
        if (!dynamoDbLock.renew()) {
            throw new IllegalStateException("Could not renew mutex at " + lockId);
        }
    }

    public String toString() {
        return "DynamoDbLockRegistry{tableName='" + this.dynamoDbLockRepository.getTableName() + "', owner='" + this.dynamoDbLockRepository.getOwner() + "}";
    }

    private final class DynamoDbLock
    implements Lock {
        private final ReentrantLock delegate = new ReentrantLock();
        private final String key;
        private volatile long lastUsed = System.currentTimeMillis();

        private DynamoDbLock(String key) {
            this.key = key;
        }

        private void rethrowAsLockException(Exception e) {
            throw new CannotAcquireLockException("Failed to lock at " + this.key, (Throwable)e);
        }

        @Override
        public void lock() {
            this.delegate.lock();
            while (true) {
                try {
                    while (!this.doLock()) {
                        this.sleepBetweenRetries();
                    }
                }
                catch (TransactionConflictException transactionConflictException) {
                    continue;
                }
                catch (InterruptedException interruptedException) {
                    continue;
                }
                catch (Exception ex) {
                    this.delegate.unlock();
                    this.rethrowAsLockException(ex);
                    continue;
                }
                break;
            }
        }

        @Override
        public void lockInterruptibly() throws InterruptedException {
            this.delegate.lockInterruptibly();
            while (true) {
                try {
                    while (!this.doLock()) {
                        this.sleepBetweenRetries();
                        if (!Thread.currentThread().isInterrupted()) continue;
                        throw new InterruptedException();
                    }
                }
                catch (TransactionConflictException transactionConflictException) {
                    continue;
                }
                catch (InterruptedException ie) {
                    this.delegate.unlock();
                    Thread.currentThread().interrupt();
                    throw ie;
                }
                catch (Exception e) {
                    this.delegate.unlock();
                    this.rethrowAsLockException(e);
                    continue;
                }
                break;
            }
        }

        @Override
        public boolean tryLock() {
            try {
                return this.tryLock(0L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                return false;
            }
        }

        @Override
        public boolean tryLock(long time, TimeUnit unit) throws InterruptedException {
            long now = System.currentTimeMillis();
            if (!this.delegate.tryLock(time, unit)) {
                return false;
            }
            long expire = now + TimeUnit.MILLISECONDS.convert(time, unit);
            while (true) {
                try {
                    boolean acquired;
                    while (!(acquired = this.doLock()) && System.currentTimeMillis() < expire) {
                        this.sleepBetweenRetries();
                        if (!Thread.currentThread().isInterrupted()) continue;
                        throw new InterruptedException();
                    }
                    if (!acquired) {
                        this.delegate.unlock();
                    }
                    return acquired;
                }
                catch (TransactionConflictException transactionConflictException) {
                    continue;
                }
                catch (Exception ex) {
                    this.delegate.unlock();
                    this.rethrowAsLockException(ex);
                    continue;
                }
                break;
            }
        }

        private boolean doLock() throws InterruptedException {
            boolean acquired = DynamoDbLockRegistry.this.dynamoDbLockRepository.acquire(this.key);
            if (acquired) {
                this.lastUsed = System.currentTimeMillis();
            }
            return acquired;
        }

        @Override
        public void unlock() {
            if (!this.delegate.isHeldByCurrentThread()) {
                throw new IllegalMonitorStateException("The current thread doesn't own mutex at '" + this.key + "'");
            }
            if (this.delegate.getHoldCount() > 1) {
                this.delegate.unlock();
                return;
            }
            while (true) {
                try {
                    DynamoDbLockRegistry.this.dynamoDbLockRepository.delete(this.key);
                    return;
                }
                catch (TransactionConflictException ex) {
                    try {
                        this.sleepBetweenRetries();
                    }
                    catch (InterruptedException interruptedException) {}
                    continue;
                }
                catch (Exception ex) {
                    throw new DataAccessResourceFailureException("Failed to release mutex at " + this.key, (Throwable)ex);
                }
                break;
            }
            finally {
                this.delegate.unlock();
            }
        }

        /*
         * Loose catch block
         */
        public boolean renew() {
            if (!this.delegate.isHeldByCurrentThread()) {
                throw new IllegalMonitorStateException("The current thread doesn't own mutex at " + this.key);
            }
            while (true) {
                try {
                    boolean renewed = DynamoDbLockRegistry.this.dynamoDbLockRepository.renew(this.key);
                    if (renewed) {
                        this.lastUsed = System.currentTimeMillis();
                    }
                    return renewed;
                }
                catch (TransactionConflictException ex) {
                    try {
                        this.sleepBetweenRetries();
                    }
                    catch (InterruptedException interruptedException) {}
                    continue;
                }
                break;
            }
            catch (Exception ex) {
                throw new DataAccessResourceFailureException("Failed to renew mutex at " + this.key, (Throwable)ex);
            }
        }

        public boolean isAcquiredInThisProcess() {
            return DynamoDbLockRegistry.this.dynamoDbLockRepository.isAcquired(this.key);
        }

        private void sleepBetweenRetries() throws InterruptedException {
            Thread.sleep(DynamoDbLockRegistry.this.idleBetweenTries.toMillis());
        }

        @Override
        public Condition newCondition() {
            throw new UnsupportedOperationException("DynamoDb locks don't support conditions.");
        }

        public String toString() {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd@HH:mm:ss.SSS");
            return "DynamoDbLock [lockKey=" + this.key + ",lockedAt=" + dateFormat.format(new Date(this.lastUsed)) + "]";
        }
    }
}

