/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.contract.stubrunner;

import java.io.IOException;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.lang.Nullable;

class DirectoryCopyingVisitor
extends SimpleFileVisitor<Path> {
    private static final List<String> FOLDERS_TO_DELETE = Arrays.asList("contracts", "mappings");
    private static final Log log = LogFactory.getLog(DirectoryCopyingVisitor.class);
    private final Path from;
    private final Path to;

    DirectoryCopyingVisitor(Path from, Path to) {
        this.from = from;
        this.to = to;
        if (log.isDebugEnabled()) {
            log.debug((Object)("Will copy from [" + from.toString() + "] to [" + to.toString() + "]"));
        }
    }

    @Override
    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
        Path relativePath = this.from.relativize(dir);
        if (".git".equals(relativePath.toString())) {
            return FileVisitResult.SKIP_SUBTREE;
        }
        Path targetPath = this.to.resolve(relativePath);
        if (!Files.exists(targetPath, new LinkOption[0])) {
            if (log.isDebugEnabled()) {
                log.debug((Object)("Created a folder [" + targetPath.toString() + "]"));
            }
            Files.createDirectory(targetPath, new FileAttribute[0]);
        } else {
            if (log.isDebugEnabled()) {
                log.debug((Object)("Folder [" + targetPath.toString() + "] already exists"));
            }
            if (FOLDERS_TO_DELETE.contains(targetPath.toFile().getName())) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)("Will remove the folder [" + targetPath.toString() + "]"));
                }
                this.deleteRecursively(targetPath);
                Files.createDirectory(targetPath, new FileAttribute[0]);
                if (log.isDebugEnabled()) {
                    log.debug((Object)("Recreated folder [" + targetPath.toString() + "]"));
                }
            }
        }
        return FileVisitResult.CONTINUE;
    }

    private boolean deleteRecursively(@Nullable Path root) throws IOException {
        if (root == null) {
            return false;
        }
        if (!Files.exists(root, new LinkOption[0])) {
            return false;
        }
        Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                if (exc == null) {
                    boolean deleted;
                    int maxTries = 5;
                    int count = 0;
                    do {
                        if (deleted = this.isDeleted(dir)) {
                            if (!log.isDebugEnabled()) break;
                            log.debug((Object)("Deleted [" + dir + "]"));
                            break;
                        }
                        if (log.isDebugEnabled()) {
                            log.debug((Object)("Failed to delete [" + dir + "]"));
                        }
                        ++count;
                        try {
                            Thread.sleep(2L);
                        }
                        catch (InterruptedException e1) {
                            Thread.currentThread().interrupt();
                            break;
                        }
                    } while (count < maxTries);
                    if (!deleted) {
                        if (log.isDebugEnabled()) {
                            log.debug((Object)("Failed to delete [" + dir + "] after [" + maxTries + "] attempts to do it"));
                        }
                        throw new DirectoryNotEmptyException(dir.toString());
                    }
                    return FileVisitResult.CONTINUE;
                }
                throw exc;
            }

            private boolean isDeleted(Path dir) throws IOException {
                try {
                    Files.delete(dir);
                    return true;
                }
                catch (DirectoryNotEmptyException e) {
                    return false;
                }
            }
        });
        return true;
    }

    @Override
    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
        Path relativePath = this.to.resolve(this.from.relativize(file));
        Files.copy(file, relativePath, StandardCopyOption.REPLACE_EXISTING);
        if (log.isDebugEnabled()) {
            log.debug((Object)("Copied file from [" + file.toString() + "] to [" + relativePath.toString() + "]"));
        }
        return FileVisitResult.CONTINUE;
    }
}

