package org.ow2.easywsdl.tooling.java2wsdl;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import javax.ejb.Remote;
import javax.jws.WebMethod;
import javax.xml.namespace.QName;
import org.ow2.easywsdl.schema.SchemaFactory;
import org.ow2.easywsdl.schema.api.ComplexContent;
import org.ow2.easywsdl.schema.api.ComplexType;
import org.ow2.easywsdl.schema.api.Element;
import org.ow2.easywsdl.schema.api.Extension;
import org.ow2.easywsdl.schema.api.Schema;
import org.ow2.easywsdl.schema.api.Sequence;
import org.ow2.easywsdl.wsdl.WSDLFactory;
import org.ow2.easywsdl.wsdl.api.Binding;
import org.ow2.easywsdl.wsdl.api.BindingOperation;
import org.ow2.easywsdl.wsdl.api.Description;
import org.ow2.easywsdl.wsdl.api.Endpoint;
import org.ow2.easywsdl.wsdl.api.Fault;
import org.ow2.easywsdl.wsdl.api.Input;
import org.ow2.easywsdl.wsdl.api.InterfaceType;
import org.ow2.easywsdl.wsdl.api.Operation;
import org.ow2.easywsdl.wsdl.api.Output;
import org.ow2.easywsdl.wsdl.api.Service;
import org.ow2.easywsdl.wsdl.api.WSDLException;
import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfDescription.WSDLVersionConstants;

public class JavaToEasyWSDL {

	private static String XML_SCHEMA_PREFIX = "http://www.w3.org/2001/XMLSchema";

	private boolean verbose;

	private String namespace;

	private String serviceName;
	
	private String webServiceName;
	
	private boolean polymorph;

	private List<String> polymorphClasses;
	
	private boolean isCurrentClassPOJO = false;
	
	public JavaToEasyWSDL(boolean verbose) {
		this.verbose = verbose;
	}

	public WSDLGenerationContext generateWSDL(List<Class<?>> classes, boolean polymorph, List<String> polymorphClasses)
			throws WSDLException {
		
		this.polymorph = polymorph;
		this.polymorphClasses = polymorphClasses;
		
		WSDLGenerationContext res = new WSDLGenerationContext();

		Description desc;
		desc = WSDLFactory.newInstance().newDescription(
				WSDLVersionConstants.WSDL11);
		res.setDescription(desc);
		for (Class<?> clazz : classes) {
			generateGlobalInfo(clazz, res);
			InterfaceType itf = generateInterface(clazz, res);
			Method[] methods = clazz.getDeclaredMethods();
			for (Method method : methods) {
				//Generate operation only for methods tagged with @WebMethod
				//if the class analysed is a JSR 181 POJO
				if ((!isCurrentClassPOJO) || (method.getAnnotation(WebMethod.class)!=null)){
					generateOperation(res, itf, method);
				}
			}

			Service service = desc.createService();
			
			service.setQName(cQName(serviceName));
			desc.addService(service);
			Endpoint ep = service.createEndpoint();

			String endpointName = webServiceName+"Port";
			if (clazz.getAnnotation(javax.jws.WebService.class) != null) {
				javax.jws.WebService webServiceAnnotation = clazz
						.getAnnotation(javax.jws.WebService.class);
				if (!"".equals(webServiceAnnotation.portName())) {
					endpointName = webServiceAnnotation.portName();
				}
			}
			
			ep.setName(endpointName);
			ep.setAddress("MODIFY_ADDRESS");
			service.addEndpoint(ep);

			// create binding
			Binding binding = desc.createDefaultSoapBinding(webServiceName+"Binding", ep,
					itf);
			// SoapAction must be an uri to be compliant with the xsd schema,
			// and with the ws i. It is better to let it empty, than to set a QName 
			// inside, which is not compliant.
			List<BindingOperation> ops =binding.getBindingOperations();
			for (BindingOperation op : ops)
			{
				op.setSoapAction("");
			}
			desc.addBinding(binding);
		}
		
		// Handle additional polymorph classes
		for (String className : this.polymorphClasses) {
			try {
				Class<?> clazz = Class.forName(className);
				generateComplexTypeSchema(res, clazz);
			} catch (ClassNotFoundException e) {
				System.err.println("Warning : Additional polymorph class not found for name " + className);
			}
		}

		return res;
	}

	private void generateGlobalInfo(Class<?> clazz,
			WSDLGenerationContext context) throws WSDLException {
		debug("Generating global info for class : " + clazz);

		//Determine if the class is a JSR181 POJO
		if (clazz.getAnnotation(Remote.class) != null){
			isCurrentClassPOJO=true;
		}else{
			isCurrentClassPOJO=false;
		}
		
		namespace = createNamespace(clazz.getPackage());
		serviceName = clazz.getSimpleName();
		webServiceName = serviceName +"PortType";
		Description desc = context.getDescription();

		if (clazz.getAnnotation(javax.jws.WebService.class) != null) {
			javax.jws.WebService webServiceAnnotation = clazz
					.getAnnotation(javax.jws.WebService.class);
			if (!"".equals(webServiceAnnotation.targetNamespace())) {
				namespace = webServiceAnnotation.targetNamespace();
			}
			if (!"".equals(webServiceAnnotation.serviceName())) {
				serviceName = webServiceAnnotation.serviceName();
			}
			if (!"".equals(webServiceAnnotation.name())) {
				webServiceName = webServiceAnnotation.name();
			}
		}

		desc.setTargetNamespace(namespace);
		desc.setQName(cQName(serviceName));
	}

	private void generateOperation(WSDLGenerationContext context,
			InterfaceType itf, Method method) throws WSDLException {
		debug("******** Processing Method : " + method);

		Operation operation = itf.createOperation();
		String opName = method.getName();
		operation.setQName(cQName(opName));
		//Removed to suppress the WARNING (unused for WSDLs 1.1)
		//operation.setPattern(MEPPatternConstants.IN_OUT);

		itf.addOperation(operation);

		generateInputParameter(context, method, operation);

		generateOutputParameter(context, method, operation);

		generateFault(context, method, operation);

	}

	private void generateFault(WSDLGenerationContext context, Method method,
			Operation operation) throws WSDLException {
		Type[] exceptionsType = method.getGenericExceptionTypes();

		if (exceptionsType.length > 0) {
			debug("    Fault Type:--");

			for (int i = 0; i < exceptionsType.length; i++) {
				Type type = exceptionsType[i];

				if (!isInstanceOf(type, RuntimeException.class)
						&& isInstanceOf(type, Exception.class)) {
					debug("                    --  " + type);
					Class<?> clazz = (Class<?>) type;
					Fault fault = operation.createFault();
					operation.addFault(fault);
					String namespace = context.getDescription()
							.getTargetNamespace();
					String faultName = clazz.getSimpleName();
					fault.setName(faultName);
					fault.setMessageName(new QName(context.getDescription()
							.getTargetNamespace(), faultName));

					Schema schema = context.getSchema(namespace);
					Element elem = schema.createElement();
					elem.setQName(new QName(context.getDescription()
							.getTargetNamespace(), faultName + "Element"));
					schema.addElement(elem);
					fault.setElement(elem);
					ComplexType cType = (ComplexType) schema
							.createComplexType();
					schema.addType(cType);
					elem.setType(cType);
					cType.setQName(new QName(namespace, faultName + "Type"));
					Sequence sequence = cType.createSequence();
					cType.setSequence(sequence);
					Field[] fields = clazz.getDeclaredFields();
					generateExceptionFields(context, sequence, namespace,
							fields);
					clazz = clazz.getSuperclass();
					while (Exception.class != clazz) {
						fields = clazz.getDeclaredFields();
						generateExceptionFields(context, sequence, namespace,
								fields);
						clazz = clazz.getSuperclass();
					}
				}
			}
		}
	}

	private void generateExceptionFields(WSDLGenerationContext context,
			Sequence sequence, String namespace, Field[] fields) {
		for (Field field : fields) {
			Element elem = sequence.createElement();
			elem.setQName(new QName(namespace, field.getName()));
			elem.setType(getXMLSchemaType(context, field.getType()));
			sequence.addElement(elem);
		}
	}

	private void generateOutputParameter(WSDLGenerationContext context,
			Method method, Operation operation) throws WSDLException {
		Type returnType = method.getGenericReturnType();
		debug("    Return Type:------  " + returnType);
		String methodName = method.getName();
		String paramName = methodName + "Response";
		
		Output output = operation.createOutput();
		output.setName(paramName);
		output.setMessageName(cQName(paramName + "Message"));
		operation.setOutput(output);

		QName elemName = cQName(paramName);
		Schema schema = context.getSchema(namespace);

		Element elem = schema.createElement();
		elem.setQName(elemName);
		schema.addElement(elem);
		output.setElement(elem);

		ComplexType elemType = (ComplexType) schema.createComplexType();
		//elemType.setQName(cQName(elemName.getLocalPart() + "Type"));
		elemType.setQName(cQName(elemName.getLocalPart() ));

		elem.setType(elemType);
		schema.addType(elemType);

		Sequence sequence = elemType.createSequence();
		elemType.setSequence(sequence);
		
		if (!returnType.equals(Void.TYPE)) {
			Element elem2 = sequence.createElement();
			
			String returnName = "return";
			//Overwrite the returnName with value of @WebReturn annotation if exists
			javax.jws.WebResult webReturn = method.getAnnotation(javax.jws.WebResult.class);
			if (webReturn != null)
			{
				returnName = webReturn.name();
			}
			elem2.setQName(cQName(returnName));

			if (returnType instanceof ParameterizedType )
			{
				ParameterizedType pType = (ParameterizedType) returnType;
				if (isInstanceOf(pType.getRawType(), List.class))
				{
					elem2.setType(getXMLSchemaType(context, pType.getActualTypeArguments()[0]));
					elem2.setMaxOccurs("unbounded");
				}
			}else if ( ((Class<?>)returnType).isArray() )
			{
				elem2.setType(getXMLSchemaType(context,((Class<?>)returnType).getComponentType()));
				elem2.setMaxOccurs("unbounded");
			}
			else
			{
				elem2.setType(getXMLSchemaType(context, returnType));
			}
			
			//elem2.setType(getXMLSchemaType(context, returnType));
			sequence.addElement(elem2);
		}
	}

	private void generateInputParameter(WSDLGenerationContext context,
			Method method, Operation operation) throws WSDLException {
		
		debug("Generating input parameters for : " + method.getName());
		
		Input input = operation.createInput();
		String methodName = method.getName();
		String paramName = methodName + "Request";
		input.setName(paramName);
		input.setMessageName(cQName(methodName + "Request" + "Message"));

		operation.setInput(input);

		//QName elemName = cQName(methodName + "Request" + "Element");
		Schema schema = context.getSchema(namespace);

		Element elem = schema.createElement();
		//elem.setQName(cQName("parameters"));
		elem.setQName(cQName(methodName));
		schema.addElement(elem);
		input.setElement(elem);

		ComplexType elemType = (ComplexType) schema.createComplexType();
		//elemType.setQName(cQName(elemName.getLocalPart() + "Type"));
		elemType.setQName(cQName(methodName));

		elem.setType(elemType);
		schema.addType(elemType);

		Type[] paramsType = method.getGenericParameterTypes();
		Annotation[][] methodParamsAnnotations = method
				.getParameterAnnotations();

		if (paramsType.length > 0) {
			Sequence sequence = elemType.createSequence();
			elemType.setSequence(sequence);
			for (int i = 0; i < paramsType.length; i++) {
				Type type = paramsType[i];
				
				Element elem2 = sequence.createElement();
				if (methodParamsAnnotations[i].length != 0) {
					Annotation[] paramAnnotations = methodParamsAnnotations[i];
					for (int j = 0; j < paramAnnotations.length; j++) {
						if (paramAnnotations[j] instanceof javax.jws.WebParam
								&& ((javax.jws.WebParam) paramAnnotations[j])
										.name() != null) {
							elem2
									.setQName(cQName(((javax.jws.WebParam) paramAnnotations[j])
											.name()));
						}
					}
				} else {
					elem2.setQName(cQName("arg" + i));
				}
				
				if (type instanceof ParameterizedType)
				{
					ParameterizedType pType = (ParameterizedType) type;
					if (isInstanceOf(pType.getRawType(), List.class))
					{
						elem2.setType(getXMLSchemaType(context, pType.getActualTypeArguments()[0]));
						elem2.setMaxOccurs("unbounded");
					}
				}else if ( ((Class<?>)type).isArray() )
				{
					elem2.setType(getXMLSchemaType(context,((Class<?>)type).getComponentType()));
					elem2.setMaxOccurs("unbounded");
				}else
				{
					elem2.setType(getXMLSchemaType(context, type));
				}
				elem2.setMinOccurs(0);
				sequence.addElement(elem2);
			}
		}
	}

	private InterfaceType generateInterface(Class<?> clazz,
			WSDLGenerationContext context) {

		debug("Generating port type for class : " + clazz);
		InterfaceType itf = context.getDescription().createInterface();
		itf.setQName(new QName(namespace, webServiceName));
		context.getDescription().addInterface(itf);

		return itf;
	}

	private org.ow2.easywsdl.schema.api.Type getXMLSchemaType(
			WSDLGenerationContext context, Type type) {
		org.ow2.easywsdl.schema.api.Type simpleType = getSchemaSimpleType(type);
		if (simpleType != null) {
			return simpleType;
		} else {
			return generateComplexTypeSchema(context, type);
		}
	}

	private org.ow2.easywsdl.schema.api.Type generateComplexTypeSchema(
			WSDLGenerationContext context, Type type) {
			
		if (type instanceof TypeVariable || type instanceof GenericArrayType
				|| type instanceof WildcardType) {
			unknownType(type);
		} else if (type instanceof ParameterizedType) {
			return generateParametrizedType(context, type);
		} else {
			Class<?> clazz = (Class<?>) type;
			if (clazz.isArray()) {
				return generateCollectionComplexType(context, context
						.getDescription().getTargetNamespace(), "Array", clazz
						.getComponentType());
			} else {
				return generateComplexType(context, clazz,
						new ArrayList<Class<?>>());
			}
		}
		return SchemaFactory.getDefaultSchema().getType(
				new QName(XML_SCHEMA_PREFIX, "anyType"));
	}

	private org.ow2.easywsdl.schema.api.Type generateParametrizedType(
			WSDLGenerationContext context, Type type) {
		ParameterizedType tParameterizedType = (ParameterizedType) type;

		Type tRawType = tParameterizedType.getRawType();

		if (tRawType instanceof TypeVariable
				|| tRawType instanceof GenericArrayType
				|| tRawType instanceof WildcardType
				|| tRawType instanceof ParameterizedType) {
			unknownType(type);
		}

		if (isInstanceOf(tRawType, List.class)) {
			Class<?> componentType = (Class<?>) tParameterizedType
			.getActualTypeArguments()[0];
			return generateCollectionComplexType(context, context
						.getDescription().getTargetNamespace(), "List",
						componentType);
		} else if (isInstanceOf(tRawType, Set.class)) {
			Class<?> componentType = (Class<?>) tParameterizedType
			.getActualTypeArguments()[0];
			return generateCollectionComplexType(context, context
					.getDescription().getTargetNamespace(), "Set",
					componentType);
		} else {
			unknownType(type);
		}

		return SchemaFactory.getDefaultSchema().getType(
				new QName(XML_SCHEMA_PREFIX, "anyType"));
	}

	private org.ow2.easywsdl.schema.api.Type getSchemaSimpleType(Type type) {
		if (Boolean.TYPE.equals(type) || Boolean.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "boolean"));
		} else if (Character.TYPE.equals(type) || Character.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "short"));
		} else if (Byte.TYPE.equals(type) || Byte.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "byte"));
		} else if (Short.TYPE.equals(type) || Short.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "short"));
		} else if (Integer.TYPE.equals(type) || Integer.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getTypeInt();
		} else if (Long.TYPE.equals(type) || Long.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "long"));
		} else if (Float.TYPE.equals(type) || Float.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "float"));
		} else if (Double.TYPE.equals(type) || Double.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "double"));
		} else if (String.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getTypeString();
		}
		else if (Date.class.equals(type)) {
			return SchemaFactory.getDefaultSchema().getType(
					new QName(XML_SCHEMA_PREFIX, "dateTime"));
		}
		return null;
	}

	private org.ow2.easywsdl.schema.api.Type generateCollectionComplexType(
			WSDLGenerationContext context, String namespace,
			String collectionType, Class<?> componentType) {
		String name = collectionType + "Of" + componentType.getSimpleName();
		
		Schema schema = context.getSchema(namespace);
		org.ow2.easywsdl.schema.api.Type type = schema.getType(new QName(
				namespace, name));
		
		if (type == null) {
			ComplexType cType = (ComplexType) schema.createComplexType();
			type = cType;
			cType.setQName(new QName(namespace, name));
			schema.addType(cType);
			Sequence sequence = cType.createSequence();
			cType.setSequence(sequence);

			Element elem = sequence.createElement();
			org.ow2.easywsdl.schema.api.Type elemType = getSchemaSimpleType(componentType);
						
			if (elemType == null)
			{
				elemType = generateComplexType(
					context, componentType, new ArrayList<Class<?>>());
					
					if ("".equals(this.namespace)){
					elem.setQName(new QName(
					createNamespace(componentType.getPackage()), componentType
							.getSimpleName()));}
					else{
						elem.setQName(new QName(this.namespace, componentType
								.getSimpleName()));
					}
			}else {
				elem.setQName(elemType.getQName());
			}
			
			elem.setType(elemType);
			elem.setMinOccurs(0);
			elem.setMaxOccurs("unbounded");
			sequence.addElement(elem);
		}
		return type;
	}

	private org.ow2.easywsdl.schema.api.Type generateComplexType(
			WSDLGenerationContext context, Class<?> clazz,
			List<Class<?>> processedClasses) {
		
		String name = clazz.getSimpleName();
		String namespace;
		if ("".equals(this.namespace)){
			namespace = createNamespace(clazz.getPackage());}
		else{
			namespace= this.namespace;
		}
		Schema schema = context.getSchema(namespace);
		org.ow2.easywsdl.schema.api.Type type = schema.getType(new QName(
				namespace, name));
		
		if (type == null) {
			debug ("Generate Schema for type {" + namespace + "}" + name);
			processedClasses.add(clazz);
			ComplexType cType = (ComplexType) schema.createComplexType();
			type = cType;
			cType.setQName(new QName(namespace, name));
			schema.addType(cType);
			generateFields(context, cType, clazz, processedClasses);
		}
		return type;
	}

	private void generateFields(WSDLGenerationContext context,
			ComplexType cType, Class<?> clazz, List<Class<?>> processedClasses) {
		if (!polymorph){
			Sequence sequence = cType.createSequence();
			cType.setSequence(sequence);
			Field[] fields = clazz.getDeclaredFields();
			generateFields(context, sequence, processedClasses, clazz, fields);
			while (Object.class != clazz) {
				clazz = clazz.getSuperclass();
				if (!processedClasses.contains(clazz)) {
					processedClasses.add(clazz);
					fields = clazz.getDeclaredFields();
					generateFields(context, sequence, processedClasses, clazz,
							fields);
				}
			}
		} else {
			// polymorph parameters
			Class<?> superClazz = clazz.getSuperclass();
			if (Object.class != superClazz) {			
				ComplexContent cContent = cType.createComplexContent();
				Extension ext = cContent.createExtension();
				Sequence sequence = ext.createSequence();
				ext.setSequence(sequence);
				Field[] fields = clazz.getDeclaredFields();
				generateFields(context, sequence, processedClasses, clazz, fields);			
			
				org.ow2.easywsdl.schema.api.Type baseType = getSchemaSimpleType(superClazz);
					if (baseType == null)
					{
						ext.setBase(generateComplexType(context, superClazz, processedClasses));
					} else {
						ext.setBase(baseType);
					}
				cContent.setExtension(ext);
				cType.setComplexContent(cContent);
			} else {
				Sequence sequence = cType.createSequence();
				cType.setSequence(sequence);
				Field[] fields = clazz.getDeclaredFields();
				generateFields(context, sequence, processedClasses, clazz, fields);		
			} 
		}
	}

	private void generateFields(WSDLGenerationContext context,
			Sequence sequence, List<Class<?>> processedClasses, Class<?> clazz,
			Field[] fields) {
		for (Field field : fields) {
			if (field.getModifiers() != (Modifier.STATIC + Modifier.FINAL + Modifier.PRIVATE)) {
				org.ow2.easywsdl.schema.api.Type fieldType = getSchemaSimpleType(field
						.getType());
				Element elem = sequence.createElement();
				if ("".equals(this.namespace)){
				elem.setQName(new QName(createNamespace(clazz.getPackage()),
						field.getName()));
				}
				else{
					elem.setQName(new QName(this.namespace,
							field.getName()));
				}	
				elem.setMinOccurs(0);
				if (fieldType == null) {					
					if ((field.getGenericType() instanceof ParameterizedType)){
						ParameterizedType pType = (ParameterizedType) field.getGenericType();
						if (isInstanceOf(pType.getRawType(), List.class)){
							elem.setType(getXMLSchemaType(context, pType.getActualTypeArguments()[0]));
							elem.setMaxOccurs("unbounded");
						}
					}else if (field.getType().isArray()){
						elem.setType(getXMLSchemaType(context,field.getType().getComponentType()));
						elem.setMaxOccurs("unbounded");
					}else{
						elem.setType(generateComplexType(context, field.getType(),
								processedClasses));
					}
				} else {
					elem.setType(fieldType);
				}
				elem.setMinOccurs(0);
				sequence.addElement(elem);
			}
		}
	}

	private String createNamespace(Package package1) {
		StringTokenizer tok = new StringTokenizer(package1.getName(), ".");
		String[] packageNameElements = new String[tok.countTokens()];
		for (int i = 0; i < packageNameElements.length; i++) {
			packageNameElements[packageNameElements.length - i - 1] = tok
					.nextToken();
		}
		StringBuffer sb = new StringBuffer();
		sb.append("http://");
		for (int i = 0; i < packageNameElements.length; i++) {
			sb.append(packageNameElements[i]);
			if (i < packageNameElements.length - 1) {
				sb.append('.');
			}
		}
		return sb.toString();
	}

	private boolean isInstanceOf(Type clazz, Type other) {
		boolean res = other.equals(clazz);
		if (clazz instanceof Class<?>) {
			Type[] interfaces = ((Class<?>) clazz).getGenericInterfaces();
			for (Type itf : interfaces) {
				if (itf instanceof ParameterizedType) {
					res = res
							|| other.equals(((ParameterizedType) itf)
									.getRawType());
				} else if (itf instanceof Class<?>) {
					res = res || other.equals(itf);
				}
			}
			if (!Object.class.equals(clazz) && clazz instanceof Class<?>) {
				res = res
						|| isInstanceOf(((Class<?>) clazz).getSuperclass(),
								other);
			}
		}
		return res;
	}

	private void unknownType(Type type) {
		debug("XML Schema Generation Not Yet Implemented for " + type
				+ "\n Please fill in a bug with and attach sample code");
	}

	private void debug(String message) {
		if (verbose)
			System.out.println("debug " + message);
	}

	private QName cQName(String name) {
		return new QName(namespace, name);
	}

}
