/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch._types.aggregations;

import jakarta.json.stream.JsonGenerator;
import java.util.List;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;

// typedef: _types.aggregations.MultiTermsAggregation

@JsonpDeserializable
public class MultiTermsAggregation extends BucketAggregationBase implements AggregationVariant {
    private final List<MultiTermLookup> terms;

    @Nullable
    private final Integer size;

    // ---------------------------------------------------------------------------------------------

    private MultiTermsAggregation(Builder builder) {
        super(builder);

        this.terms = ApiTypeHelper.unmodifiableRequired(builder.terms, this, "terms");
        this.size = builder.size;
    }

    public static MultiTermsAggregation of(Function<Builder, ObjectBuilder<MultiTermsAggregation>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Aggregation variant kind.
     */
    @Override
    public Aggregation.Kind _aggregationKind() {
        return Aggregation.Kind.MultiTerms;
    }

    /**
     * Required - API name: {@code terms}
     */
    public final List<MultiTermLookup> terms() {
        return this.terms;
    }

    /**
     * API name: {@code size}
     */
    @Nullable
    public final Integer size() {
        return this.size;
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        super.serializeInternal(generator, mapper);
        if (this.size != null) {
            generator.writeKey("size");
            generator.write(this.size);

        }
        if (ApiTypeHelper.isDefined(this.terms)) {
            generator.writeKey("terms");
            generator.writeStartArray();
            for (MultiTermLookup item0 : this.terms) {
                item0.serialize(generator, mapper);

            }
            generator.writeEnd();

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link MultiTermsAggregation}.
     */

    public static class Builder extends BucketAggregationBase.AbstractBuilder<Builder> implements ObjectBuilder<MultiTermsAggregation> {
        private List<MultiTermLookup> terms;

        @Nullable
        private Integer size;

        /**
         * Required - API name: {@code terms}
         * <p>
         * Adds all elements of <code>list</code> to <code>terms</code>.
         */
        public final Builder terms(List<MultiTermLookup> list) {
            this.terms = _listAddAll(this.terms, list);
            return this;
        }

        /**
         * Required - API name: {@code terms}
         * <p>
         * Adds one or more values to <code>terms</code>.
         */
        public final Builder terms(MultiTermLookup value, MultiTermLookup... values) {
            this.terms = _listAdd(this.terms, value, values);
            return this;
        }

        /**
         * Required - API name: {@code terms}
         * <p>
         * Adds a value to <code>terms</code> using a builder lambda.
         */
        public final Builder terms(Function<MultiTermLookup.Builder, ObjectBuilder<MultiTermLookup>> fn) {
            return terms(fn.apply(new MultiTermLookup.Builder()).build());
        }

        /**
         * API name: {@code size}
         */
        public final Builder size(@Nullable Integer value) {
            this.size = value;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * Builds a {@link MultiTermsAggregation}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public MultiTermsAggregation build() {
            _checkSingleUse();

            return new MultiTermsAggregation(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link MultiTermsAggregation}
     */
    public static final JsonpDeserializer<MultiTermsAggregation> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        MultiTermsAggregation::setupMultiTermsAggregationDeserializer
    );

    protected static void setupMultiTermsAggregationDeserializer(ObjectDeserializer<MultiTermsAggregation.Builder> op) {
        BucketAggregationBase.setupBucketAggregationBaseDeserializer(op);
        op.add(Builder::size, JsonpDeserializer.integerDeserializer(), "size");
        op.add(Builder::terms, JsonpDeserializer.arrayDeserializer(MultiTermLookup._DESERIALIZER), "terms");
    }

}
