/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch._types.aggregations;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpUtils;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;

// typedef: _types.aggregations.TTestAggregate

@JsonpDeserializable
public class TTestAggregate extends AggregateBase implements AggregateVariant {
    private final double value;

    @Nullable
    private final String valueAsString;

    // ---------------------------------------------------------------------------------------------

    private TTestAggregate(Builder builder) {
        super(builder);

        this.value = ApiTypeHelper.requireNonNull(builder.value, this, "value");
        this.valueAsString = builder.valueAsString;

    }

    public static TTestAggregate of(Function<Builder, ObjectBuilder<TTestAggregate>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Aggregate variant kind.
     */
    @Override
    public Aggregate.Kind _aggregateKind() {
        return Aggregate.Kind.TTest;
    }

    /**
     * Required - API name: {@code value}
     * <p>
     * Defaults to {@code 0} if parsed from a JSON {@code null} value.
     */
    public final double value() {
        return this.value;
    }

    /**
     * API name: {@code value_as_string}
     */
    @Nullable
    public final String valueAsString() {
        return this.valueAsString;
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        super.serializeInternal(generator, mapper);
        generator.writeKey("value");
        JsonpUtils.serializeDoubleOrNull(generator, this.value, 0);
        if (this.valueAsString != null) {
            generator.writeKey("value_as_string");
            generator.write(this.valueAsString);

        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link TTestAggregate}.
     */

    public static class Builder extends AggregateBase.AbstractBuilder<Builder> implements ObjectBuilder<TTestAggregate> {
        private Double value;

        @Nullable
        private String valueAsString;

        /**
         * Required - API name: {@code value}
         * <p>
         * Defaults to {@code 0} if parsed from a JSON {@code null} value.
         */
        public final Builder value(double value) {
            this.value = value;
            return this;
        }

        /**
         * API name: {@code value_as_string}
         */
        public final Builder valueAsString(@Nullable String value) {
            this.valueAsString = value;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * Builds a {@link TTestAggregate}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public TTestAggregate build() {
            _checkSingleUse();

            return new TTestAggregate(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link TTestAggregate}
     */
    public static final JsonpDeserializer<TTestAggregate> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        TTestAggregate::setupTTestAggregateDeserializer
    );

    protected static void setupTTestAggregateDeserializer(ObjectDeserializer<TTestAggregate.Builder> op) {
        AggregateBase.setupAggregateBaseDeserializer(op);
        op.add(Builder::value, JsonpDeserializer.doubleOrNullDeserializer(0), "value");
        op.add(Builder::valueAsString, JsonpDeserializer.stringDeserializer(), "value_as_string");

    }

}
