/*
 * Copyright 2012-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * copy from org.springframework.boot.configurationprocessor.metadata.JsonMarshaller
 */

package org.noear.solon.configurationprocessor.metadata;

import org.noear.solon.configurationprocessor.json.JSONArray;
import org.noear.solon.configurationprocessor.json.JSONObject;
import org.noear.solon.core.util.IoUtil;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * Marshaller to read and write {@link ConfigurationMetadata} as JSON.
 *
 * @author Stephane Nicoll
 * @author Phillip Webb
 * @author Moritz Halbritter
 * @since 1.2.0
 */
public class JsonMarshaller {

    public void write(ConfigurationMetadata metadata, OutputStream outputStream) throws IOException {
        try {
            JSONObject object = new JSONObject();
            JsonConverter converter = new JsonConverter();
            object.put("groups", converter.toJsonArray(metadata, ItemMetadata.ItemType.GROUP));
            object.put("properties", converter.toJsonArray(metadata, ItemMetadata.ItemType.PROPERTY));
            object.put("hints", converter.toJsonArray(metadata.getHints()));
            outputStream.write(object.toString(2).getBytes(StandardCharsets.UTF_8));
        } catch (Exception ex) {
            if (ex instanceof IOException) {
                throw (IOException) ex;
            }
            if (ex instanceof RuntimeException) {
                throw (RuntimeException) ex;
            }
            throw new IllegalStateException(ex);
        }
    }

    public ConfigurationMetadata read(InputStream inputStream) throws Exception {
        ConfigurationMetadata metadata = new ConfigurationMetadata();
        String strContent = toString(inputStream);

        JSONObject object = new JSONObject(strContent);
        JsonPath path = JsonPath.root();
        checkAllowedKeys(object, path, "groups", "properties", "hints");
        JSONArray groups = object.optJSONArray("groups");
        if (groups != null) {
            for (int i = 0; i < groups.length(); i++) {
                metadata
                        .add(toItemMetadata((JSONObject) groups.get(i), path.resolve("groups").index(i), ItemMetadata.ItemType.GROUP));
            }
        }
        JSONArray properties = object.optJSONArray("properties");
        if (properties != null) {
            for (int i = 0; i < properties.length(); i++) {
                metadata.add(toItemMetadata((JSONObject) properties.get(i), path.resolve("properties").index(i),
                        ItemMetadata.ItemType.PROPERTY));
            }
        }
        JSONArray hints = object.optJSONArray("hints");
        if (hints != null) {
            for (int i = 0; i < hints.length(); i++) {
                metadata.add(toItemHint((JSONObject) hints.get(i), path.resolve("hints").index(i)));
            }
        }
        return metadata;
    }

    private ItemMetadata toItemMetadata(JSONObject object, JsonPath path, ItemMetadata.ItemType itemType) throws Exception {
        switch (itemType) {
            case GROUP:
                checkAllowedKeys(object, path, "name", "type", "description", "sourceType", "sourceMethod");
                break;
            case PROPERTY:
                checkAllowedKeys(object, path, "name", "type", "description", "sourceType", "defaultValue",
                        "deprecation", "deprecated");
                break;
        }
        String name = object.getString("name");
        String type = object.optString("type", null);
        String description = object.optString("description", null);
        String sourceType = object.optString("sourceType", null);
        String sourceMethod = object.optString("sourceMethod", null);
        Object defaultValue = readItemValue(object.opt("defaultValue"));
        ItemDeprecation deprecation = toItemDeprecation(object, path);
        return new ItemMetadata(itemType, name, null, type, sourceType, sourceMethod, description, defaultValue,
                deprecation);
    }

    private ItemDeprecation toItemDeprecation(JSONObject object, JsonPath path) throws Exception {
        if (object.has("deprecation")) {
            JSONObject deprecationJsonObject = object.getJSONObject("deprecation");
            checkAllowedKeys(deprecationJsonObject, path.resolve("deprecation"), "level", "reason", "replacement",
                    "since");
            ItemDeprecation deprecation = new ItemDeprecation();
            deprecation.setLevel(deprecationJsonObject.optString("level", null));
            deprecation.setReason(deprecationJsonObject.optString("reason", null));
            deprecation.setReplacement(deprecationJsonObject.optString("replacement", null));
            deprecation.setSince(deprecationJsonObject.optString("since", null));
            return deprecation;
        }
        return object.optBoolean("deprecated") ? new ItemDeprecation() : null;
    }

    private ItemHint toItemHint(JSONObject object, JsonPath path) throws Exception {
        checkAllowedKeys(object, path, "name", "values", "providers");
        String name = object.getString("name");
        List<ItemHint.ValueHint> values = new ArrayList<>();
        if (object.has("values")) {
            JSONArray valuesArray = object.getJSONArray("values");
            for (int i = 0; i < valuesArray.length(); i++) {
                values.add(toValueHint((JSONObject) valuesArray.get(i), path.resolve("values").index(i)));
            }
        }
        List<ItemHint.ValueProvider> providers = new ArrayList<>();
        if (object.has("providers")) {
            JSONArray providersObject = object.getJSONArray("providers");
            for (int i = 0; i < providersObject.length(); i++) {
                providers.add(toValueProvider((JSONObject) providersObject.get(i), path.resolve("providers").index(i)));
            }
        }
        return new ItemHint(name, values, providers);
    }

    private ItemHint.ValueHint toValueHint(JSONObject object, JsonPath path) throws Exception {
        checkAllowedKeys(object, path, "value", "description");
        Object value = readItemValue(object.get("value"));
        String description = object.optString("description", null);
        return new ItemHint.ValueHint(value, description);
    }

    private ItemHint.ValueProvider toValueProvider(JSONObject object, JsonPath path) throws Exception {
        checkAllowedKeys(object, path, "name", "parameters");
        String name = object.getString("name");
        Map<String, Object> parameters = new HashMap<>();
        if (object.has("parameters")) {
            JSONObject parametersObject = object.getJSONObject("parameters");
            for (Iterator<?> iterator = parametersObject.keys(); iterator.hasNext(); ) {
                String key = (String) iterator.next();
                Object value = readItemValue(parametersObject.get(key));
                parameters.put(key, value);
            }
        }
        return new ItemHint.ValueProvider(name, parameters);
    }

    private Object readItemValue(Object value) throws Exception {
        if (value instanceof JSONArray) {
            JSONArray array = (JSONArray) value;
            Object[] content = new Object[array.length()];
            for (int i = 0; i < array.length(); i++) {
                content[i] = array.get(i);
            }
            return content;
        }
        return value;
    }

    private String toString(InputStream inputStream) throws IOException {
        return IoUtil.transferToString(inputStream, StandardCharsets.UTF_8.toString());
    }

    @SuppressWarnings("unchecked")
    private void checkAllowedKeys(JSONObject object, JsonPath path, String... allowedKeys) {
        Set<String> availableKeys = new TreeSet<>();
        object.keys().forEachRemaining((key) -> availableKeys.add((String) key));
        Arrays.stream(allowedKeys).forEach(availableKeys::remove);
        if (!availableKeys.isEmpty()) {
            throw new IllegalStateException(String.format("Expected only keys %s, but found additional keys %s. Path: %s"
                    , new TreeSet<>(Arrays.asList(allowedKeys)), availableKeys, path));
        }
    }

    private static final class JsonPath {

        private final String path;

        private JsonPath(String path) {
            this.path = path;
        }

        static JsonPath root() {
            return new JsonPath(".");
        }

        JsonPath resolve(String path) {
            if (this.path.endsWith(".")) {
                return new JsonPath(this.path + path);
            }
            return new JsonPath(this.path + "." + path);
        }

        JsonPath index(int index) {
            return resolve(String.format("[%d]", index));
        }

        @Override
        public String toString() {
            return this.path;
        }

    }

}
