/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.connector.socket;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ByteChannel;
import java.nio.channels.SocketChannel;
import java.security.GeneralSecurityException;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLSession;
import org.neo4j.driver.Config;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.internal.connector.socket.SSLContextFactory;
import org.neo4j.driver.internal.spi.Logger;
import org.neo4j.driver.internal.util.BytePrinter;

public class SSLSocketChannel
implements ByteChannel {
    private final SocketChannel channel;
    private final Logger logger;
    private final SSLContext sslContext;
    private SSLEngine sslEngine;
    private ByteBuffer cipherOut;
    private ByteBuffer cipherIn;
    private ByteBuffer plainIn;
    private ByteBuffer plainOut;

    public SSLSocketChannel(String host, int port, SocketChannel channel, Logger logger, Config.TlsAuthenticationConfig authConfig) throws GeneralSecurityException, IOException {
        logger.debug("TLS connection enabled", new Object[0]);
        this.logger = logger;
        this.channel = channel;
        this.channel.configureBlocking(true);
        this.sslContext = new SSLContextFactory(host, port, authConfig).create();
        this.createSSLEngine(host, port);
        this.createBuffers();
        this.runSSLHandShake();
        logger.debug("TLS connection established", new Object[0]);
    }

    SSLSocketChannel(SocketChannel channel, Logger logger, SSLEngine sslEngine, ByteBuffer plainIn, ByteBuffer cipherIn, ByteBuffer plainOut, ByteBuffer cipherOut) throws GeneralSecurityException, IOException {
        logger.debug("Testing TLS buffers", new Object[0]);
        this.logger = logger;
        this.channel = channel;
        this.sslContext = SSLContext.getInstance("TLS");
        this.sslEngine = sslEngine;
        this.resetBuffers(plainIn, cipherIn, plainOut, cipherOut);
    }

    private void runSSLHandShake() throws IOException {
        this.sslEngine.beginHandshake();
        SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
        while (handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED && handshakeStatus != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
            switch (handshakeStatus) {
                case NEED_TASK: {
                    handshakeStatus = this.runDelegatedTasks();
                    break;
                }
                case NEED_UNWRAP: {
                    handshakeStatus = this.unwrap(null);
                    this.plainIn.clear();
                    break;
                }
                case NEED_WRAP: {
                    handshakeStatus = this.wrap(this.plainOut);
                }
            }
        }
        this.plainIn.clear();
        this.plainOut.clear();
    }

    private SSLEngineResult.HandshakeStatus runDelegatedTasks() {
        Runnable runnable;
        while ((runnable = this.sslEngine.getDelegatedTask()) != null) {
            runnable.run();
        }
        return this.sslEngine.getHandshakeStatus();
    }

    private SSLEngineResult.HandshakeStatus unwrap(ByteBuffer buffer) throws IOException {
        SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
        if (this.channel.read(this.cipherIn) < 0) {
            throw new ClientException("Failed to establish SSL socket connection.");
        }
        this.cipherIn.flip();
        SSLEngineResult.Status status = null;
        do {
            status = this.sslEngine.unwrap(this.cipherIn, this.plainIn).getStatus();
            switch (status) {
                case OK: {
                    this.plainIn.flip();
                    SSLSocketChannel.bufferCopy(this.plainIn, buffer);
                    this.plainIn.compact();
                    handshakeStatus = this.runDelegatedTasks();
                    break;
                }
                case BUFFER_OVERFLOW: {
                    this.plainIn.flip();
                    int curAppSize = this.plainIn.capacity();
                    int appSize = this.sslEngine.getSession().getApplicationBufferSize();
                    int newAppSize = appSize + this.plainIn.remaining();
                    if (newAppSize > appSize * 2) {
                        throw new ClientException(String.format("Failed ro enlarge application input buffer from %s to %s, as the maximum buffer size allowed is %s. The content in the buffer is: %s\n", curAppSize, newAppSize, appSize * 2, BytePrinter.hex(this.plainIn)));
                    }
                    ByteBuffer newPlainIn = ByteBuffer.allocateDirect(newAppSize);
                    newPlainIn.put(this.plainIn);
                    this.plainIn = newPlainIn;
                    this.logger.debug("Enlarged application input buffer from %s to %s. This operation should be a rare operation.", curAppSize, newAppSize);
                    break;
                }
                case BUFFER_UNDERFLOW: {
                    int curNetSize = this.cipherIn.capacity();
                    int netSize = this.sslEngine.getSession().getPacketBufferSize();
                    if (netSize > curNetSize) {
                        ByteBuffer newCipherIn = ByteBuffer.allocateDirect(netSize);
                        newCipherIn.put(this.cipherIn);
                        this.cipherIn = newCipherIn;
                        this.logger.debug("Enlarged network input buffer from %s to %s. This operation should be a rare operation.", curNetSize, netSize);
                    } else {
                        this.cipherIn.compact();
                    }
                    return handshakeStatus;
                }
                default: {
                    throw new ClientException("Got unexpected status " + (Object)((Object)status));
                }
            }
        } while (this.cipherIn.hasRemaining());
        this.cipherIn.compact();
        return handshakeStatus;
    }

    private SSLEngineResult.HandshakeStatus wrap(ByteBuffer buffer) throws IOException {
        SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
        SSLEngineResult.Status status = this.sslEngine.wrap(buffer, this.cipherOut).getStatus();
        switch (status) {
            case OK: {
                handshakeStatus = this.runDelegatedTasks();
                this.cipherOut.flip();
                this.channel.write(this.cipherOut);
                this.cipherOut.clear();
                break;
            }
            case BUFFER_OVERFLOW: {
                int curNetSize = this.cipherOut.capacity();
                int netSize = this.sslEngine.getSession().getPacketBufferSize();
                if (curNetSize >= netSize || buffer.capacity() > netSize) {
                    throw new ClientException(String.format("Failed to enlarge network buffer from %s to %s. This is either because the new size is however less than the old size, or because the application buffer size %s is so big that the application data still cannot fit into the new network buffer.", curNetSize, netSize, buffer.capacity()));
                }
                this.cipherOut = ByteBuffer.allocateDirect(netSize);
                this.logger.debug("Enlarged network output buffer from %s to %s. This operation should be a rare operation.", curNetSize, netSize);
                break;
            }
            default: {
                throw new ClientException("Got unexpected status " + (Object)((Object)status));
            }
        }
        return handshakeStatus;
    }

    static int bufferCopy(ByteBuffer from, ByteBuffer to) {
        if (from == null || to == null) {
            return 0;
        }
        int i = 0;
        while (to.remaining() > 0 && from.remaining() > 0) {
            to.put(from.get());
            ++i;
        }
        return i;
    }

    private void createBuffers() throws IOException {
        SSLSession session = this.sslEngine.getSession();
        int appBufferSize = session.getApplicationBufferSize();
        int netBufferSize = session.getPacketBufferSize();
        this.plainOut = ByteBuffer.allocateDirect(appBufferSize);
        this.plainIn = ByteBuffer.allocateDirect(appBufferSize);
        this.cipherOut = ByteBuffer.allocateDirect(netBufferSize);
        this.cipherIn = ByteBuffer.allocateDirect(netBufferSize);
    }

    void resetBuffers(ByteBuffer plainIn, ByteBuffer cipherIn, ByteBuffer plainOut, ByteBuffer cipherOut) {
        this.plainIn = plainIn;
        this.cipherIn = cipherIn;
        this.plainOut = plainOut;
        this.cipherOut = cipherOut;
    }

    private void createSSLEngine(String host, int port) {
        this.sslEngine = this.sslContext.createSSLEngine(host, port);
        this.sslEngine.setUseClientMode(true);
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        int toRead = dst.remaining();
        this.plainIn.flip();
        if (this.plainIn.remaining() >= toRead) {
            SSLSocketChannel.bufferCopy(this.plainIn, dst);
            this.plainIn.compact();
        } else {
            dst.put(this.plainIn);
            do {
                this.plainIn.clear();
                this.unwrap(dst);
            } while (dst.remaining() > 0);
        }
        return toRead;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        int toWrite = src.remaining();
        while (src.remaining() > 0) {
            this.wrap(src);
        }
        return toWrite;
    }

    @Override
    public boolean isOpen() {
        return this.channel.isOpen();
    }

    @Override
    public void close() throws IOException {
        this.plainOut.clear();
        this.sslEngine.closeOutbound();
        while (!this.sslEngine.isOutboundDone()) {
            int num;
            SSLEngineResult res = this.sslEngine.wrap(this.plainOut, this.cipherOut);
            while (this.cipherOut.hasRemaining() && (num = this.channel.write(this.cipherOut)) != -1) {
            }
        }
        this.channel.close();
        this.logger.debug("TLS connection closed", new Object[0]);
    }
}

