package org.mule.weave.v2.module.xmlschema

import org.mule.apache.xerces.dom.DOMInputImpl
import org.mule.apache.xerces.impl.XMLEntityManager.expandSystemId
import org.mule.apache.xerces.impl.xs.util.SimpleLocator
import org.mule.apache.xerces.xni.XMLResourceIdentifier
import org.mule.apache.xerces.xni.parser.XMLEntityResolver
import org.mule.apache.xerces.xni.parser.XMLInputSource
import org.mule.apache.xerces.xni.parser.XMLParseException
import org.mule.weave.v2.module.xmlschema.StringUtils.isEmpty
import org.mule.weave.v2.module.xmlschema.StringUtils.isNotEmpty
import org.mule.weave.v2.module.xmlschema.utils.SchemaHelper
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.w3c.dom.ls.LSInput
import org.w3c.dom.ls.LSResourceResolver

import java.io.ByteArrayInputStream
import java.io.InputStream
import java.net.URL
import java.util
import scala.collection.mutable

object ResourceResolver {
  val NO_NAMESPACE = "NO_NAMESPACE"
}

class EntityResolverWrapper(var entityResolver: XMLEntityResolver) extends XMLEntityResolver {
  private val IMPORT_REMOTE_ERROR = "Xml schema location %s could not be resolved"

  override def resolveEntity(xmlResourceIdentifier: XMLResourceIdentifier): XMLInputSource = {
    val url = new URL(xmlResourceIdentifier.getExpandedSystemId)
    val source = entityResolver.resolveEntity(xmlResourceIdentifier)
    if (source == null && !url.getProtocol.equals("file")) {
      throw new XMLParseException(new SimpleLocator(xmlResourceIdentifier.getLiteralSystemId, xmlResourceIdentifier.getExpandedSystemId, 0, 0), String.format(IMPORT_REMOTE_ERROR, xmlResourceIdentifier.getExpandedSystemId))
    }
    source
  }
}

class ResourceResolver(var schemaByTargetNamespace: util.Map[String, util.List[DOMInputImpl]], weaveResourceResolver: WeaveResourceResolver, isDisallowDocTypeDecl: Boolean) extends LSResourceResolver {

  override def resolveResource(`type`: String, namespaceURI: String, publicId: String, systemId: String, baseURI: String): LSInput = {
    val resolvedURI = getAbsoluteUri(systemId, baseURI)
    val path = new URL(resolvedURI).getPath
    if (isNotEmpty(namespaceURI) && schemaByTargetNamespace.get(namespaceURI) != null) {
      resolveResourceForNamespace(schemaByTargetNamespace.get(namespaceURI), resolvedURI)
    } else if (isEmpty(namespaceURI) && isNotEmpty(systemId)) {
      resolveResourceForNamespace(schemaByTargetNamespace.get(ResourceResolver.NO_NAMESPACE), resolvedURI)
    } else {
      val resolvedResource = weaveResourceResolver.resolve(NameIdentifierHelper.fromWeaveFilePath(path))
      resolvedResource.map(resource => {
        val keyValueMap = mutable.Map[String, InputStream]()
        val content = resource.content()
        val stream = new ByteArrayInputStream(content.getBytes)
        keyValueMap.put(resolvedURI, stream)
        val stringToImpls = SchemaHelper.getSchemasByTargetNamespace(keyValueMap, isDisallowDocTypeDecl, Some(this))
        schemaByTargetNamespace.putAll(stringToImpls)
        resolveResourceForNamespace(schemaByTargetNamespace.get(namespaceURI), resolvedURI)
      }).orNull
    }
  }

  private def getAbsoluteUri(systemId: String, baseURI: String) = try if (baseURI == null) systemId
  else expandSystemId(systemId, baseURI, false)
  catch {
    case e: Exception =>
      null
  }

  private def resolveResourceForNamespace(domInputs: util.List[DOMInputImpl], systemId: String): LSInput = {
    if (domInputs != null) {
      domInputs.forEach(input => {
        if (systemId == null || systemId == input.getSystemId) return input
      })
    }
    null
  }
}