package org.mule.weave.v2.runtime

import org.mule.weave.v2.core.util.TypeAliases
import org.mule.weave.v2.core.exception.UnknownContentTypeException
import org.mule.weave.v2.core.exception.UnknownDataFormatException
import org.mule.weave.v2.interpreted.listener.WeaveExecutionListener
import org.mule.weave.v2.interpreted.profiler.ExecutionTelemetryListener
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.option.Settings
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.annotation.InjectedNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.DirectivesCapableNode
import org.mule.weave.v2.parser.ast.header.directives.ContentType
import org.mule.weave.v2.parser.ast.header.directives.DataFormatId
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.header.directives.OutputDirective
import org.mule.weave.v2.parser.module.InvalidMimeTypeExpression
import org.mule.weave.v2.parser.module.MimeType

import scala.collection.mutable

/**
  * Represents an executable weave script. It can be either a mapping or an executable module
  */
trait ExecutableWeave[T <: DirectivesCapableNode] {

  private var _outputDataFormatMimeType: Option[String] = _
  private var _outputMimeType: Option[String] = _
  private var _outputDirective: Option[OutputDirective] = _
  //This is just to store some runtime properties
  private val _properties: mutable.Map[String, Any] = new mutable.HashMap()
  private val _telemetryListener = new ExecutionTelemetryListener

  private var telemetry: Boolean = false

  private def getDataFormatByMime(mime: ContentType)(implicit ctx: EvaluationContext): DataFormat[_, _] = {
    DataFormatManager.byContentType(mime.mime).getOrElse(throw new UnknownContentTypeException(mime.location(), mime.mime))
  }

  private def getDataFormatById(format: DataFormatId)(implicit ctx: EvaluationContext): DataFormat[_, _] = {
    DataFormatManager.byName(format.id).getOrElse(throw new UnknownDataFormatException(format.location(), format.id))
  }

  private def getDataFormat(directive: InputDirective)(implicit ctx: EvaluationContext): DataFormat[Settings, Settings] = {
    val dataFormat = directive.dataFormat match {
      case Some(formatId) => getDataFormatById(formatId)
      case None =>
        directive.mime match {
          case Some(mime) => getDataFormatByMime(mime)
          case None       => throw new IllegalStateException("Missing input format definition.")
        }
    }
    dataFormat.asInstanceOf[DataFormat[Settings, Settings]]
  }

  /**
    * Returns the list of declared inputs in the document
    *
    * @return A map with the readers
    */
  def declaredInputs()(implicit ctx: EvaluationContext): Map[String, DataFormat[_, _]] = {
    val inputDirectives = astDocument().directives
      .collect({
        case id: InputDirective if id.annotation(classOf[InjectedNodeAnnotation]).isEmpty && (id.dataFormat.nonEmpty || id.mime.nonEmpty) => id
      })
    inputDirectives
      .map(directive => (directive.variable.name, getDataFormat(directive)))
      .toMap[String, DataFormat[_, _]]
  }

  /**
    * The output mime type specified by the user
    *
    * @return
    */
  def declaredOutputMimeType: Option[String] = {
    outputDirective.flatMap((od) => {
      od.mime match {
        case Some(mime) => Some(mime.mime)
        case None       => None
      }
    })
  }

  def getProperty[PropType](name: String, callback: => PropType): PropType = {
    _properties.getOrElseUpdate(name, callback).asInstanceOf[PropType]
  }

  /**
    * The output directive in the AST
    * @return
    */
  def outputDirective: Option[OutputDirective] = {
    if (_outputDirective == null) {
      _outputDirective = AstNodeHelper.getOutputDirective(astDocument())
    }
    _outputDirective
  }

  /**
    * The MimeType to be used to lookup the Output Data Format
    * @param ctx The evaluation context
    * @return The MimeType
    */
  def outputDataFormatMimeType(implicit ctx: EvaluationContext): Option[String] = {
    if (_outputDataFormatMimeType == null) {
      _outputDataFormatMimeType = outputDirective
        .flatMap((od) => {
          val dataFormat: Option[DataFormatId] = od.dataFormat
          dataFormat match {
            case Some(id) => {
              val maybeFormat = DataFormatManager.byName(id.id)
              maybeFormat
                .map(_.defaultMimeType.toStringWithoutParameters)
                .orElse({
                  throw new UnknownDataFormatException(od.location(), id.id)
                })
            }
            case None => None
          }
        })
        .orElse(outputDirective
          .flatMap((od) => {
            od.mime match {
              case Some(mime) => Some(mime.mime)
              case None       => None
            }
          }))
    }

    _outputDataFormatMimeType
  }

  /**
    * The MimeType to be used as the mimeType of the result
    * @param ctx The evaluation context
    * @return The MimeType
    */
  def outputMimeType(implicit ctx: EvaluationContext): Option[String] = {
    if (_outputMimeType == null) {
      _outputMimeType = outputDirective
        .flatMap((od) => {
          od.mime match {
            case Some(mime) => Some(mime.mime)
            case None       => None
          }
        })
        .orElse(outputDirective
          .flatMap((od) => {
            od.dataFormat match {
              case Some(id) => {
                val maybeFormat = DataFormatManager.byName(id.id)
                maybeFormat
                  .map(_.defaultMimeType.toStringWithoutParameters)
                  .orElse({
                    throw new UnknownDataFormatException(od.location(), id.id)
                  })
              }
              case None => None
            }
          }))
    }
    _outputMimeType
  }

  def enableTelemetry(): Unit = {
    if (!telemetry) {
      addExecutionListener(_telemetryListener)
    }
    telemetry = true
  }

  def disableTelemetry(): Unit = {
    if (!telemetry) {
      removeExecutionListener(_telemetryListener)
    }
    telemetry = false
  }

  /**
    * Specifies the max amount of time an execution can take
    *
    * @param maxExecutionTime The max time in millis
    */
  def withMaxTime(maxExecutionTime: Long): ExecutableWeave[T]

  /**
    * Adds an execution listener
    *
    * @param listener The execution listener
    */
  def addExecutionListener(listener: WeaveExecutionListener): Unit

  def removeExecutionListener(listener: WeaveExecutionListener): Unit

  /**
    * Marks that all values needs to be materialized
    *
    * @param materialize True for all values to be materialized
    */
  def materializedValuesExecution(materialize: Boolean): Unit

  /**
    * Returns the declared output module if any
    *
    * @return The writer if declared
    */
  def declaredOutput()(implicit ctx: EvaluationContext): Option[DataFormat[_, _]] = {
    outputDataFormatMimeType.map((mimeType) => {
      var maybeFormat: Option[DataFormat[_, _]] = None
      try {
        maybeFormat = DataFormatManager.byContentType(mimeType)
      } catch {
        case e: InvalidMimeTypeExpression => {
          val maybeDirective = outputDirective.flatMap(_.mime)
          throw new InvalidMimeTypeExpression(mimeType, maybeDirective.getOrElse(UnknownLocationCapable).location())
        }
      }
      maybeFormat
        .getOrElse({
          val maybeDirective = outputDirective.flatMap(_.mime)
          throw new UnknownContentTypeException(maybeDirective.getOrElse(UnknownLocationCapable).location(), mimeType)
        })

    })
  }

  /**
    * Returns the underlying ast document node
    *
    * @return the document node
    */
  def astDocument(): T

  /**
    * Executes with the given inputs.
    * The context is not going to be closed so is up to the caller to close it when the value is no longer needed
    *
    * @param values  The values to be populated in the context
    * @param readers The readers to populated in the context
    * @param ctx     The evaluation context
    * @return
    */
  def execute(readers: Map[String, Reader] = Map.empty, values: Map[String, Value[_]] = Map.empty)(implicit ctx: EvaluationContext): Value[_]

  /**
    * Writes the result of the execution. The writer should be configured and ready to use.
    *
    * @param writer          The writer to be used
    * @param values          The values to be populated in the context
    * @param readers         The readers to populated in the context
    * @param ctx             The evaluation context
    * @param closeAfterWrite If the write method should close the context once it finish
    * @return
    */
  def writeWith(writer: Writer, readers: Map[String, Reader] = Map.empty, values: Map[String, Value[_]] = Map.empty, closeAfterWrite: Boolean = true)(implicit ctx: EvaluationContext): (Any, TypeAliases#JCharset)

  /**
    * Writes the result of the execution using the writer specified by the weave file.
    * @param readers The readers that are going to be bind to values
    * @param values Additional values to bind to the execution
    * @param closeAfterWrite If it should close the context when it finish
    * @param ctx The context
    * @return The result of the execution
    */
  def write(readers: Map[String, Reader] = Map.empty, values: Map[String, Value[_]] = Map.empty, closeAfterWrite: Boolean = true)(implicit ctx: EvaluationContext): (Any, TypeAliases#JCharset) = {
    val writer = createImplicitWriter()
    writeWith(writer, readers, values, closeAfterWrite)
  }

  /**
    * Configures the given writer with the writer properties specified on this script
    *
    * @param writer The writer to be configured
    * @return The configured writer
    */
  def configureWriter(writer: Writer)(implicit ctx: EvaluationContext): Writer = writer

  /**
    * Helper method that creates the implicit writer or fails if not defined
    *
    * @return The writer
    */
  def createImplicitWriter()(implicit ctx: EvaluationContext): Writer = {
    implicitWriterOption()
      .getOrElse(throw new RuntimeException("Unable to find an output directive out."))
  }

  def implicitWriterOption(target: Option[Any] = None)(implicit ctx: EvaluationContext) = {
    declaredOutput()
      .map((module) => {
        val mimeType = declaredOutputMimeType
        val writer = mimeType match {
          case Some(declaredMimeType) => {
            module.writer(target, MimeType.fromSimpleString(declaredMimeType))
          }
          case None => module.writer(target)
        }
        configureWriter(writer)
        writer
      })
  }
}
