package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.core.functions.BaseUnaryFunctionValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.UnexpectedFunctionCallTypesException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.findMatchingFunctionWithCoercion
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValuesHelper
import org.mule.weave.v2.parser.location.WeaveLocation

import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicReference

class UnaryFunctionExecutor(
  override val node: ValueNode[_],
  val name: String,
  val firstArgConstantType: Boolean,
  override val showInStacktrace: Boolean = false,
  override val location: WeaveLocation) extends UnaryExecutor with Product3[ValueNode[_], String, Boolean] {

  private val validated: AtomicBoolean = new AtomicBoolean(false)

  private val cachedCoercedDispatch: AtomicReference[Type] = new AtomicReference()

  override def executeUnary(fv: Value[_])(implicit ctx: ExecutionContext): Value[Any] = {
    val activeFrame: Frame = ctx.executionStack().activeFrame()
    try {
      activeFrame.updateCallSite(node)
      val functionValue = target().asInstanceOf[BaseUnaryFunctionValue]
      //This is the fast path
      if (validated.get()) {
        //If values are literal we do not need to validate every just the first time when we load the operation
        //Then is ok every time as its type will NEVER change

        val firstValue = if (!firstArgConstantType && functionValue.rightParam.typeRequiresMaterialization) {
          fv.materialize
        } else {
          fv
        }
        if ((firstArgConstantType || functionValue.R.accepts(firstValue))) {
          return doCall(functionValue, firstValue)
        }
      }

      val firstValue = if (functionValue.rightParam.typeRequiresMaterialization) {
        fv.materialize
      } else {
        fv
      }

      if (cachedCoercedDispatch.get() != null) {
        val coercedOperation = cachedCoercedDispatch.get
        //If values are literal we do not need to validate every just the first time when we load the coerced operation
        //Then is ok every time as its type will NEVER change
        if (firstArgConstantType || coercedOperation.accepts(firstValue)) {
          val valueMaybe = functionValue.R.coerceMaybe(firstValue)
          if (valueMaybe.isDefined) {
            return functionValue.call(valueMaybe.get)
          }
        }
      }

      if (functionValue.R.accepts(firstValue)) {
        validated.set(true)
        doCall(functionValue, firstValue)
      } else {
        //VERY SLOW PATH
        val materializedValues: Array[Value[Any]] = ValuesHelper.array(firstValue.materialize)
        val functionToCallWithCoercion: Option[(Int, Array[Value[_]], Seq[Int])] = findMatchingFunctionWithCoercion(materializedValues, Array(functionValue), this)
        functionToCallWithCoercion match {
          case Some((_, argumentsWithCoercion, paramsToCoerce)) => {
            //Cache the coercion use the base type to avoid Memory Leaks as Types may have references to Streams or Objects
            cachedCoercedDispatch.set(firstValue.valueType.baseType)
            val firstCoercedValue: Value[_] = argumentsWithCoercion(0)
            doCall(functionValue, firstCoercedValue)
          }
          case None =>
            throw new UnexpectedFunctionCallTypesException(location, name, materializedValues, Seq(functionValue.functionParamTypes.map(_.theType)))
        }
      }
    } finally {
      activeFrame.cleanCallSite()
    }
  }

  private def doCall(functionValue: BaseUnaryFunctionValue, firstValue: Value[_])(implicit ctx: ExecutionContext) = {
    try {
      functionValue.call(firstValue)
    } catch {
      case ex: ExecutionException =>
        if (showInStacktrace) {
          ex.addCallToStacktrace(location, name())
        }
        throw ex
    }
  }

  def target()(implicit ctx: ExecutionContext): FunctionValue = {
    node.execute.asInstanceOf[FunctionValue]
  }

  override def _1: ValueNode[_] = node

  override def _2: String = name

  override def _3: Boolean = firstArgConstantType

  override def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any] = {
    executeUnary(arguments(0))
  }

  override def name()(implicit ctx: ExecutionContext): String = {
    this.name
  }

}
