package org.mule.weave.v2.module.pojo.reader

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.java.reflection.ReflectionUtils
import org.mule.weave.v2.module.pojo.BeanIntrospectionService
import org.mule.weave.v2.module.pojo.exception.IllegalReadPropertyAccessException
import org.mule.weave.v2.module.pojo.exception.IllegalWritePropertyAccessException
import org.mule.weave.v2.module.pojo.exception.InvalidPropertyNameException
import org.mule.weave.v2.module.pojo.exception.JavaPropertyAccessException
import org.mule.weave.v2.module.pojo.exception.JavaPropertyWriteException
import org.mule.weave.v2.parser.location.LocationCapable

import java.lang.reflect.AccessibleObject
import java.lang.reflect.Field
import java.lang.reflect.InvocationTargetException
import java.lang.reflect.Method
import java.lang.reflect.Type

class PropertyDefinition(val name: String, beanIntrospectionService: BeanIntrospectionService, maybeReadMethod: Option[Method], honourBeanDefinitionAccessor: Boolean) {

  private val baseName: String = NameGenerator.capitalizeJavaConvention(name)
  private lazy val clazz = beanIntrospectionService.underlyingClass

  @volatile
  private var _readInitialized: Boolean = false
  private var _readMethod: Option[Method] = _

  @volatile
  private var _writeInitialized: Boolean = false
  private var _writeMethod: Option[Method] = _

  @volatile
  private var _fieldInitialized: Boolean = false
  private var _field: Option[Field] = _

  @volatile
  private var _classTypeInitialized: Boolean = false
  private var _classType: Class[_] = _

  def read(instance: Any, location: LocationCapable)(implicit ctx: EvaluationContext): Any = {
    try {
      readMethod match {
        case Some(method) =>
          method.invoke(instance)
        case _ =>
          field match {
            case Some(f) =>
              f.get(instance)
            case _ =>
              throw new InvalidPropertyNameException(location.location(), name, clazz)
          }
      }
    } catch {
      case e: InvalidPropertyNameException =>
        throw e
      case e @ (_: IllegalAccessException | _: IllegalArgumentException) =>
        val cause = e.getCause
        throw new IllegalReadPropertyAccessException(name, clazz.getName, location, if (cause != null) cause else e)
      case ite: InvocationTargetException =>
        throw new JavaPropertyAccessException(name, clazz.getName, location, ite.getTargetException)
      case e =>
        val cause = e.getCause
        throw new JavaPropertyAccessException(name, clazz.getName, location, if (cause != null) cause else e)
    }
  }

  private def readMethod(implicit ctx: EvaluationContext): Option[Method] = {
    if (!_readInitialized) {
      synchronized {
        if (!_readInitialized) {
          _readMethod = trySetAccessible(maybeReadMethod)
          _readInitialized = true
        }
      }
    }
    _readMethod
  }

  private def field(implicit ctx: EvaluationContext): Option[Field] = {
    if (!_fieldInitialized) {
      synchronized {
        if (!_fieldInitialized) {
          val maybeField = if (honourBeanDefinitionAccessor) {
            None
          } else {
            beanIntrospectionService.getDeclaredFieldFromHierarchy(name)
          }
          _field = trySetAccessible(maybeField)
          _fieldInitialized = true
        }
      }
    }
    _field
  }

  def write(instance: Any, value: AnyRef, location: LocationCapable)(implicit ctx: EvaluationContext): Boolean = {
    try {
      writeMethod match {
        case Some(method) =>
          method.invoke(instance, value)
          true
        case _ =>
          field match {
            case Some(field) =>
              field.set(instance, value)
              true
            case _ =>
              false
          }
      }
    } catch {
      case e @ (_: IllegalAccessException | _: IllegalArgumentException) =>
        val cause = e.getCause
        throw new IllegalWritePropertyAccessException(name, clazz.getName, location, if (cause != null) cause else e)
      case ite: InvocationTargetException =>
        throw new JavaPropertyWriteException(name, clazz.getName, location, ite.getTargetException)
      case e: Exception =>
        val cause = e.getCause
        throw new JavaPropertyWriteException(name, clazz.getName, location, if (cause != null) cause else e)
    }
  }

  private def writeMethod(implicit ctx: EvaluationContext): Option[Method] = {
    if (!_writeInitialized) {
      synchronized {
        if (!_writeInitialized) {
          val methodName = BeanConstants.SET_PREFIX + baseName
          val maybeMethod = beanIntrospectionService.findMethod(m => m.getName.equals(methodName) && m.getParameterTypes.length == 1)
          _writeMethod = trySetAccessible(maybeMethod)
          _writeInitialized = true
        }
      }
    }
    _writeMethod
  }

  def genericType()(implicit ctx: EvaluationContext): Option[Type] = {
    readMethod.map(method => method.getGenericReturnType)
  }

  def classType()(implicit ctx: EvaluationContext): Class[_] = {
    resolveClassType
  }

  private def resolveClassType(implicit ctx: EvaluationContext): Class[_] = {
    if (!_classTypeInitialized) {
      synchronized {
        if (!_classTypeInitialized) {
          _classType = doResolveClassType
          _classTypeInitialized = true
        }
      }
    }
    _classType
  }

  private def doResolveClassType(implicit ctx: EvaluationContext): Class[_] = {
    writeMethod match {
      case Some(m) =>
        m.getParameterTypes.apply(0)
      case _ =>
        readMethod match {
          case Some(m) =>
            m.getReturnType
          case _ =>
            field match {
              case Some(field) =>
                field.getType
              case _ =>
                throw new IllegalStateException("No read or write handler for " + name)
            }
        }
    }
  }

  private def trySetAccessible[T <: AccessibleObject](accessibleObject: Option[T])(implicit ctx: EvaluationContext): Option[T] = {
    if (accessibleObject.isDefined && !ctx.serviceManager.settingsService.java().disableSetAccessible) {
      ReflectionUtils.safeTrySetAccessible(accessibleObject.get)
    }
    accessibleObject
  }
}

object PropertyDefinition {

  def apply(name: String, beanIntrospectionService: BeanIntrospectionService, maybeReadMethod: Option[Method], honourBeanDefinitionAccessor: Boolean): PropertyDefinition = {
    new PropertyDefinition(name, beanIntrospectionService, maybeReadMethod, honourBeanDefinitionAccessor)
  }
}
