package org.mule.weave.v2.module.dwb.reader.indexed

import org.mule.weave.v2.dwb.api.IWeaveValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.EagerObjectSeq
import org.mule.weave.v2.model.structure.IndexedObjectSeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.SimpleObjectSeq
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.reader.RedefinedValueRetriever
import org.mule.weave.v2.module.core.xml.reader.indexed.FilteredIndexedIterator
import org.mule.weave.v2.module.core.xml.reader.indexed.TokenHelpers.LCEntry
import org.mule.weave.v2.module.core.xml.reader.indexed.TokenHelpers._
import org.mule.weave.v2.parser.location.LocationCapable

import java.util.{ Map => JMap }

class WeaveBinaryObjectSeq(depth: Int, firstLcIndex: Long, lastLcIndex: Long, input: BinaryParserInput) extends EagerObjectSeq with SimpleObjectSeq with IndexedObjectSeq {
  private lazy val childrenLC = input.locationCaches(depth + 1)

  override def keyValueOfWithIndex(key: Value[QualifiedName])(implicit ctx: EvaluationContext): IndexedSelection[KeyValuePair] = {
    val iterator = createFilteredIterator(key)
    if (iterator.isEmpty) {
      null
    } else {
      val keyValuePair: KeyValuePair = iterator.next()
      val index = iterator.elementIndex()
      IndexedSelection(index, keyValuePair)
    }
  }

  override def allKeyValuesOf(key: Value[QualifiedName])(implicit ctx: EvaluationContext): Option[ObjectSeq] = {
    val objectSeq = filter(key)
    if (objectSeq.isEmpty()) {
      None
    } else {
      Some(objectSeq)
    }
  }

  override def size()(implicit ctx: EvaluationContext): Long = {
    lastLcIndex - (firstLcIndex - 1)
  }

  override def isEmpty()(implicit ctx: EvaluationContext): Boolean = {
    size() == 0
  }

  private def filter(key: Value[QualifiedName])(implicit ctx: EvaluationContext): ObjectSeq = {
    ObjectSeq(createFilteredIterator(key), materializedValues = true)
  }

  private def createFilteredIterator(k: Value[QualifiedName])(implicit ctx: EvaluationContext): FilteredIndexedIterator = {
    new FilteredIndexedIterator(k, input.tokenArray, childrenLC, firstLcIndex, lastLcIndex) {
      override def getTokenIndex(lcEntry: LCEntry): Long = lcEntry.getTokenIndex

      override def getTokenHash(token: Array[Long]): Int = DwTokenHelper.getNameHash(token)

      override def getKeyLength(token: Array[Long]): Long = DwTokenHelper.getKeyLength(token)

      override def getKvpValue(index: Long): KeyValuePair = WeaveBinaryObjectSeq.this.getKvpValue(index)

      override def hashCodeCalculator: String => Int = DwTokenHelper.hash
    }
  }

  private def getKvpValue(lcIndex: Long)(implicit ctx: EvaluationContext): KeyValuePair = {
    val entry = childrenLC(lcIndex)
    val keyTokenIndex = entry.getTokenIndex
    val keyToken = input.tokenArray(keyTokenIndex)
    val key = BinaryValueRetriever.readKey(keyTokenIndex, keyToken, input)
    val attrsCount = key.attributes.map(_.evaluate.size()).getOrElse(0)
    val attrTokensToSkip = attrsCount * 2
    val value = WeaveBinaryValue.apply(keyTokenIndex + attrTokensToSkip + 1, Some(lcIndex), input)
    KeyValuePair(key, value)
  }

  override def selectValueWithIndex(key: Value[QualifiedName])(implicit ctx: EvaluationContext): IndexedSelection[Value[_]] = {
    val iterator = createFilteredIterator(key)
    if (iterator.isEmpty) {
      null
    } else {
      val keyValuePair: KeyValuePair = iterator.next()
      val index = iterator.elementIndex()
      IndexedSelection(index, keyValuePair._2)
    }
  }
  override def apply(index: Long)(implicit ctx: EvaluationContext): KeyValuePair = {
    if (index >= 0 && index < size()) {
      getKvpValue(firstLcIndex + index)
    } else {
      null
    }
  }
}

class RedefinedWeaveBinaryObjectSeq(depth: Int, firstLcIndex: Long, lastLcIndex: Long, input: BinaryParserInput, processorClass: String, schema: JMap[String, IWeaveValue[_]], locationCapable: LocationCapable, ctx: EvaluationContext)
    extends WeaveBinaryObjectSeq(depth, firstLcIndex, lastLcIndex, input) {

  val retriever = new RedefinedValueRetriever(this, processorClass, schema, locationCapable)(ctx)

  override def keyValueOfWithIndex(key: Value[QualifiedName])(implicit ctx: EvaluationContext): IndexedSelection[KeyValuePair] = {
    val tuple = super.keyValueOfWithIndex(key)
    if (tuple != null) {
      tuple
    } else {
      val name = key.evaluate.name
      val v = retriever.getRedefinedValue(name)
      val keyValue = KeyValue(name)
      IndexedSelection(-1, KeyValuePair(keyValue, v))
    }
  }

  override def selectValueWithIndex(key: Value[QualifiedName])(implicit ctx: EvaluationContext): IndexedSelection[Value[_]] = {
    val tuple = super.selectValueWithIndex(key)
    if (tuple != null) {
      tuple
    } else {
      val name = key.evaluate.name
      val v = retriever.getRedefinedValue(name)
      IndexedSelection(-1, v)
    }
  }

  override def allKeyValuesOf(key: Value[QualifiedName])(implicit ctx: EvaluationContext): Option[ObjectSeq] = {
    super.allKeyValuesOf(key) match {
      case None =>
        val name = key.evaluate.name
        val redefined = retriever.getRedefinedValue(name)
        if (redefined == null) {
          None
        } else {
          Some(ObjectSeq(Map(name -> redefined)))
        }

      case some =>
        some

    }
  }

}
