package org.mule.weave.v2.module.dwb.reader.indexed

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.module.dwb.BinaryLocation
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.reader.NumberPrecisionHelper
import org.mule.weave.v2.parser.location.Location

abstract class WeaveBinaryNumber extends NumberValue {
  val token: Array[Long]
  val input: BinaryParserInput
  private lazy val offset = DwTokenHelper.getOffset(token)
  private lazy val value: Number = {
    input.seekableStream.seek(offset)
    readNumber()
  }

  override def evaluate(implicit ctx: EvaluationContext): T = value

  override def location(): Location = new BinaryLocation(offset)

  protected def readNumber(): Number
}

object WeaveBinaryNumber {
  def readInt(input: BinaryParserInput): Int = {
    input.dataInputStream.readInt()
  }

  def readLong(input: BinaryParserInput): Long = {
    input.dataInputStream.readLong()
  }

  def readDouble(input: BinaryParserInput): Double = {
    input.dataInputStream.readDouble()
  }

  def readBigInt(token: Array[Long], input: BinaryParserInput): BigInt = {
    val dataInputStream = input.dataInputStream
    val length = DwTokenHelper.getValueLength(token).toInt
    val buffer = new Array[Byte](length)
    dataInputStream.readFully(buffer, 0, length)
    BigInt.apply(buffer)
  }

  def readBigDecimal(token: Array[Long], input: BinaryParserInput): BigDecimal = {
    val length = DwTokenHelper.getValueLength(token).toInt
    val str = BinaryValueRetriever.readString(input.dataInputStream, length)
    BigDecimal.apply(str)
  }
}

class WeaveBinaryInt(val token: Array[Long], val input: BinaryParserInput) extends WeaveBinaryNumber {
  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    val prop = SchemaProperty(StringValue("class"), StringValue(NumberPrecisionHelper.int_name1))
    val sch = Schema(Seq(prop))
    Some(sch)
  }

  override protected def readNumber(): T = {
    val number = WeaveBinaryNumber.readInt(input)
    Number(number)
  }
}

class WeaveBinaryLong(val token: Array[Long], val input: BinaryParserInput) extends WeaveBinaryNumber {
  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    val prop = SchemaProperty(StringValue("class"), StringValue(NumberPrecisionHelper.long_name1))
    val sch = Schema(Seq(prop))
    Some(sch)
  }

  override protected def readNumber(): T = {
    val number = WeaveBinaryNumber.readLong(input)
    Number(number)
  }
}

class WeaveBinaryDouble(val token: Array[Long], val input: BinaryParserInput) extends WeaveBinaryNumber {
  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    val prop = SchemaProperty(StringValue("class"), StringValue(NumberPrecisionHelper.double_name1))
    val sch = Schema(Seq(prop))
    Some(sch)
  }

  override protected def readNumber(): T = {
    val number = WeaveBinaryNumber.readDouble(input)
    Number(number)
  }
}

class WeaveBinaryBigInt(val token: Array[Long], val input: BinaryParserInput) extends WeaveBinaryNumber {
  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    val prop = SchemaProperty(StringValue("class"), StringValue(NumberPrecisionHelper.big_int_name))
    val sch = Schema(Seq(prop))
    Some(sch)
  }

  override protected def readNumber(): T = {
    Number(WeaveBinaryNumber.readBigInt(token, input))
  }
}

class WeaveBinaryBigDecimal(val token: Array[Long], val input: BinaryParserInput) extends WeaveBinaryNumber {

  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    val prop = SchemaProperty(StringValue("class"), StringValue(NumberPrecisionHelper.big_decimal_name))
    val sch = Schema(Seq(prop))
    Some(sch)
  }

  override protected def readNumber(): T = {
    WeaveBinaryNumber.readBigDecimal(token, input)
  }
}
