/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.metadata.cache;

import com.google.common.base.Optional;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.mule.common.Result;

import java.io.*;
import java.nio.channels.FileLock;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

/**
 * @author MuleSoft, Inc
 */
public class DefaultMetaDataCache implements PersistentMetaDataCache {

    private static final Logger log = Logger.getLogger(DefaultMetaDataCache.class);

    private Map<Serializable, Serializable> cache = new HashMap<Serializable, Serializable>();
    private long lastModifiedFileTimeOnLoad = 0;
    private boolean wasModified = true;
    private Path location = null;

    @Override
    public Result.Status load(File persistentCache) throws IOException, ClassNotFoundException {
        log.debug("Loading cache from " + persistentCache.getPath());
        Result.Status result = Result.Status.SUCCESS;
        if (persistentCache.length() > 0 ){
            FileInputStream cacheIS = new FileInputStream(persistentCache);
            try {
                FileLock fileLock = cacheIS.getChannel().lock(0, persistentCache.length(), true);
                ObjectInputStream objIn = new ObjectInputStream(new BufferedInputStream(cacheIS));
                try {
                    cache = (Map<Serializable, Serializable>) objIn.readObject();
                    wasModified = false;
                    log.debug("Loaded successfully");
                } catch (Exception e) {
                    log.error("An error occurred while loading the Cache " + persistentCache.getPath());
                    result = Result.Status.FAILURE;
                } finally {
                    fileLock.release();
                    IOUtils.closeQuietly(objIn);
                }
            } finally {
                IOUtils.closeQuietly(cacheIS);
            }
        }

        location = persistentCache.toPath();
        lastModifiedFileTimeOnLoad = persistentCache.lastModified();
        return result;
    }

    @Override
    public Result.Status save(File persistentCache, boolean overwrite){

        if (!this.wasModified){
            return Result.Status.SUCCESS;
        }

        Result.Status result = Result.Status.FAILURE;
        try {
            if (!persistentCache.exists()) {
                log.error("Failed to write cache in path " + persistentCache.getPath() + ". File not found");
            } else {
                if (!persistedCacheWasModified(persistentCache) || overwrite){
                    writeCache(persistentCache);
                    lastModifiedFileTimeOnLoad = persistentCache.lastModified();
                    wasModified = false;
                    result = Result.Status.SUCCESS;
                    log.debug("Saved MetaDataCache to " + persistentCache.getPath());
                } else {
                    log.error("Cache was modified before this instance was persisted. To keep it consistent the cache will be destroyed");
                }
            }
        } catch (Exception e){
            log.error("An error occurred while persisting the cache");
        }

        return result;
    }

    @Override
    public Path getLocation(){
        return location;
    }

    @Override
    public void put(Serializable key, Serializable value) {
        cache.put(key, value);
        wasModified = true;
    }

    @Override
    public void putAll(Map<? extends Serializable, ? extends Serializable> values) {
        cache.putAll(values);
        wasModified = true;
    }

    @Override
    public <T> Optional<T> get(Serializable key) {
        return (Optional<T>) Optional.fromNullable(cache.get(key));
    }

    private boolean persistedCacheWasModified(File persistentCache) {
        return persistentCache.lastModified() != this.lastModifiedFileTimeOnLoad;
    }

    private void writeCache(File persistentCache) throws IOException, ClassNotFoundException {
        FileOutputStream cacheOS = null;
        ObjectOutputStream objOut = null;
        try {
            cacheOS = new FileOutputStream(persistentCache, false);
            FileLock fileLock = cacheOS.getChannel().lock();
            try {
                objOut = new ObjectOutputStream(new BufferedOutputStream(cacheOS));
                objOut.writeObject(cache);
                objOut.flush();
            } finally {
                fileLock.release();
                IOUtils.closeQuietly(objOut);
            }
        } finally {
            IOUtils.closeQuietly(cacheOS);
        }
    }

}
