/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.dsql;

import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.api.expression.ExpressionManager;
import org.mule.el.datetime.DateTime;
import org.mule.util.TemplateParser;

import javax.xml.datatype.XMLGregorianCalendar;
import java.util.Calendar;
import java.util.Date;

/**
 * @author MuleSoft, Inc.
 */
public class DsqlMelParserUtils {

    private TemplateParser parser = TemplateParser.createMuleStyleParser();

    /**
     * Given a possible string {@code query} element, if it contains MELs that are of "type dates" it must transform them
     * into a meaningful type of string so that the DSQL grammar can process them properly.
     *
     * @param muleContext context to obtain the expression manager
     * @param event       event to work with
     * @param query       object to parse. Ideally, it should be a string.
     * @return completed parsed query if it was a String, the object otherwise (DevKit will evaluate it if needed in the generated message processor)
     */
    public Object parseDsql(MuleContext muleContext, MuleEvent event, Object query) {
        // just in case we receive anything different from a String, we want to make sure that the DSQL behaviour remains the same.
        if (!(query instanceof String)) {
            return query;
        }

        String stringQuery = (String) query;
        TemplateParser.PatternInfo style = TemplateParser.createMuleStyleParser().getStyle();
        ExpressionManager expressionManager = muleContext.getExpressionManager();

        if (stringQuery.startsWith(style.getPrefix()) && stringQuery.endsWith(style.getSuffix())) {
            return expressionManager.evaluate(stringQuery, event);
        } else {
            return parse(expressionManager, event, stringQuery);
        }
    }

    /**
     * @param expressionManager
     * @param event
     * @param source
     * @return the parsed element if it was a "type date", the evaluated object otherwise
     */
    private Object parse(final ExpressionManager expressionManager, final MuleEvent event, String source) {
        return parser.parse(new TemplateParser.TemplateCallback() {

            public Object match(String token) {
                Object result = expressionManager.evaluate(token, event, false);
                if (result instanceof Calendar) {
                    return new DateTime((Calendar) result).toString();
                } else if (result instanceof Date) {
                    return new DateTime((Date) result).toString();
                } else if (result instanceof XMLGregorianCalendar) {
                    return new DateTime((XMLGregorianCalendar) result).toString();
                } else if (result instanceof MuleMessage) {
                    return ((MuleMessage) result).getPayload();
                } else {
                    return result;
                }
            }
        }, source);
    }
}
