/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.mule.devkit.model.code;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * A block of Java code, which may contain statements and local declarations.
 * <p/>
 * <p/>
 * {@link GeneratedBlock} contains a large number of factory methods that creates new
 * statements/declarations. Those newly created statements/declarations are
 * inserted into the {@link #pos() "current position"}. The position advances
 * one every time you add a new instruction.
 */
public final class GeneratedBlock implements Generable, Statement {

    /**
     * Declarations and statements contained in this block.
     * Either {@link Statement} or {@link Declaration}.
     */
    private final List<Object> content = new ArrayList<Object>();

    /**
     * Whether or not this block must be braced and indented
     */
    private boolean bracesRequired = true;
    private boolean indentRequired = true;

    /**
     * Current position.
     */
    private int pos;

    public GeneratedBlock() {
        this(true, true);
    }

    public GeneratedBlock(boolean bracesRequired, boolean indentRequired) {
        this.bracesRequired = bracesRequired;
        this.indentRequired = indentRequired;
    }

    /**
     * Returns a read-only view of {@link Statement}s and {@link Declaration}
     * in this block.
     */
    public List<Object> getContents() {
        return Collections.unmodifiableList(content);
    }

    private <T> T insert(T statementOrDeclaration) {
        content.add(pos, statementOrDeclaration);
        pos++;
        return statementOrDeclaration;
    }

    /**
     * Gets the current position to which new statements will be inserted.
     * <p/>
     * For example if the value is 0, newly created instructions will be
     * inserted at the very beginning of the block.
     *
     * @see #pos(int)
     */
    public int pos() {
        return pos;
    }

    /**
     * Sets the current position.
     *
     * @return the old value of the current position.
     * @throws IllegalArgumentException if the new position value is illegal.
     * @see #pos()
     */
    public int pos(int newPos) {
        int r = pos;
        if (newPos > content.size() || newPos < 0) {
            throw new IllegalArgumentException();
        }
        pos = newPos;

        return r;
    }

    /**
     * Returns true if this block is empty and does not contain
     * any statement.
     */
    public boolean isEmpty() {
        return content.isEmpty();
    }

    /**
     * Adds a local variable declaration to this block
     *
     * @param type Type of the variable
     * @param name Name of the variable
     * @return Newly generated Variable
     */
    public GeneratedVariable decl(Type type, String name) {
        return decl(Modifier.NONE, type, name, null);
    }

    /**
     * Adds a local variable declaration to this block
     *
     * @param type Type of the variable
     * @param name Name of the variable
     * @param init Initialization expression for this variable.  May be null.
     * @return Newly generated Variable
     */
    public GeneratedVariable decl(Type type, String name, GeneratedExpression init) {
        return decl(Modifier.NONE, type, name, init);
    }

    /**
     * Adds a local variable declaration to this block
     *
     * @param mods Modifiers for the variable
     * @param type Type of the variable
     * @param name Name of the variable
     * @param init Initialization expression for this variable.  May be null.
     * @return Newly generated Variable
     */
    public GeneratedVariable decl(int mods, Type type, String name, GeneratedExpression init) {
        GeneratedVariable v = new GeneratedVariable(Modifiers.forVar(mods), type, name, init);
        insert(v);
        bracesRequired = true;
        indentRequired = true;
        return v;
    }

    /**
     * Creates an assignment statement and adds it to this block.
     *
     * @param lhs Assignable variable or field for left hand side of expression
     * @param exp Right hand side expression
     */
    public GeneratedBlock assign(AssignmentTarget lhs, GeneratedExpression exp) {
        insert(new Assignment(lhs, exp));
        return this;
    }

    public GeneratedBlock assignPlus(AssignmentTarget lhs, GeneratedExpression exp) {
        insert(new Assignment(lhs, exp, "+"));
        return this;
    }

    /**
     * Creates an invocation statement and adds it to this block.
     *
     * @param expr   Expression evaluating to the class or object upon which
     *               the named method will be invoked
     * @param method Name of method to invoke
     * @return Newly generated Invocation
     */
    public GeneratedInvocation invoke(GeneratedExpression expr, String method) {
        GeneratedInvocation i = new GeneratedInvocation(expr, method);
        insert(i);
        return i;
    }

    /**
     * Creates an invocation statement and adds it to this block.
     *
     * @param expr   Expression evaluating to the class or object upon which
     *               the method will be invoked
     * @param method Method to invoke
     * @return Newly generated Invocation
     */
    public GeneratedInvocation invoke(GeneratedExpression expr, GeneratedMethod method) {
        return insert(new GeneratedInvocation(expr, method));
    }

    /**
     * Creates a static invocation statement.
     */
    public GeneratedInvocation staticInvoke(TypeReference type, String method) {
        return insert(new GeneratedInvocation(type, method));
    }

    /**
     * Creates an invocation statement and adds it to this block.
     *
     * @param method Name of method to invoke
     * @return Newly generated Invocation
     */
    public GeneratedInvocation invoke(String method) {
        return insert(new GeneratedInvocation((GeneratedExpression) null, method));
    }

    /**
     * Creates an invocation statement and adds it to this block.
     *
     * @param method Method to invoke
     * @return Newly generated Invocation
     */
    public GeneratedInvocation invoke(GeneratedMethod method) {
        return insert(new GeneratedInvocation((GeneratedExpression) null, method));
    }

    /**
     * Adds a statement to this block
     *
     * @param s Statement to be added
     * @return This block
     */
    public GeneratedBlock add(Statement s) { // ## Needed?
        insert(s);
        return this;
    }

    /**
     * Create an If statement and add it to this block
     *
     * @param expr Expression to be tested to determine branching
     * @return Newly generated conditional statement
     */
    public GeneratedConditional _if(GeneratedExpression expr) {
        return insert(new GeneratedConditional(expr));
    }

    /**
     * Create a For statement and add it to this block
     *
     * @return Newly generated For statement
     */
    public GeneratedForLoop _for() {
        return insert(new GeneratedForLoop());
    }

    /**
     * Create a While statement and add it to this block
     *
     * @return Newly generated While statement
     */
    public GeneratedWhileLoop _while(GeneratedExpression test) {
        return insert(new GeneratedWhileLoop(test));
    }

    /**
     * Create a switch/case statement and add it to this block
     */
    public GeneratedSwitch _switch(GeneratedExpression test) {
        return insert(new GeneratedSwitch(test));
    }

    /**
     * Create a Do statement and add it to this block
     *
     * @return Newly generated Do statement
     */
    public GeneratedDoLoop _do(GeneratedExpression test) {
        return insert(new GeneratedDoLoop(test));
    }

    /**
     * Create a Try statement and add it to this block
     *
     * @return Newly generated Try statement
     */
    public GeneratedTry _try() {
        return insert(new GeneratedTry());
    }

    /**
     * Create a return statement and add it to this block
     */
    public void _return() {
        insert(new GeneratedReturn(null));
    }

    /**
     * Create a return statement and add it to this block
     */
    public void _return(GeneratedExpression exp) {
        insert(new GeneratedReturn(exp));
    }

    /**
     * Create a throw statement and add it to this block
     */
    public void _throw(GeneratedExpression exp) {
        insert(new GeneratedThrow(exp));
    }

    /**
     * Create a break statement and add it to this block
     */
    public void _break() {
        _break(null);
    }

    public void _break(GeneratedLabel label) {
        insert(new GeneratedBreak(label));
    }

    /**
     * Create a label, which can be referenced from
     * <code>continue</code> and <code>break</code> statements.
     */
    public GeneratedLabel label(String name) {
        GeneratedLabel l = new GeneratedLabel(name);
        insert(l);
        return l;
    }

    /**
     * Create a continue statement and add it to this block
     */
    public void _continue(GeneratedLabel label) {
        insert(new ContinueStatement(label));
    }

    public void _continue() {
        _continue(null);
    }

    /**
     * Create a sub-block and add it to this block
     */
    public GeneratedBlock block() {
        GeneratedBlock b = new GeneratedBlock();
        b.bracesRequired = false;
        b.indentRequired = false;
        return insert(b);
    }

    /**
     * Creates a "literal" statement directly.
     * <p/>
     * <p/>
     * Specified string is printed as-is.
     * This is useful as a short-cut.
     * <p/>
     * <p/>
     * For example, you can invoke this method as:
     * <code>directStatement("a=b+c;")</code>.
     */
    public Statement directStatement(final String source) {
        Statement s = new Statement() {
            public void state(Formatter f) {
                f.p(source).nl();
            }
        };
        add(s);
        return s;
    }

    public void generate(Formatter f) {
        if (bracesRequired) {
            f.p('{').nl();
        }
        if (indentRequired) {
            f.i();
        }
        generateBody(f);
        if (indentRequired) {
            f.o();
        }
        if (bracesRequired) {
            f.p('}');
        }
    }

    void generateBody(Formatter f) {
        for (Object o : content) {
            if (o instanceof Declaration) {
                f.d((Declaration) o);
            } else {
                f.s((Statement) o);
            }
        }
    }

    /**
     * Creates an enhanced For statement based on j2se 1.5 JLS
     * and add it to this block
     *
     * @return Newly generated enhanced For statement per j2se 1.5
     *         specification
     */
    public GeneratedForEach forEach(Type varType, String name, GeneratedExpression collection) {
        return insert(new GeneratedForEach(varType, name, collection));

    }

    public void state(Formatter f) {
        f.g(this);
        if (bracesRequired) {
            f.nl();
        }
    }

}
