/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.model.module.oauth;

import org.mule.devkit.model.Field;
import org.mule.devkit.model.Method;
import org.mule.security.oauth.OnNoTokenPolicy;

import javax.lang.model.type.DeclaredType;
import java.util.List;

/**
 * Interface that sums up the complete behaviour that a Component/Module({@link org.mule.api.annotations.oauth.OAuth2})
 * must have in order to trigger the generation of the OAuth2/1 feature
 */
public interface OAuthCapability {
    OAuthVersion getOAuthVersion();

    /**
     * Retrieve the field annotated with {@link org.mule.api.annotations.oauth.OAuthConsumerKey}
     */
    Field getConsumerKeyField();

    /**
     * Retrieve the field annotated with {@link org.mule.api.annotations.oauth.OAuthConsumerSecret}
     */
    Field getConsumerSecretField();

    /**
     * Retrieve the field annotated with {@link org.mule.api.annotations.oauth.OAuthAccessToken}
     */
    Field getAccessTokenField();

    /**
     * Retrieve the field annotated with {@link org.mule.api.annotations.oauth.OAuthScope}
     */
    Field getScopeField();

    /**
     * Retrieve the field annotated with {@link org.mule.api.annotations.oauth.OAuthAccessTokenSecret}
     */
    Field getAccessTokenSecretField();

    /**
     * A Java regular expression used to extract the verifier from the Service Provider response as a result
     * of the Resource Owner authorizing the Consumer.
     */
    String getVerifierRegex();

    /**
     * The URL defined by the Service Provider used to obtain an un-authorized Request Token
     */
    String getRequestTokenUrl();

    /**
     * The URL defined by the Service Provider to obtain an Access Token
     */
    String getAccessTokenUrl();

    /**
     * The URL defined by the Service Provider where the Resource Owner will be redirected to grant
     * authorization to the Consumer
     */
    String getAuthorizationUrl();

    /**
     * A Java regular expression used to extract the expiration time of the Access Token (in seconds) from the
     * Service Provider response. If the this regular expression is not found in the Service Provider response
     * (whether the regular expression is wrong or the Access Token never expires), the Access Token will be
     * treated as if it would never expire.
     */
    String getExpirationRegex();

    /**
     * A Java regular expression used to extract the Access Token from the Service Provider response.
     */
    String getAccessTokenRegex();

    /**
     * A Java regular expression used to extract the Refresh Token from the Service Provider response.
     */
    String getRefreshTokenRegex();

    /**
     * Retrieve the method that should be executed once the authorization process has ended.
     */
    Method<OAuthModule> getPostAuthorizationMethod();

    /**
     * Extra authorization parameters that will be appended as a query string arguments when building
     * the authorization URL.
     */
    List<OAuthAuthorizationParameter> getAuthorizationParameters();

    /**
     * Retrieve a list of fields that need to be populated once the callback in OAuth is executed.
     */
    List<OAuthCallbackParameterField> getCallbackParameters();

    /**
     * Retrieve the method marked with {@link org.mule.api.annotations.oauth.OAuthAccessTokenIdentifier}
     */
    Method<OAuthModule> getUserIdentifierMethod();

    DeclaredType getOAuthProvider();

    OnNoTokenPolicy getTokenPolicy();
}
