/*
 * Decompiled with CFR 0.152.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import java.io.IOException;
import java.io.Writer;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.mule.runtime.api.config.Feature;
import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.FeatureFlaggingRegistry;
import org.mule.runtime.core.api.policy.PolicyProvider;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.application.ApplicationStatus;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.deployment.model.api.domain.DomainStatus;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPlugin;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplate;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.deployment.impl.internal.application.MuleApplicationPolicyProvider;
import org.mule.runtime.module.deployment.impl.internal.policy.ApplicationPolicyInstance;
import org.mule.runtime.module.troubleshooting.api.ArgumentDefinition;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperation;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationCallback;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultTroubleshootingOperationDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeploymentsOperation
implements TroubleshootingOperation {
    public static final String DEPLOYMENTS_OPERATION_NAME = "deployments";
    public static final String DEPLOYMENTS_OPERATION_DESCRIPTION = "Lists all current deployments: applications, domains and policies";
    private static final TroubleshootingOperationDefinition definition = DeploymentsOperation.createOperationDefinition();
    private static final Logger LOGGER = LoggerFactory.getLogger((String)DeploymentsOperation.class.getName());
    private final DeploymentService deploymentService;

    public DeploymentsOperation(DeploymentService deploymentService) {
        this.deploymentService = deploymentService;
    }

    @Override
    public TroubleshootingOperationDefinition getDefinition() {
        return definition;
    }

    @Override
    public TroubleshootingOperationCallback getCallback() {
        return (arguments, writer) -> this.writeDeploymentsInfo(writer);
    }

    private void writeDeploymentsInfo(Writer writer) throws IOException {
        LinkedHashMap<String, PolicyTemplateInfo> policyTemplates = new LinkedHashMap<String, PolicyTemplateInfo>();
        List<DeploymentInfo> deployments = this.collectDeployments(policyTemplates);
        if (deployments.isEmpty() && policyTemplates.isEmpty()) {
            writer.write("No deployments found." + System.lineSeparator());
            return;
        }
        for (DeploymentInfo deployment : deployments) {
            this.writeDeploymentInfo(deployment, writer);
            writer.write(System.lineSeparator());
        }
        if (!policyTemplates.isEmpty()) {
            for (PolicyTemplateInfo template : policyTemplates.values()) {
                this.writePolicyTemplateInfo(template, writer);
                writer.write(System.lineSeparator());
            }
        }
    }

    private List<DeploymentInfo> collectDeployments(Map<String, PolicyTemplateInfo> policyTemplates) {
        ArrayList<DeploymentInfo> deployments = new ArrayList<DeploymentInfo>();
        for (Application app : this.deploymentService.getApplications()) {
            List<PolicyInfo> policies = this.collectPoliciesFromApplication(app, policyTemplates);
            deployments.add(this.createApplicationDeploymentInfo(app, policies));
        }
        for (Domain domain : this.deploymentService.getDomains()) {
            deployments.add(this.createDomainDeploymentInfo(domain));
        }
        return deployments;
    }

    private List<PolicyInfo> collectPoliciesFromApplication(Application app, Map<String, PolicyTemplateInfo> policyTemplates) {
        ArrayList<PolicyInfo> policies = new ArrayList<PolicyInfo>();
        this.getMuleApplicationPolicyProvider(app).ifPresent(policyProvider -> policies.addAll(this.extractPoliciesFromProvider((MuleApplicationPolicyProvider)policyProvider, policyTemplates)));
        return policies;
    }

    private Optional<MuleApplicationPolicyProvider> getMuleApplicationPolicyProvider(Application app) {
        try {
            Object t;
            Optional policyProviderOpt;
            if (app.getArtifactContext() != null && app.getArtifactContext().getRegistry() != null && (policyProviderOpt = app.getArtifactContext().getRegistry().lookupByType(PolicyProvider.class)).isPresent() && (t = policyProviderOpt.get()) instanceof MuleApplicationPolicyProvider) {
                MuleApplicationPolicyProvider muleProvider = (MuleApplicationPolicyProvider)t;
                return Optional.of(muleProvider);
            }
        }
        catch (Exception e) {
            LOGGER.debug("Unable to access policy provider for application '{}'", (Object)app.getArtifactName(), (Object)e);
        }
        return Optional.empty();
    }

    private List<PolicyInfo> extractPoliciesFromProvider(MuleApplicationPolicyProvider provider, Map<String, PolicyTemplateInfo> policyTemplates) {
        try {
            return provider.getRegisteredPolicyInstanceProviders().stream().filter(providerObj -> providerObj.getPolicyId() != null).map(providerObj -> {
                String policyId = providerObj.getPolicyId();
                Optional.ofNullable(providerObj.getApplicationPolicyInstance()).map(ApplicationPolicyInstance::getPolicyTemplate).ifPresent(policyTemplate -> {
                    String uniqueKey = this.extractTemplateUniqueKey((PolicyTemplate)policyTemplate);
                    if (uniqueKey != null && !policyTemplates.containsKey(uniqueKey)) {
                        String templateName = policyTemplate.getArtifactName();
                        String templateVersion = this.extractTemplateVersion((PolicyTemplate)policyTemplate);
                        List<PluginInfo> plugins = this.extractPluginsFromPolicyTemplate((PolicyTemplate)policyTemplate);
                        policyTemplates.put(uniqueKey, new PolicyTemplateInfo(uniqueKey, templateName, templateVersion, plugins));
                    }
                });
                return new PolicyInfo(policyId);
            }).collect(Collectors.toList());
        }
        catch (Exception e) {
            LOGGER.debug("Unable to extract policies from provider", (Throwable)e);
            return Collections.emptyList();
        }
    }

    private String extractTemplateUniqueKey(PolicyTemplate template) {
        if (template.getDescriptor() != null && ((PolicyTemplateDescriptor)template.getDescriptor()).getBundleDescriptor() != null) {
            BundleDescriptor bundleDesc = ((PolicyTemplateDescriptor)template.getDescriptor()).getBundleDescriptor();
            return bundleDesc.getGroupId() + ":" + bundleDesc.getArtifactId() + ":" + bundleDesc.getVersion();
        }
        return null;
    }

    private String extractTemplateVersion(PolicyTemplate template) {
        if (template.getDescriptor() != null && ((PolicyTemplateDescriptor)template.getDescriptor()).getBundleDescriptor() != null) {
            return ((PolicyTemplateDescriptor)template.getDescriptor()).getBundleDescriptor().getVersion();
        }
        return "unknown";
    }

    private List<PluginInfo> extractPluginsFromPolicyTemplate(PolicyTemplate policyTemplate) {
        ArrayList<PluginInfo> plugins = new ArrayList<PluginInfo>();
        try {
            List artifactPlugins = policyTemplate.getArtifactPlugins();
            if (artifactPlugins != null) {
                for (ArtifactPlugin plugin : artifactPlugins) {
                    String pluginName = ((ArtifactPluginDescriptor)plugin.getDescriptor()).getName();
                    String version = this.extractPluginVersion((ArtifactPluginDescriptor)plugin.getDescriptor());
                    plugins.add(new PluginInfo(pluginName, version));
                }
            }
        }
        catch (Exception e) {
            LOGGER.debug("Unable to extract plugins from policy template", (Throwable)e);
        }
        plugins.sort((p1, p2) -> p1.name.compareTo(p2.name));
        return plugins;
    }

    private DeploymentInfo createApplicationDeploymentInfo(Application app, List<PolicyInfo> policies) {
        String domainArtifactName;
        String artifactName = app.getArtifactName();
        String artifactType = "APPLICATION";
        String state = this.mapApplicationStatus(app.getStatus());
        String domainName = null;
        Domain domain = app.getDomain();
        if (domain != null && (domainArtifactName = domain.getArtifactName()) != null && !"default".equals(domainArtifactName)) {
            domainName = domainArtifactName;
        }
        List<PluginInfo> plugins = this.extractPlugins(app);
        List<String> featureFlags = this.extractFeatureFlags(app);
        String lastStartTime = this.calculateLastStartTime(app.getArtifactContext(), artifactName);
        return new DeploymentInfo(artifactName, artifactType, state, plugins, featureFlags, lastStartTime, policies, domainName);
    }

    private DeploymentInfo createDomainDeploymentInfo(Domain domain) {
        String artifactName = domain.getArtifactName();
        String artifactType = "DOMAIN";
        String state = this.mapDomainStatus(domain.getStatus());
        List<PluginInfo> plugins = this.extractPlugins(domain);
        List<String> featureFlags = Collections.emptyList();
        String lastStartTime = this.calculateLastStartTime(domain.getArtifactContext(), artifactName);
        return new DeploymentInfo(artifactName, artifactType, state, plugins, featureFlags, lastStartTime, Collections.emptyList(), null);
    }

    private String mapApplicationStatus(ApplicationStatus status) {
        return switch (status) {
            default -> throw new IncompatibleClassChangeError();
            case ApplicationStatus.CREATED -> "INITIAL";
            case ApplicationStatus.INITIALISED -> "DEPLOYED";
            case ApplicationStatus.STARTED -> "STARTED";
            case ApplicationStatus.STOPPED -> "STOPPED";
            case ApplicationStatus.DEPLOYMENT_FAILED -> "FAILED";
            case ApplicationStatus.DESTROYED -> "UNDEPLOYED";
        };
    }

    private String mapDomainStatus(DomainStatus status) {
        return switch (status) {
            default -> throw new IncompatibleClassChangeError();
            case DomainStatus.CREATED -> "INITIAL";
            case DomainStatus.INITIALISED -> "DEPLOYED";
            case DomainStatus.STARTED -> "STARTED";
            case DomainStatus.STOPPED -> "STOPPED";
            case DomainStatus.DEPLOYMENT_FAILED -> "FAILED";
            case DomainStatus.DESTROYED -> "UNDEPLOYED";
        };
    }

    private List<PluginInfo> extractPlugins(Application app) {
        return this.extractPlugins(() -> ((Application)app).getArtifactPlugins(), () -> ((Application)app).getDescriptor());
    }

    private List<PluginInfo> extractPlugins(Domain domain) {
        return this.extractPlugins(() -> ((Domain)domain).getArtifactPlugins(), () -> ((Domain)domain).getDescriptor());
    }

    private List<PluginInfo> extractPlugins(Supplier<List<ArtifactPlugin>> artifactPluginsSupplier, Supplier<DeployableArtifactDescriptor> descriptorSupplier) {
        ArrayList<PluginInfo> plugins = new ArrayList<PluginInfo>();
        try {
            List<ArtifactPlugin> artifactPlugins = artifactPluginsSupplier.get();
            if (artifactPlugins != null) {
                for (ArtifactPlugin plugin : artifactPlugins) {
                    String pluginName = ((ArtifactPluginDescriptor)plugin.getDescriptor()).getName();
                    String version = this.extractPluginVersion((ArtifactPluginDescriptor)plugin.getDescriptor());
                    plugins.add(new PluginInfo(pluginName, version));
                }
            }
            if (plugins.isEmpty()) {
                this.addPluginsFromDescriptor(plugins, descriptorSupplier.get());
            }
        }
        catch (Exception e) {
            LOGGER.debug("Unable to extract plugins from artifact, trying descriptor", (Throwable)e);
            this.addPluginsFromDescriptor(plugins, descriptorSupplier.get());
        }
        Collections.sort(plugins, (p1, p2) -> p1.name.compareTo(p2.name));
        return plugins;
    }

    private void addPluginsFromDescriptor(List<PluginInfo> plugins, DeployableArtifactDescriptor descriptor) {
        if (descriptor != null) {
            try {
                Set descriptorPlugins = descriptor.getPlugins();
                if (descriptorPlugins != null) {
                    for (ArtifactPluginDescriptor pluginDesc : descriptorPlugins) {
                        String pluginName = pluginDesc.getName();
                        String version = this.extractPluginVersion(pluginDesc);
                        plugins.add(new PluginInfo(pluginName, version));
                    }
                }
            }
            catch (Exception e) {
                LOGGER.debug("Unable to extract plugin information from descriptor '{}'", (Object)descriptor.getName(), (Object)e);
            }
        }
    }

    private String extractPluginVersion(ArtifactPluginDescriptor pluginDesc) {
        BundleDescriptor bundleDesc = pluginDesc.getBundleDescriptor();
        if (bundleDesc != null) {
            return bundleDesc.getVersion();
        }
        return "unknown";
    }

    private List<String> extractFeatureFlags(Application app) {
        ArrayList<String> enabledFeatures = new ArrayList<String>();
        try {
            FeatureFlaggingService featureFlaggingService;
            ApplicationStatus status = app.getStatus();
            if (status != ApplicationStatus.DEPLOYMENT_FAILED && status != ApplicationStatus.DESTROYED && (featureFlaggingService = (FeatureFlaggingService)app.getArtifactContext().getRegistry().lookupByType(FeatureFlaggingService.class).orElse(null)) != null) {
                Map registeredFeatures = FeatureFlaggingRegistry.getInstance().getFeatureFlagConfigurations();
                for (Feature feature : registeredFeatures.keySet()) {
                    try {
                        if (!featureFlaggingService.isEnabled(feature)) continue;
                        enabledFeatures.add(feature.name());
                    }
                    catch (Exception e) {
                        LOGGER.debug("Unable to check if feature '{}' is enabled", (Object)feature.name(), (Object)e);
                    }
                }
            }
        }
        catch (Exception e) {
            LOGGER.debug("Unable to extract feature flags from application '{}'", (Object)app.getArtifactName(), (Object)e);
        }
        Collections.sort(enabledFeatures);
        return enabledFeatures;
    }

    private String calculateLastStartTime(ArtifactContext artifactContext, String deploymentName) {
        try {
            long startDate;
            MuleContext muleContext;
            if (artifactContext != null && (muleContext = artifactContext.getMuleContext()) != null && (startDate = muleContext.getStartDate()) > 0L) {
                return this.formatTimestamp(startDate);
            }
        }
        catch (Exception e) {
            LOGGER.debug("Unable to calculate last start time for deployment '{}'", (Object)deploymentName, (Object)e);
        }
        return "unknown";
    }

    private String formatTimestamp(long timestampMillis) {
        LocalDateTime dateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(timestampMillis), ZoneId.systemDefault());
        return dateTime.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
    }

    private void writeDeploymentInfo(DeploymentInfo deployment, Writer writer) throws IOException {
        writer.write(String.format("\"%s\" - type: %s, state: %s", deployment.artifactName, deployment.artifactType, deployment.state));
        writer.write(System.lineSeparator());
        if (deployment.domainName != null && "APPLICATION".equals(deployment.artifactType)) {
            writer.write("  Domain: " + deployment.domainName);
            writer.write(System.lineSeparator());
        }
        writer.write("  Last started: " + deployment.timeSinceLastUpdate);
        writer.write(System.lineSeparator());
        if (!deployment.policies.isEmpty() && "APPLICATION".equals(deployment.artifactType)) {
            writer.write("  Policies:");
            writer.write(System.lineSeparator());
            for (PolicyInfo policy : deployment.policies) {
                writer.write("    - " + policy.id);
                writer.write(System.lineSeparator());
            }
        }
        if (!deployment.plugins.isEmpty()) {
            writer.write("  Plugins:");
            writer.write(System.lineSeparator());
            for (PluginInfo plugin : deployment.plugins) {
                writer.write(String.format("    - %s : %s", plugin.name, plugin.version));
                writer.write(System.lineSeparator());
            }
        }
        if (!deployment.featureFlags.isEmpty() && (deployment.state.equals("DEPLOYED") || deployment.state.equals("STARTED") || deployment.state.equals("STOPPED"))) {
            writer.write("  Enabled Feature Flags:");
            writer.write(System.lineSeparator());
            for (String featureFlag : deployment.featureFlags) {
                writer.write("    - " + featureFlag);
                writer.write(System.lineSeparator());
            }
        }
    }

    private void writePolicyTemplateInfo(PolicyTemplateInfo template, Writer writer) throws IOException {
        writer.write(String.format("\"%s\" - type: POLICY_TEMPLATE, version: %s", template.name != null ? template.name : template.id, template.version));
        writer.write(System.lineSeparator());
        if (!template.plugins.isEmpty()) {
            writer.write("  Plugins:");
            writer.write(System.lineSeparator());
            for (PluginInfo plugin : template.plugins) {
                writer.write(String.format("    - %s : %s", plugin.name, plugin.version));
                writer.write(System.lineSeparator());
            }
        }
    }

    private static TroubleshootingOperationDefinition createOperationDefinition() {
        return new DefaultTroubleshootingOperationDefinition(DEPLOYMENTS_OPERATION_NAME, DEPLOYMENTS_OPERATION_DESCRIPTION, new ArgumentDefinition[0]);
    }

    private record DeploymentInfo(String artifactName, String artifactType, String state, List<PluginInfo> plugins, List<String> featureFlags, String timeSinceLastUpdate, List<PolicyInfo> policies, String domainName) {
    }

    private record PolicyTemplateInfo(String id, String name, String version, List<PluginInfo> plugins) {
    }

    private record PluginInfo(String name, String version) {
    }

    private record PolicyInfo(String id) {
    }
}

