/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.dsl.model.extension.xml;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.repeat;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.extension.api.property.XmlExtensionModelProperty;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import org.jgrapht.Graph;
import org.jgrapht.graph.DefaultEdge;
import org.jgrapht.graph.DirectedMultigraph;
import org.jgrapht.traverse.GraphIterator;
import org.jgrapht.traverse.TopologicalOrderIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link MacroExpansionModulesModel} goes over all the parametrized {@link ExtensionModel} by filtering them if they have the
 * {@link XmlExtensionModelProperty} (implies that has to be macro expanded).
 * <p/>
 * For every occurrence that happens, it will expand the operations/configurations by working with the
 * {@link MacroExpansionModuleModel} passing through just one {@link ExtensionModel} to macro expand in the current Mule
 * Application (held by the {@link ApplicationModel}.
 *
 * @since 4.0
 */
public class MacroExpansionModulesModel {

  private static final Logger LOGGER = LoggerFactory.getLogger(MacroExpansionModulesModel.class);
  private static final String FILE_MACRO_EXPANSION_DELIMITER = repeat('*', 80) + lineSeparator();
  private static final String FILE_MACRO_EXPANSION_SECTION_DELIMITER = repeat('-', 80) + lineSeparator();

  private ArtifactAst applicationModel;
  private final List<ExtensionModel> sortedExtensions;

  /**
   * From a mutable {@code applicationModel}, it will store it to apply changes when the {@link #expand()} method is executed.
   *
   * @param applicationModel to modify given the usages of elements that belong to the {@link ExtensionModel}s contained in the
   *                         {@code extensions} map.
   * @param extensions       set with all the loaded {@link ExtensionModel}s from the deployment that will be filtered by looking
   *                         up only those that are coming from an XML context through the {@link XmlExtensionModelProperty}
   *                         property.
   */
  public MacroExpansionModulesModel(ArtifactAst applicationModel, Set<ExtensionModel> extensions) {
    this.applicationModel = applicationModel;
    this.sortedExtensions = calculateExtensionByTopologicalOrder(extensions);
  }

  /**
   * Goes through the entire xml mule application looking for the message processors that can be expanded, and then takes care of
   * the global elements if there are at least one {@link ExtensionModel} to macro expand.
   */
  public ArtifactAst expand() {
    boolean hasMacroExpansionExtension = false;

    for (ExtensionModel extensionModel : sortedExtensions) {
      if (extensionModel.getModelProperty(XmlExtensionModelProperty.class).isPresent()) {
        hasMacroExpansionExtension = true;

        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(format("macro expanding '%s' connector, xmlns:%s=\"%s\"",
                              extensionModel.getName(),
                              extensionModel.getXmlDslModel().getPrefix(),
                              extensionModel.getXmlDslModel().getNamespace()));
        }
        applicationModel = new MacroExpansionModuleModel(applicationModel, extensionModel).expand();
      }
    }

    if (hasMacroExpansionExtension) {
      if (LOGGER.isDebugEnabled()) {
        // only log the macro expanded app if there are smart connectors in it
        final StringBuilder buf = new StringBuilder(1024);
        buf.append(lineSeparator()).append(FILE_MACRO_EXPANSION_DELIMITER);

        AtomicReference<String> lastFile = new AtomicReference<>();

        applicationModel.topLevelComponentsStream().forEach(comp -> {
          final String fileName = comp.getMetadata().getFileName().orElse("<unnamed>");

          if (!fileName.equals(lastFile.get())) {
            if (lastFile.get() != null) {
              buf.append(lineSeparator()).append(FILE_MACRO_EXPANSION_SECTION_DELIMITER);
            }
            buf.append("Filename: ").append(fileName);
            buf.append(lineSeparator()).append(FILE_MACRO_EXPANSION_SECTION_DELIMITER);

            lastFile.set(fileName);
          }

          buf
              .append(comp.getMetadata().getSourceCode().orElse(""))
              .append(lineSeparator());
        });

        buf.append(lineSeparator()).append(FILE_MACRO_EXPANSION_DELIMITER);
        LOGGER.debug(buf.toString());
      }
    }

    return applicationModel;
  }

  /**
   * Constructs a Direct Acyclic Graph (DAG) with the dependencies at namespace level of those {@link ExtensionModel} that must be
   * macro expanded with a topological order.
   * <p/>
   * It starts by taking the namespaces of macro expandable <module/>s from the Mule Application, to then assembly a DAG using
   * those namespaces as starting point. For each <module/> namespace, it will go over it's dependencies using
   * {@link #fillDependencyGraph(DirectedGraph, String, Map)}.
   * <p/>
   * Once finished, generates a {@link TopologicalOrderIterator} as the macro expansion relies entirely in the correct order to
   * plain it in a simple {@link List} to be later used in the {@link #expand()} method.
   *
   * @param extensions complete set of {@link ExtensionModel}s used in the app that might or might not be macro expandable (it
   *                   will filter them.
   * @return a <bold>sorted</bold> collection of {@link ExtensionModel} to macro expand. This order must not be altered.
   */
  private List<ExtensionModel> calculateExtensionByTopologicalOrder(Set<ExtensionModel> extensions) {
    final List<ExtensionModel> result = new ArrayList<>();
    final Map<String, ExtensionModel> allExtensionsByNamespace = extensions.stream()
        .filter(extensionModel -> extensionModel.getModelProperty(XmlExtensionModelProperty.class).isPresent())
        .collect(toMap(extModel -> extModel.getXmlDslModel().getNamespace(), Function.identity()));

    // we first check there are at least one extension to macro expand
    if (!allExtensionsByNamespace.isEmpty()) {
      Set<String> extensionsUsedInApp = applicationModel.recursiveStream()
          .map(comp -> comp.getIdentifier().getNamespaceUri())
          .filter(ns -> allExtensionsByNamespace.keySet().contains(ns))
          .collect(toSet());

      // then we assure there are at least one of those extensions being used by the app
      if (!extensionsUsedInApp.isEmpty()) {
        // generation of the DAG and then the topological iterator.
        // it's important to be 100% sure the DAG is not empty, or the TopologicalOrderIterator will fail at start up.
        Graph<String, DefaultEdge> namespaceDAG = new DirectedMultigraph<>(DefaultEdge.class);
        extensionsUsedInApp.forEach(namespace -> fillDependencyGraph(namespaceDAG, namespace, allExtensionsByNamespace));
        GraphIterator<String, DefaultEdge> graphIterator = new TopologicalOrderIterator<>(namespaceDAG);
        while (graphIterator.hasNext()) {
          final String namespace = graphIterator.next();
          if (allExtensionsByNamespace.containsKey(namespace)) {
            result.add(allExtensionsByNamespace.get(namespace));

          }
        }
      }
    }
    return result;
  }

  private void fillDependencyGraph(Graph<String, DefaultEdge> g, String sourceVertex,
                                   Map<String, ExtensionModel> allExtensionsByNamespace) {
    final ExtensionModel extensionModel = allExtensionsByNamespace.get(sourceVertex);
    g.addVertex(sourceVertex);
    for (String dependencyNamespace : getDependenciesOrFail(extensionModel)) {
      if (allExtensionsByNamespace.containsKey(dependencyNamespace)) {
        g.addVertex(dependencyNamespace);
        g.addEdge(sourceVertex, dependencyNamespace);
        fillDependencyGraph(g, dependencyNamespace, allExtensionsByNamespace);
      }
    }
  }

  private Set<String> getDependenciesOrFail(ExtensionModel extensionModel) {
    return extensionModel.getModelProperty(XmlExtensionModelProperty.class)
        .orElseThrow(() -> new IllegalArgumentException(format("The current extension [%s] (namespace [%s]) does not have the macro expansion model property, it should have never reach here.",
                                                               extensionModel.getName(),
                                                               extensionModel.getXmlDslModel().getNamespace())))
        .getNamespacesDependencies();
  }

}
