/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.common;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.ThreadSafe;
import org.jfrog.common.ArgUtils;
import org.jfrog.common.ExecutionFailed;
import org.jfrog.common.RetryException;
import org.jfrog.common.Retryable;
import org.jfrog.common.TimeUnitFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;

@ThreadSafe
public abstract class ExecutionUtils {
    private static final Logger log = LoggerFactory.getLogger(ExecutionUtils.class);

    private ExecutionUtils() {
    }

    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions) {
        return ExecutionUtils.retry(funcToRetry, retryOptions, Executors.newCachedThreadPool());
    }

    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool) {
        CompletableFuture future = new CompletableFuture();
        Runnable task = ExecutionUtils.generateExecutionRunnable(funcToRetry, retryOptions, threadPool, future, 0, System.currentTimeMillis());
        threadPool.submit(task);
        return future;
    }

    private static <T> Runnable generateExecutionRunnable(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime) {
        return () -> {
            try {
                ExecutionUtils.handleFunctionExecution(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime);
            }
            catch (Throwable e) {
                ExecutionUtils.handleStopError(future, "code exception", e.getMessage(), e);
            }
        };
    }

    private static <T> void handleFunctionExecution(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime) throws InterruptedException {
        try {
            future.complete(funcToRetry.tryExecute());
        }
        catch (RetryException e) {
            if (executedRetries + 1 >= retryOptions.numberOfRetries) {
                ExecutionUtils.handleStopError(future, "exceeded number of attempts (" + retryOptions.numberOfRetries + ")", e.getMessage(), e);
            }
            ExecutionUtils.handleRetry(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime, e);
        }
    }

    private static <T> void handleRetry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime, RetryException e) throws InterruptedException {
        if (ExecutionUtils.shouldLog(retryOptions, executedRetries, startTime)) {
            String elapsedTime = TimeUnitFormat.getTimeString(System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
            log.warn("Retry {} Elapsed {} failed: {}. Trying again", new Object[]{executedRetries + 1, elapsedTime, e.getMessage()});
        }
        Thread.sleep(retryOptions.timeout);
        int timeout = retryOptions.timeout * retryOptions.exponentialBackoffMultiplier;
        if (retryOptions.backoffMaxDelay > 0 && retryOptions.backoffMaxDelay < timeout) {
            timeout = retryOptions.backoffMaxDelay;
        }
        RetryOptions newOptions = RetryOptions.fromOther(retryOptions).timeout(timeout).build();
        threadPool.submit(ExecutionUtils.generateExecutionRunnable(funcToRetry, newOptions, threadPool, future, executedRetries + 1, startTime));
    }

    private static boolean shouldLog(RetryOptions retryOptions, int executedRetries, long startTime) {
        return System.currentTimeMillis() > startTime + retryOptions.getReportDelayMillis() && (executedRetries + 1) % retryOptions.getBulkExceptionReportSize() == 0;
    }

    private static <T> void handleStopError(CompletableFuture<T> future, String exceptionMessage, String reasonToFinish, Throwable exception) {
        String message = MessageFormatter.format((String)"Last retry failed: {}. Not trying again ({})", (Object)exceptionMessage, (Object)reasonToFinish).getMessage();
        log.debug(message, exception);
        log.error(message);
        future.completeExceptionally(new ExecutionFailed(message, exception));
    }

    public static class RetryOptions {
        private final int numberOfRetries;
        private final int exponentialBackoffMultiplier;
        private final int timeout;
        private final int backoffMaxDelay;
        private final int bulkExceptionReportSize;
        private final long reportDelayMillis;

        private RetryOptions(int numberOfRetries, int exponentialBackoffMultiplier, int timeout, int backoffMaxDelay, int bulkExceptionReportSize, long reportDelayMillis) {
            this.numberOfRetries = numberOfRetries;
            this.exponentialBackoffMultiplier = exponentialBackoffMultiplier;
            this.timeout = timeout;
            this.backoffMaxDelay = backoffMaxDelay;
            this.bulkExceptionReportSize = bulkExceptionReportSize;
            this.reportDelayMillis = reportDelayMillis;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static Builder fromOther(RetryOptions retryOptions) {
            return new Builder().numberOfRetries(retryOptions.numberOfRetries).exponentialBackoffMultiplier(retryOptions.exponentialBackoffMultiplier).timeout(retryOptions.timeout).backoffMaxDelay(retryOptions.backoffMaxDelay).bulkExceptionReportSize(retryOptions.bulkExceptionReportSize).reportDelayMillis(retryOptions.getReportDelayMillis());
        }

        public int getNumberOfRetries() {
            return this.numberOfRetries;
        }

        public int getExponentialBackoffMultiplier() {
            return this.exponentialBackoffMultiplier;
        }

        public int getTimeout() {
            return this.timeout;
        }

        public int getBackoffMaxDelay() {
            return this.backoffMaxDelay;
        }

        public int getBulkExceptionReportSize() {
            return this.bulkExceptionReportSize;
        }

        public long getReportDelayMillis() {
            return this.reportDelayMillis;
        }

        public static class Builder {
            private int numberOfRetries = 5;
            private int exponentialBackoffMultiplier = 2;
            private int timeout = 100;
            private int backoffMaxDelay = 1000;
            private int bulkExceptionReportSize = 1;
            private long reportDelayMillis;

            public Builder timeout(int timeout) {
                this.timeout = ArgUtils.requireNonNegative(timeout, "timeout must be non negative");
                return this;
            }

            public Builder numberOfRetries(int numberOfRetries) {
                this.numberOfRetries = ArgUtils.requirePositive(numberOfRetries, "numberOfRetries must be non negative");
                return this;
            }

            public Builder exponentialBackoffMultiplier(int exponentialBackoffMultiplier) {
                this.exponentialBackoffMultiplier = ArgUtils.requirePositive(exponentialBackoffMultiplier, "exponentialBackoffMultiplier must be positive");
                return this;
            }

            public Builder backoffMaxDelay(int backoffMaxDelay) {
                this.backoffMaxDelay = ArgUtils.requireNonNegative(backoffMaxDelay, "backoffMaxDelay must be non negative");
                return this;
            }

            public Builder bulkExceptionReportSize(int bulkExceptionReportSize) {
                this.bulkExceptionReportSize = ArgUtils.requireNonNegative(bulkExceptionReportSize, "backoffMaxDelay must be non negative");
                return this;
            }

            public Builder reportDelayMillis(long reportDelayMillis) {
                this.reportDelayMillis = ArgUtils.requireNonNegative(reportDelayMillis, "reportDelayMillis must be non negative");
                return this;
            }

            public RetryOptions build() {
                return new RetryOptions(this.numberOfRetries, this.exponentialBackoffMultiplier, this.timeout, this.backoffMaxDelay, this.bulkExceptionReportSize, this.reportDelayMillis);
            }
        }
    }
}

