/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.plugins.dependency;

import java.lang.annotation.Annotation;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.inject.Qualifier;

import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.beans.metadata.api.model.QualifierPoint;
import org.jboss.beans.metadata.api.model.QualifierType;
import org.jboss.beans.metadata.plugins.AbstractBeanQualifierMetaData;
import org.jboss.beans.metadata.plugins.ContextualInjectionDependencyItem;
import org.jboss.beans.metadata.spi.ConstructorMetaData;
import org.jboss.beans.metadata.spi.LifecycleMetaData;
import org.jboss.beans.metadata.spi.MetaDataVisitorNode;
import org.jboss.beans.metadata.spi.ParameterMetaData;
import org.jboss.beans.metadata.spi.PropertyMetaData;
import org.jboss.beans.metadata.spi.RelatedClassMetaData;
import org.jboss.beans.metadata.spi.factory.BeanFactory;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.DependencyItem;
import org.jboss.joinpoint.plugins.Config;
import org.jboss.kernel.plugins.config.Configurator;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.logging.Logger;
import org.jboss.metadata.spi.MetaData;
import org.jboss.metadata.spi.MutableMetaData;
import org.jboss.metadata.spi.repository.MetaDataRepository;
import org.jboss.metadata.spi.repository.MutableMetaDataRepository;
import org.jboss.metadata.spi.retrieval.MetaDataItem;
import org.jboss.metadata.spi.retrieval.MetaDataRetrieval;
import org.jboss.metadata.spi.retrieval.RetrievalUtils;
import org.jboss.metadata.spi.scope.ScopeKey;
import org.jboss.metadata.spi.signature.ConstructorParametersSignature;
import org.jboss.metadata.spi.signature.ConstructorSignature;
import org.jboss.metadata.spi.signature.DeclaredMethodSignature;
import org.jboss.metadata.spi.signature.FieldSignature;
import org.jboss.metadata.spi.signature.MethodParametersSignature;
import org.jboss.reflect.plugins.introspection.IntrospectionTypeInfoFactoryImpl;
import org.jboss.reflect.spi.ConstructorInfo;
import org.jboss.reflect.spi.FieldInfo;
import org.jboss.reflect.spi.MethodInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.reflect.spi.TypeInfoFactory;
import org.jboss.util.JBossObject;
import org.jboss.util.collection.ConcurrentSet;

/**
 * Utility class to access the MDR for qualifiers.<p>
 * 
 * Supplied qualifiers are stored under the key {@link #SUPPLIED_QUALIFIER_KEY}.</p>
 * 
 * Wanted qualifiers are split into optional (key prefix: {@link #OPTIONAL_QUALIFIER_KEY}) 
 * or required (key prefix: {@link #REQUIRED_QUALIFIER_KEY}). They can apply to a sub-set of the following injection points. Property 
 * (key suffix {@link #PROPERTY_SUFFIX}), method (key suffix: {@link #METHOD_SUFFIX}), constructor (key suffix: {@link #CONSTRUCTOR_SUFFIX}).
 * The full MDR key is the <i><b>key prefix</b> + <b>key suffix</b></i>. If there is no injection point specified the qualifiers will be applied
 * to property, method and constructor.  
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class QualifiersMdrUtil
{
   private final static Logger log = Logger.getLogger(QualifiersMdrUtil.class);
   
   private final static MDRStrategy SUPPLIED_MDR = new SuppliedMDRStrategy();
   
   private final static MDRStrategy WANTED_MDR = new WantedMDRStrategy();

   /** The key under which we will store a contexts supplied qualifier metadata */
   public final static String SUPPLIED_QUALIFIER_KEY = QualifiersMdrUtil.class.getSimpleName() + "#SUPPLIED_QUALIFIERS";

   /** The key under which we will store a contexts wanted qualifier metadata */
   public final static String REQUIRED_QUALIFIER_KEY = QualifiersMdrUtil.class.getSimpleName() + "#REQUIRED_QUALIFIERS";
   
   /** The key under which we will store a contexts required qualifier metadata */
   public final static String OPTIONAL_QUALIFIER_KEY = QualifiersMdrUtil.class.getSimpleName() + "#OPTIONAL_QUALIFIERS";
   
   public final static String CONSTRUCTOR_SUFFIX = "#CONSTRUCTOR";
   
   public final static String PROPERTY_SUFFIX = "#PROPERTY";
   
   public final static String METHOD_SUFFIX = "#METHOD";
   
   @SuppressWarnings("unchecked")
   private final static Class<? extends Annotation>[] QUALIFIERS = new Class[] {Qualifier.class};
   
   private final static TypeInfo BEAN_FACTORY_TYPE;
   static
   {
      TypeInfoFactory factory = new IntrospectionTypeInfoFactoryImpl();
      BEAN_FACTORY_TYPE = factory.getTypeInfo(BeanFactory.class);
   }
   
   /**
    * Get the wanted qualifiers defined from the bean's metadata or in the Mdr.
    * This method is used when creating injection values, which is before the Mdr
    * has been set up for a context.
    * 
    * @param context the context we want to check
    * @param point the injection point we want to narrow down qualifiers for
    * @return true if the bean metadata or the Mdr contains qualifiers
    */
   public static boolean hasWantedQualifiersInParentMdrOrBeanMetaData(KernelControllerContext context, QualifierPoint point)
   {
      Set<RelatedClassMetaData> related = context.getBeanMetaData().getRelated();
      if (related == null)
         return false;
      if (related.size() == 0)
         return false;
      
      Set<RelatedClassMetaData> md = null; 
      for (RelatedClassMetaData rcmd : related)
      {
         if (rcmd.getClassName().equals(REQUIRED_QUALIFIER_KEY) || rcmd.getClassName().equals(OPTIONAL_QUALIFIER_KEY))
         {
            if (md == null)
               md = new HashSet<RelatedClassMetaData>();
            md.add(rcmd);
         }
      }
      
      if (md != null && md.size() > 0)
         return true;

      MetaDataRepository repository = context.getKernel().getMetaDataRepository().getMetaDataRepository();
      ScopeKey key = context.getScopeInfo().getScope();
      while (key != null)
      {
         MetaData metaData = repository.getMetaData(key);
         if (metaData != null)
         {
            if (metaData.getMetaData(WANTED_MDR.getKey(REQUIRED_QUALIFIER_KEY, point)) != null)
               return true;
            if (metaData.getMetaData(WANTED_MDR.getKey(OPTIONAL_QUALIFIER_KEY, point)) != null)
               return true;
         }
         key = key.getParent();
      }
      
      return false;
   }
   
   
   /**
    * Adds the qualifiers for a context's bean metadata to the context's MDR metadata,
    * and checks the qualifier injection points for additional qualifiers coming from annotations 
    * 
    * @param context the context
    * @throws Exception for any error
    */
   public static void populateQualifiersForContext(KernelControllerContext context) throws Exception
   {
      Set<RelatedClassMetaData> qualifiers = context.getBeanMetaData().getRelated();
      
      if (qualifiers != null && qualifiers.size() > 0)
      {
         MetaDataRetrieval retrieval = context.getKernel().getMetaDataRepository().getMetaDataRepository().getMetaDataRetrieval(context.getScopeInfo().getMutableScope());
         if (retrieval instanceof MutableMetaData == false)
         {
            log.warn("Can not add qualifier to non mutable metadata" + context +  ":" + retrieval);
            return;
         }

         Set<RelatedClassMetaData> suppliedMetaData = new HashSet<RelatedClassMetaData>();
         Set<RelatedClassMetaData> requiredMetaData = new HashSet<RelatedClassMetaData>();
         Set<RelatedClassMetaData> optionalMetaData = new HashSet<RelatedClassMetaData>();

         splitRelatedClassMetaData(qualifiers, suppliedMetaData, requiredMetaData, optionalMetaData);
         
         for (RelatedClassMetaData rcmd : qualifiers)
         {
            if (JBossObject.equals(SUPPLIED_QUALIFIER_KEY, rcmd.getClassName()))
               SUPPLIED_MDR.addQualifiersToSetInMdr(suppliedMetaData, rcmd.getClassName(), retrieval);
            else if (JBossObject.equals(REQUIRED_QUALIFIER_KEY, rcmd.getClassName()))
               WANTED_MDR.addQualifiersToSetInMdr(requiredMetaData, rcmd.getClassName(), retrieval);
            else if (JBossObject.equals(OPTIONAL_QUALIFIER_KEY, rcmd.getClassName()))
               WANTED_MDR.addQualifiersToSetInMdr(optionalMetaData, rcmd.getClassName(), retrieval);
         }
      }

      addQualifiersFromQualifierAnnotations(context);
   }
   
   /**
    * Get the qualifiers coming from annotations for an injection point
    * 
    * @param context the kernel controller context
    * @param parents the parents of the injection point
    * @return the annotations specifying qualifiers
    * @throws Exception if an error occurred
    */
   public static Set<Annotation> getQualifiersFromAnnotationsForInjectionPointParents(KernelControllerContext context, List<MetaDataVisitorNode> parents) throws Exception
   {
      if (context == null)
         throw new IllegalArgumentException();
      MetaData metaData = context.getKernel().getMetaDataRepository().getMetaDataRepository().getMetaData(context.getScopeInfo().getScope());
      if (metaData == null)
         return null;
      return getQualifiersFromAnnotationsForInjectionPointParents(context, parents, metaData);
   }
   

   /**
    * Add qualifiers to a MDR metadata retrieval
    * 
    * @param retrieval the retrieval to add the data to. If it is not an instance of MutableMetaDataRetrieval no qualifiers will be added, but no error will be thrown
    * @param type the type of qualifier to add
    * @param point the injection point type to add this qualifier to. If it is null, it is added for constructors, fields and properties
    * @param qualifiers the qualfiers to add to the retrieval
    */
   public static void addQualifiersToMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers)
   {
      switch (type)
      {
         case SUPPLIED:
            SUPPLIED_MDR.addQualifiersToMdrRetrieval(retrieval, type, point, qualifiers);
            break;
         case REQUIRED:
         case OPTIONAL:
            WANTED_MDR.addQualifiersToMdrRetrieval(retrieval, type, point, qualifiers);
            break;
         default:
            throw new IllegalArgumentException("Unhandled type " + type);
      }
   }
   
   /**
    * Remove qualifiers from a MDR metadata retrieval
    * 
    * @param retrieval the retrieval to remove the data from. If it is not an instance of MutableMetaDataRetrieval no qualifiers will be removed, but no error will be thrown
    * @param type the type of qualifier to remove
    * @param point the injection point type to remove this qualifier from. If it is null, it is removed for constructors, fields and properties
    * @param qualifiers the qualfiers to remnove from the retrieval
    */
   public static void removeQualifiersFromMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers)
   {
      switch (type)
      {
         case SUPPLIED:
            SUPPLIED_MDR.removeQualifiersFromMdrRetrieval(retrieval, type, point, qualifiers);
            break;
         case REQUIRED:
         case OPTIONAL:
            WANTED_MDR.removeQualifiersFromMdrRetrieval(retrieval, type, point, qualifiers);
            break;
         default:
            throw new IllegalArgumentException("Unhandled type " + type);
      }
   }
   
   /**
    * Removes the qualifiers for a context's bean metadata from the context's MDR metadata. 
    * 
    * @param context the context 
    */
   public static void removeQualifiersFromMdr(KernelControllerContext context)
   {
      Set<RelatedClassMetaData> qualifiers =  context.getBeanMetaData().getRelated();
      
      if (qualifiers != null && qualifiers.size() > 0)
      {
         MetaDataRetrieval retrieval = context.getKernel().getMetaDataRepository().getMetaDataRepository().getMetaDataRetrieval(context.getScopeInfo().getMutableScope());
         if (retrieval instanceof MutableMetaData == false)
         {
            log.warn("Can not remove qualifier from non mutable metadata" + context +  ":" + retrieval);
            return;
         }
         
         Set<RelatedClassMetaData> suppliedMetaData = new HashSet<RelatedClassMetaData>();
         Set<RelatedClassMetaData> requiredMetaData = new HashSet<RelatedClassMetaData>();
         Set<RelatedClassMetaData> optionalMetaData = new HashSet<RelatedClassMetaData>();

         splitRelatedClassMetaData(qualifiers, suppliedMetaData, requiredMetaData, optionalMetaData);

         for (RelatedClassMetaData rcmd : qualifiers)
         {
            if (JBossObject.equals(SUPPLIED_QUALIFIER_KEY, rcmd.getClassName()))
               SUPPLIED_MDR.removeQualifiersFromSetInMdr(suppliedMetaData, rcmd.getClassName(), retrieval);
            else if (JBossObject.equals(REQUIRED_QUALIFIER_KEY, rcmd.getClassName()))
                  WANTED_MDR.removeQualifiersFromSetInMdr(requiredMetaData, rcmd.getClassName(), retrieval);
            else if (JBossObject.equals(OPTIONAL_QUALIFIER_KEY, rcmd.getClassName()))
               WANTED_MDR.removeQualifiersFromSetInMdr(optionalMetaData, rcmd.getClassName(), retrieval);
         }
      }
   }
   
   /**
    * Gets all the supplied qualifiers for the context from the MDR. The returned set combines the qualifiers 
    * found at all scope levels
    * 
    * @param context the context 
    * @return the found qualifiers
    */
   public static Set<Object> mergeSuppliedQualifiersFromMdr(ControllerContext context)
   {
      return SUPPLIED_MDR.mergeQualifiersFromMdr(context, QualifierType.SUPPLIED, null);
   }
   
   /**
    * Gets all the required qualifiers for the context from the MDR. The returned set combines the qualifiers 
    * found at all scope levels
    * 
    * @param context the context 
    * @param point the qualifier point
    * @return the found qualifiers
    */
   public static Set<Object> mergeRequiredQualifiersFromMdr(ControllerContext context, QualifierPoint point)
   {
      return WANTED_MDR.mergeQualifiersFromMdr(context, QualifierType.REQUIRED, point);
   }
   
   /**
    * Gets all the optional qualifiers for the context from the MDR. The returned set combines the qualifiers 
    * found at all scope levels
    * 
    * @param context the context 
    * @param point the qualifier point
    * @return the found qualifiers
    */
   public static Set<Object> mergeOptionalQualifiersFromMdr(ControllerContext context, QualifierPoint point)
   {
      return WANTED_MDR.mergeQualifiersFromMdr(context, QualifierType.OPTIONAL, point);
   }

   /**
    * Get an MDR key for the supplied type and the type of injection point
    * 
    * @param type the type
    * @param point the point
    * @return the key
    */
   public static String getKey(QualifierType type, QualifierPoint point)
   {
      switch (type)
      {
         case OPTIONAL :
         case REQUIRED :
            return WANTED_MDR.getKey(type, point);
         case SUPPLIED :
            return SUPPLIED_MDR.getKey(type, point);
         default:
            throw new IllegalArgumentException("Unhandled type " + type);
      }
   }
   
   private static void splitRelatedClassMetaData(Set<RelatedClassMetaData> related, Set<RelatedClassMetaData> suppliedMetaData, Set<RelatedClassMetaData> requiredMetaData, Set<RelatedClassMetaData> optionalMetaData)
   {
      for (RelatedClassMetaData rcmd : related)
      {
         if (JBossObject.equals(SUPPLIED_QUALIFIER_KEY, rcmd.getClassName()))
            suppliedMetaData.add(rcmd);
         else if (JBossObject.equals(REQUIRED_QUALIFIER_KEY, rcmd.getClassName()))
            requiredMetaData.add(rcmd);
         else if (JBossObject.equals(OPTIONAL_QUALIFIER_KEY, rcmd.getClassName()))
            optionalMetaData.add(rcmd);
      }
   }        

   private static void addQualifiersFromQualifierAnnotations(KernelControllerContext context) throws Exception
   {
      MetaData metaData = context.getKernel().getMetaDataRepository().getMetaDataRepository().getMetaData(context.getScopeInfo().getScope());
      for (DependencyItem item : context.getDependencyInfo().getIDependOn(ContextualInjectionDependencyItem.class))
      {
         Set<Annotation> annotations = getQualifiersFromAnnotationsForInjectionPointParents(context, ((ContextualInjectionDependencyItem)item).getParents(), metaData);
         if (annotations != null && annotations.size() > 0)
            ((ContextualInjectionDependencyItem)item).addQualifierAnnotations(annotations);
      }
   }
   
   private static Set<Annotation> getQualifiersFromAnnotationsForInjectionPointParents(KernelControllerContext context, List<MetaDataVisitorNode>parents, MetaData metaData) throws Exception
   {
      Set<Annotation> annotationQualifiers = null;
      
      //Use cached metadata context
      metaData = RetrievalUtils.createCachedMetaData(metaData);
      
      for (int i = 0 ; i < parents.size() ; i++)
      {
         MetaDataVisitorNode node = parents.get(i);
         if (node instanceof PropertyMetaData)
         {
            annotationQualifiers = populateQualifiersFromAnnotationsForProperty(context, metaData, (PropertyMetaData)node);
         }
         else if (node instanceof ParameterMetaData)
         {
            ParameterMetaData pmd = (ParameterMetaData)node;
            i++;
            for ( ; i < parents.size() ; i++)
            {
               node = parents.get(i);
               if (node instanceof LifecycleMetaData)
               {
                  annotationQualifiers = populateQualifiersFromAnnotationsForMethod(context, metaData, pmd, (LifecycleMetaData)node);
                  break;
               }
               else if (node instanceof ConstructorMetaData)
               {
                  annotationQualifiers = populateQualifiersFromAnnotationsForConstructor(context, metaData, pmd, (ConstructorMetaData)node);
                  break;
               }
            }
            break;
         }
      }
      return annotationQualifiers;
   }

   private static Set<Annotation> populateQualifiersFromAnnotationsForProperty(KernelControllerContext context, MetaData metaData, PropertyMetaData property)
   {
      PropertyInfo info = context.getBeanInfo().getProperty(property.getName());
      MethodInfo setter = info.getSetter();
      if (setter != null)
      {
         MetaData methodMetaData = metaData.getComponentMetaData(new DeclaredMethodSignature(setter));
         MetaData paramMetaData = metaData.getComponentMetaData(new MethodParametersSignature(setter, 0));
         return populateQualifiersFromAnnotationsMetaData(methodMetaData, paramMetaData);
      }
      FieldInfo field = info.getFieldInfo();
      if (field != null)
      {
         MetaData fieldMetaData = metaData.getComponentMetaData(new FieldSignature(field));
         return populateQualifiersFromAnnotationsMetaData((Set<Annotation>)null, fieldMetaData);
      }
      MethodInfo getter = info.getGetter();
      if (getter != null)
      {
         MetaData methodMetaData = metaData.getComponentMetaData(new DeclaredMethodSignature(getter));
         return populateQualifiersFromAnnotationsMetaData((Set<Annotation>)null, methodMetaData);
      }
      return null;
   }
   
   private static Set<Annotation> populateQualifiersFromAnnotationsForConstructor(KernelControllerContext context, MetaData metaData, ParameterMetaData pmd, ConstructorMetaData cmd) throws Exception
   {
      if (cmd.getFactory() != null || cmd.getFactoryClass() != null || cmd.getFactoryMethod() != null || cmd.getValue() != null)
         return null;
      
      if (BEAN_FACTORY_TYPE.isAssignableFrom(context.getBeanInfo().getClassInfo()))
         return null;

      ConstructorInfo ctor = Configurator.findConstructor(false, context.getBeanInfo(), cmd).getConstructorInfo();
      MetaData ctorMetaData = metaData.getComponentMetaData(new ConstructorSignature(ctor));
      MetaData paramMetaData = metaData.getComponentMetaData(new ConstructorParametersSignature(ctor, pmd.getIndex()));
      
      return populateQualifiersFromAnnotationsMetaData(ctorMetaData, paramMetaData);
   }
   
   private static Set<Annotation> populateQualifiersFromAnnotationsForMethod(KernelControllerContext context, MetaData metaData, ParameterMetaData pmd, LifecycleMetaData lmd) throws Exception
   {
      String[] paramTypes = new String[lmd.getParameters().size()];
      for (int i = 0 ; i < paramTypes.length ; i++)
         paramTypes[i] = lmd.getParameters().get(i).getType();
      MethodInfo method = Config.findMethodInfo(context.getBeanInfo().getClassInfo(), lmd.getMethodName(), paramTypes, false);
      
      MetaData methodMetaData = metaData.getComponentMetaData(new DeclaredMethodSignature(method));
      MetaData paramMetaData = metaData.getComponentMetaData(new MethodParametersSignature(method, pmd.getIndex()));
      
      return populateQualifiersFromAnnotationsMetaData(methodMetaData, paramMetaData);
   }
   
   private static Set<Annotation> populateQualifiersFromAnnotationsMetaData(MetaData methodOrConstructor, MetaData parameter)
   {
      Set<Annotation> qualifiers = null;
      qualifiers = populateQualifiersFromAnnotationsMetaData(qualifiers, methodOrConstructor);
      qualifiers = populateQualifiersFromAnnotationsMetaData(qualifiers, parameter);
      return qualifiers;
   }
   
   private static Set<Annotation> populateQualifiersFromAnnotationsMetaData(Set<Annotation> qualifiers, MetaData metaData)
   {
      if (metaData == null)
         return qualifiers;
      
      for (Class<? extends Annotation> meta : QUALIFIERS)
      {
         for (Annotation annotation : metaData.getAnnotationsAnnotatedWith(meta))
         {
            if (qualifiers == null)
               qualifiers = new HashSet<Annotation>();
            qualifiers.add(annotation);
         }
      }
      return qualifiers;
   }
   
   private static abstract class MDRStrategy
   {
      abstract void addQualifiersToSetInMdr(Set<RelatedClassMetaData> relateds, String keyRoot, MetaDataRetrieval retrieval);

      abstract void addQualifiersToMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers);

      abstract void removeQualifiersFromSetInMdr(Set<RelatedClassMetaData> relateds, String keyRoot, MetaDataRetrieval retrieval);
      
      abstract void removeQualifiersFromMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers);

      abstract Set<Object> mergeQualifiersFromMdr(ControllerContext context, QualifierType type, QualifierPoint point);
      
      abstract String getKey(String className, QualifierPoint point);

      abstract String getKey(QualifierType type, QualifierPoint point);

      Set<Object> addQualifiersToSetInMdr(Set<Object> qualifierSet, MetaDataRetrieval retrieval, String key, Object... qualifiersToAdd)
      {
         if (qualifierSet == null)
            qualifierSet = getQualifiersSetInMdr(retrieval, key, true); 
         for (Object qualifier : qualifiersToAdd)
         {
            qualifierSet.add(qualifier);
         }
         
         return qualifierSet;
      }
      
      void removeQualifiersFromSetInMdr(Set<Object> qualifierSet, MetaDataRetrieval retrieval, String key, Object...qualifiersToRemove)
      {
         if (qualifierSet == null)
            qualifierSet = getQualifiersSetInMdr(retrieval, key, false);
         if (qualifierSet != null)
         {
            for (Object qualifier : qualifiersToRemove)
               qualifierSet.remove(qualifier);
         }
         removeEmptyQualifiersSetFromMdr(qualifierSet, key, retrieval, null);
      }

      @SuppressWarnings("unchecked")
      Set<Object> mergeQualifiersFromMdr(ControllerContext context, String key)
      {
         if (context instanceof KernelControllerContext == false)
            return null;
         
         MutableMetaDataRepository repository = ((KernelControllerContext)context).getKernel().getMetaDataRepository().getMetaDataRepository();
         ScopeKey scope = context.getScopeInfo().getScope();

         Set<Object> set = null;
         while (scope != null)
         {
            MetaData metaData = repository.getMetaData(scope);
            if (metaData != null)
            {
               Set<Object> entry = metaData.getMetaData(key, Set.class);
               if (entry != null)
               {
                  if (set == null)
                     set = entry;
                  else
                  {
                     set = new HashSet<Object>(set);
                     set.addAll(entry);
                  }
               }
            }         
            scope = scope.getParent();
         }
         return set;
      }

      void removeEmptyQualifiersSetFromMdr(Set<Object> set, String keyRoot, MetaDataRetrieval retrieval, QualifierPoint point)
      {
         if (set != null && set.size() == 0)
            ((MutableMetaData)retrieval).removeMetaData(getKey(keyRoot, point), Set.class);
      }
      
      String typeToKeyRoot(QualifierType type)
      {
         switch (type)
         {
            case OPTIONAL:
               return OPTIONAL_QUALIFIER_KEY;
            case REQUIRED:
               return REQUIRED_QUALIFIER_KEY;
            case SUPPLIED:
               return SUPPLIED_QUALIFIER_KEY;
            default:
               throw new IllegalArgumentException("Unhandled type " + type);
         }
      }
      
      @SuppressWarnings("unchecked")
      Set<Object> getQualifiersSetInMdr(MetaDataRetrieval retrieval, String key, boolean create)
      {
         MetaDataItem<?> item = retrieval.retrieveMetaData(key);
         //TODO - The following is not threadsafe
         Set<Object> set = null;
         if (item == null)
         {
            if (create)
            {
               set = new ConcurrentSet<Object>(16, .75f, 2);
               ((MutableMetaData)retrieval).addMetaData(key, set, Set.class);
            }
            else
            {
               return null;
            }
         }
         else
         {
            set = (Set<Object>)item.getValue();
         }
         return set;
      }
   }
   
   private static class SuppliedMDRStrategy extends MDRStrategy
   {
      @Override
      void addQualifiersToSetInMdr(Set<RelatedClassMetaData> relateds, String keyRoot, MetaDataRetrieval retrieval)
      {
         if (keyRoot.equals(SUPPLIED_QUALIFIER_KEY) == false)
            throw new IllegalArgumentException("Wrong key " + keyRoot);
         Set<Object> set = null;
         for (RelatedClassMetaData rcmd : relateds)
         {
            set = addQualifiersToSetInMdr(set, retrieval, keyRoot, rcmd.getEnabled().toArray());
         }         
      }

      @Override
      void addQualifiersToMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers)
      {
         if (type != QualifierType.SUPPLIED)
            throw new IllegalArgumentException("Wrong type " + type);
         if (point != null)
            throw new IllegalArgumentException("Injection poing passed in for supplied qualifier");

         addQualifiersToSetInMdr(null, retrieval, SUPPLIED_QUALIFIER_KEY, qualifiers);
      }
      
      @Override
      void removeQualifiersFromSetInMdr(Set<RelatedClassMetaData> relateds, String keyRoot, MetaDataRetrieval retrieval) 
      {
         try
         {
            if (relateds == null || relateds.size() == 0)
               return;
            
            if (keyRoot.equals(keyRoot) == false)
               throw new IllegalArgumentException("Wrong key " + keyRoot);
            
            Set<Object> set = getQualifiersSetInMdr(retrieval, keyRoot, false);
            if (set != null)
            {
               for (RelatedClassMetaData rcmd : relateds)
               {
                  if (rcmd.getEnabled() != null && rcmd.getEnabled().size() > 0)
                  {
                     set.removeAll(rcmd.getEnabled());
                  }
               }
            }
            removeEmptyQualifiersSetFromMdr(set, keyRoot, retrieval, null);
         }
         catch(Exception e)
         {
            log.warn("Error removing qualifiers", e);
         }
      }
      
      void removeQualifiersFromMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers)
      {
         if (type != QualifierType.SUPPLIED)
            throw new IllegalArgumentException("Wrong type " + type);
         if (point != null)
            throw new IllegalArgumentException("Injection poing passed in for supplied qualifier");

         removeQualifiersFromSetInMdr(null, retrieval, SUPPLIED_QUALIFIER_KEY, qualifiers);
      }
      
      @Override
      Set<Object> mergeQualifiersFromMdr(ControllerContext context, QualifierType type, QualifierPoint point)
      {
         return super.mergeQualifiersFromMdr(context, typeToKeyRoot(type));
      }

      @Override
      public String getKey(QualifierType type, QualifierPoint point)
      {
         return typeToKeyRoot(type);
      }

      @Override
      String getKey(String className, QualifierPoint point)
      {
         return className;
      }
   }
   
   private static class WantedMDRStrategy extends MDRStrategy
   {
      @Override
      void addQualifiersToSetInMdr(Set<RelatedClassMetaData> relateds, String keyRoot, MetaDataRetrieval retrieval)
      {
         Set<Object> constructorSet = null;
         Set<Object> methodSet = null;
         Set<Object> propertySet = null;
         for (RelatedClassMetaData rcmd : relateds)
         {
            if (rcmd instanceof AbstractBeanQualifierMetaData)
            {
               AbstractBeanQualifierMetaData aqmd = (AbstractBeanQualifierMetaData)rcmd;
               if (aqmd.getPoints() != null && aqmd.getPoints().size() > 0)
               {
                  constructorSet = addQualifiersToSetInMdrIfHasCorrectType(constructorSet, retrieval, aqmd, QualifierPoint.CONSTRUCTOR);
                  methodSet = addQualifiersToSetInMdrIfHasCorrectType(methodSet, retrieval, aqmd, QualifierPoint.METHOD);
                  propertySet = addQualifiersToSetInMdrIfHasCorrectType(propertySet, retrieval, aqmd, QualifierPoint.PROPERTY);
                  continue;
               }
            }

            constructorSet = addQualifiersToSetInMdr(constructorSet, retrieval, rcmd, QualifierPoint.CONSTRUCTOR);
            methodSet = addQualifiersToSetInMdr(methodSet, retrieval, rcmd, QualifierPoint.METHOD);
            propertySet = addQualifiersToSetInMdr(propertySet, retrieval, rcmd, QualifierPoint.PROPERTY);
         }
      }

      private Set<Object> addQualifiersToSetInMdrIfHasCorrectType(Set<Object> qualifierSet, MetaDataRetrieval retrieval, AbstractBeanQualifierMetaData aqmd, QualifierPoint point)
      {
         
         if (aqmd.getPoints().contains(point))
            return addQualifiersToSetInMdr(qualifierSet, retrieval, aqmd, point);
         return qualifierSet;
      }

      private Set<Object> addQualifiersToSetInMdr(Set<Object> qualifierSet, MetaDataRetrieval retrieval, RelatedClassMetaData rcmd, QualifierPoint point)
      {
         return addQualifiersToSetInMdr(qualifierSet, retrieval, getKey(rcmd.getClassName(), point), rcmd.getEnabled().toArray());
      }
      
      @Override
      void addQualifiersToMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object...qualifiers)
      {
         if (point == null)
         {
            addQualifiersToSetInMdr(null, retrieval, getKey(type, QualifierPoint.CONSTRUCTOR), qualifiers);
            addQualifiersToSetInMdr(null, retrieval, getKey(type, QualifierPoint.METHOD), qualifiers);
            addQualifiersToSetInMdr(null, retrieval, getKey(type, QualifierPoint.PROPERTY ), qualifiers);
         }
         else
         {
            addQualifiersToSetInMdr(null, retrieval, getKey(type, point), qualifiers);
         }
      }

      @Override
      void removeQualifiersFromSetInMdr(Set<RelatedClassMetaData> relateds, String keyRoot, MetaDataRetrieval retrieval)
      {
         if (relateds == null || relateds.size() == 0)
            return;
         
         Set<Object> constructorSet = getQualifiersSetInMdr(retrieval, getKey(keyRoot, QualifierPoint.CONSTRUCTOR), false);
         Set<Object> methodSet = getQualifiersSetInMdr(retrieval, getKey(keyRoot, QualifierPoint.METHOD), false);
         Set<Object> propertySet = getQualifiersSetInMdr(retrieval, getKey(keyRoot, QualifierPoint.PROPERTY), false);
         for (RelatedClassMetaData rcmd : relateds)
         {
            if (rcmd.getEnabled() != null && rcmd.getEnabled().size() > 0)
            {
               if (rcmd instanceof AbstractBeanQualifierMetaData)
               {
                  AbstractBeanQualifierMetaData aqmd = (AbstractBeanQualifierMetaData)rcmd;
                  if (aqmd.getPoints() != null && aqmd.getPoints().size() > 0)
                  {
                     if (aqmd.getPoints().contains(QualifierPoint.CONSTRUCTOR) && constructorSet != null)
                        constructorSet.removeAll(aqmd.getEnabled());
                     else if (aqmd.getPoints().contains(QualifierPoint.METHOD) && methodSet != null)
                        methodSet.removeAll(aqmd.getEnabled());
                     else if (aqmd.getPoints().contains(QualifierPoint.PROPERTY) && propertySet != null)
                        propertySet.removeAll(aqmd.getEnabled());
                  }
               }
            }
         }
         removeEmptyQualifiersSetFromMdr(constructorSet, keyRoot, retrieval, QualifierPoint.CONSTRUCTOR);
         removeEmptyQualifiersSetFromMdr(methodSet, keyRoot, retrieval, QualifierPoint.METHOD);
         removeEmptyQualifiersSetFromMdr(propertySet, keyRoot, retrieval, QualifierPoint.PROPERTY);
      }
      
      @Override
      void removeQualifiersFromMdrRetrieval(MetaDataRetrieval retrieval, QualifierType type, QualifierPoint point, Object... qualifiers)
      {
         removeQualifiersFromSetInMdr(null, retrieval, getKey(type, point), qualifiers);
      }

      @Override
      Set<Object> mergeQualifiersFromMdr(ControllerContext context, QualifierType type, QualifierPoint point)
      {
         return super.mergeQualifiersFromMdr(context, getKey(typeToKeyRoot(type), point));
      }

      @Override
      public String getKey(QualifierType type, QualifierPoint point)
      {
         return getKey(typeToKeyRoot(type), point);
      }
      
      String getKey(String className, QualifierPoint point)
      {
         if (point == null)
            throw new IllegalArgumentException("Null point for required/optional qualifier");
       
         switch (point)
         {
            case CONSTRUCTOR :
               return className + CONSTRUCTOR_SUFFIX;
            case METHOD :
               return className + METHOD_SUFFIX;
            case PROPERTY :
               return className + PROPERTY_SUFFIX;
            default :
               throw new IllegalArgumentException("Unknown point " + point);
         }
    }

   }
}
