/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.plugins.qualifier;

import java.lang.annotation.Annotation;

import org.jboss.annotation.factory.AnnotationCreator;
import org.jboss.beans.metadata.api.model.QualifierContent;
import org.jboss.kernel.spi.qualifier.QualifierParser;

/**
 * Handles {@link QualifierContent#ANNOTATION} and parses raw qualifiers into annotations. 
 * The raw qualifier values must either be of type:
 * <ul>
 *   <li><b>java.lang.Annotation</b> in which case the raw qualifier is returned
 *   <li><b>java.lang.String</b> in which case the string is parsed into a qualifier
 * </ul>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class AnnotationQualiferParser implements QualifierParser
{
   public static AnnotationQualiferParser INSTANCE = new AnnotationQualiferParser();
   
   private AnnotationQualiferParser() {}
   
   public QualifierContent getHandledContent()
   {
      return QualifierContent.ANNOTATION;
   }

   public Object parseSupplied(ClassLoader cl, Object rawQualifier)
   {
      return parse(cl, rawQualifier);
   }

   public Object parseWanted(ClassLoader cl, Object rawQualifier)
   {
      return parse(cl, rawQualifier);
   }

   private Object parse(ClassLoader cl, Object rawQualifier)
   {
      if (rawQualifier instanceof Annotation)
         return rawQualifier;
      if (rawQualifier instanceof String == false)
         throw new IllegalArgumentException(rawQualifier + " is neither an annotation nor a string");
         
      String annString = (String)rawQualifier;
      try
      {
         if (cl == null) 
         {
            cl = Thread.currentThread().getContextClassLoader();
         }
         return AnnotationCreator.createAnnotation(annString, cl);
      }
      catch (Throwable e)
      {
         throw new RuntimeException("Error creating annotation for " + rawQualifier, e);
      }
   }
}