/*
* JBoss, Home of Professional Open Source
* Copyright 2007, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.beans.metadata.spi.factory;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.Stack;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.beans.metadata.plugins.TypeProvider;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.MetaDataVisitor;
import org.jboss.beans.metadata.spi.MetaDataVisitorNode;
import org.jboss.beans.metadata.spi.RelatedClassMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.kernel.spi.config.KernelConfigurator;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;

/**
 * PropertyMap.
 * Used to copy the properties from the {@link GenericBeanFactoryMetaData}
 * to the resulting {@link BeanFactory} implementation following the call to 
 * {@link GenericBeanFactoryMetaData#getBeans()}.
 *
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 */
class PropertyMap extends HashMap<String, ValueMetaData> implements MetaDataVisitorNode, TypeProvider
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -4295725682462294630L;

   public void initialVisit(MetaDataVisitor visitor)
   {
      visitor.initialVisit(this);
   }

   public void describeVisit(MetaDataVisitor vistor)
   {
      vistor.describeVisit(this);
   }

   public Iterator<? extends MetaDataVisitorNode> getChildren()
   {
      return values().iterator();
   }

   public ValueMetaData put(String name, ValueMetaData value)
   {
      ValueInfo info = new ValueInfo(name, value);
      return super.put(name, info);
   }

   public TypeInfo getType(MetaDataVisitor visitor, MetaDataVisitorNode previous) throws Throwable
   {
      ValueInfo valueInfo = null;
      for (ValueMetaData value : values())
      {
         ValueInfo vi = ValueInfo.class.cast(value);
         if (vi == previous)
         {
            valueInfo = vi;
            break;
         }
      }
      if (valueInfo == null)
         throw new IllegalArgumentException("No matching value (" + previous + ") found: " + values());

      KernelControllerContext context = visitor.getControllerContext();
      BeanMetaData bmd = context.getBeanMetaData();
      Set<RelatedClassMetaData> related = bmd.getRelated();
      if (related == null || related.size() != 1)
         throw new IllegalArgumentException("Invalid information for contextual injection: " + bmd);
      // TODO - perhaps match which related metadata is the right one
      RelatedClassMetaData beanClassMetaData = related.iterator().next();
      KernelConfigurator configurator = context.getKernel().getConfigurator();
      BeanInfo beanInfo = configurator.getBeanInfo(beanClassMetaData.getClassName(), context.getClassLoader());
      PropertyInfo pi = beanInfo.getProperty(valueInfo.name);
      TypeInfo typeInfo = pi.getType();
      if (typeInfo.isCollection() || typeInfo.isMap())
      {
         throw new IllegalArgumentException("Cannot handle collection or map: " + valueInfo);
      }
      return typeInfo;
   }

   private static class ValueInfo extends JBossObject implements ValueMetaData, Serializable, TypeProvider
   {
      private static final long serialVersionUID = 1L;

      private String name;
      private ValueMetaData value;

      private ValueInfo(String name, ValueMetaData value)
      {
         this.name = name;
         this.value = value;
      }

      public TypeInfo getType(MetaDataVisitor visitor, MetaDataVisitorNode previous) throws Throwable
      {
         Stack<MetaDataVisitorNode> visitorNodeStack = visitor.visitorNodeStack();
         // see AbstractInjectionValueMetaData.describeVisit
         MetaDataVisitorNode node = visitorNodeStack.pop();
         try
         {
            if (node instanceof TypeProvider)
            {
               TypeProvider typeProvider = (TypeProvider) node;
               return typeProvider.getType(visitor, this);
            }
            else
            {
               throw new IllegalArgumentException(TypeProvider.ERROR_MSG);
            }
         }
         finally
         {
            visitorNodeStack.push(node);
         }
      }

      public Object getUnderlyingValue()
      {
         return value.getUnderlyingValue();
      }

      public Object getValue(TypeInfo info, ClassLoader cl) throws Throwable
      {
         return value.getValue(info, cl);
      }

      public Object ungetValue(TypeInfo info, ClassLoader cl) throws Throwable
      {
         return value.ungetValue(info, cl);
      }

      public void initialVisit(MetaDataVisitor vistor)
      {
         vistor.initialVisit(this);
      }

      public void describeVisit(MetaDataVisitor vistor)
      {
         vistor.describeVisit(this);
      }

      public Iterator<? extends MetaDataVisitorNode> getChildren()
      {
         return Collections.singleton(value).iterator();
      }

      public void toShortString(JBossStringBuilder buffer)
      {
         value.toShortString(buffer);
      }

      @Override
      public int hashCode()
      {
         return value.hashCode();
      }

      @Override
      public boolean equals(Object obj)
      {
         return value.equals(obj);
      }

      public String toShortString()
      {
         return value.toShortString();
      }
   }
}
