/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.beans.metadata.api.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * A value that is calculated by making a method call on another bean.
 * 
 * <pre>
 * &#64;ValueFactory(
 *    bean="PersonRegister", 
 *    method="getPersonFromId", 
 *    parameters={&#64;Parameter(string=@StringValue(value="123", type="int"))}) 
 * public void setCalculatedValue(Person person) {} 
 * </pre>
 * Here, the <code>person</code> parameter is calculated, by calling <code>getPersonFromId(123)</code>
 * on the bean called <code>PersonRegister</code>.
 * 
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.METHOD, ElementType.PARAMETER, ElementType.FIELD})
public @interface ValueFactory
{
   /**
    * Get the bean name containing the method.
    *
    * @return the bean
    */
   String bean();

   /**
    * Get the method name.
    *
    * @return the method
    */
   String method();

   /**
    * Get single parameter. Cannot be used in conjunction with {@link #parameters()}
    *
    * @return the single parameter
    */
   String parameter() default "";

   /**
    * Get parameters. Cannot be used in conjunction with {@link #parameter()}
    *
    * @return the parameters
    */
   Parameter[] parameters() default {};

   /**
    * Get default value to be used if the result of the method call is null.
    *
    * @return the default value
    */
   String defaultValue() default "";

   /**
    * Get dependant state.
    *
    * @return the dependant state
    */
   String dependantState() default "Installed";

   /**
    * Get when required state.
    *
    * @return the when required state
    */
   String whenRequiredState() default "Configured";
}
