/*
 * Decompiled with CFR 0.152.
 */
package org.jivesoftware.openfire.keystore;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import javax.net.ssl.KeyManagerFactory;
import org.bouncycastle.operator.OperatorCreationException;
import org.jivesoftware.openfire.XMPPServer;
import org.jivesoftware.openfire.keystore.CertificateStore;
import org.jivesoftware.openfire.keystore.CertificateStoreConfigException;
import org.jivesoftware.openfire.keystore.CertificateStoreConfiguration;
import org.jivesoftware.openfire.keystore.CertificateUtils;
import org.jivesoftware.openfire.net.DNSUtil;
import org.jivesoftware.util.CertificateManager;
import org.jivesoftware.util.JiveGlobals;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IdentityStore
extends CertificateStore {
    private static final Logger Log = LoggerFactory.getLogger(IdentityStore.class);

    public IdentityStore(CertificateStoreConfiguration configuration, boolean createIfAbsent) throws CertificateStoreConfigException {
        super(configuration, createIfAbsent);
        try {
            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(this.getStore(), configuration.getPassword());
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException ex) {
            throw new CertificateStoreConfigException("Unable to initialize identity store (a common cause: the password for a key is different from the password of the entire store).", ex);
        }
    }

    public String generateCSR(String alias) throws CertificateStoreConfigException {
        if (alias == null || alias.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'alias' cannot be null or an empty String.");
        }
        alias = alias.trim();
        try {
            if (!this.store.containsAlias(alias)) {
                throw new CertificateStoreConfigException("Cannot generate CSR for alias '" + alias + "': the alias does not exist in the store.");
            }
            Certificate certificate = this.store.getCertificate(alias);
            if (certificate == null || !(certificate instanceof X509Certificate)) {
                throw new CertificateStoreConfigException("Cannot generate CSR for alias '" + alias + "': there is no corresponding certificate in the store, or it is not an X509 certificate.");
            }
            Key key = this.store.getKey(alias, this.configuration.getPassword());
            if (key == null || !(key instanceof PrivateKey)) {
                throw new CertificateStoreConfigException("Cannot generate CSR for alias '" + alias + "': there is no corresponding key in the store, or it is not a private key.");
            }
            String pemCSR = CertificateManager.createSigningRequest((X509Certificate)certificate, (PrivateKey)key);
            return pemCSR;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | OperatorCreationException e) {
            throw new CertificateStoreConfigException("Cannot generate CSR for alias '" + alias + "'", e);
        }
    }

    public void installCSRReply(String alias, String pemCertificates) throws CertificateStoreConfigException {
        if (alias == null || alias.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'alias' cannot be null or an empty String.");
        }
        if (pemCertificates == null || pemCertificates.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'pemCertificates' cannot be null or an empty String.");
        }
        alias = alias.trim();
        pemCertificates = pemCertificates.trim();
        try {
            Collection<X509Certificate> certificates = CertificateManager.parseCertificates(pemCertificates);
            if (certificates.isEmpty()) {
                throw new CertificateStoreConfigException("No certificate was found in the input.");
            }
            List<X509Certificate> ordered = CertificateUtils.order(certificates);
            if (!IdentityStore.isForThisDomain(ordered.get(0))) {
                throw new CertificateStoreConfigException("The supplied certificate chain does not cover the domain of this XMPP service.");
            }
            if (!this.corresponds(alias, ordered)) {
                throw new IllegalArgumentException("The provided CSR reply does not match an existing certificate in the store under the provided alias '" + alias + "'.");
            }
            this.store.setKeyEntry(alias, this.store.getKey(alias, this.configuration.getPassword()), this.configuration.getPassword(), ordered.toArray(new X509Certificate[ordered.size()]));
        }
        catch (IOException | RuntimeException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
            this.reload();
            throw new CertificateStoreConfigException("Unable to install a singing reply into an identity store.", e);
        }
    }

    protected boolean corresponds(String alias, List<X509Certificate> certificates) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException {
        if (!this.store.containsAlias(alias)) {
            return false;
        }
        Key key = this.store.getKey(alias, this.configuration.getPassword());
        if (key == null) {
            return false;
        }
        if (!(key instanceof PrivateKey)) {
            return false;
        }
        Certificate certificate = this.store.getCertificate(alias);
        if (certificate == null) {
            return false;
        }
        if (!(certificate instanceof X509Certificate)) {
            return false;
        }
        X509Certificate x509Certificate = (X509Certificate)certificate;
        return x509Certificate.getPublicKey().equals(certificates.get(0).getPublicKey());
    }

    public String replaceCertificate(String pemCertificates, String pemPrivateKey, String passPhrase) throws CertificateStoreConfigException {
        if (pemCertificates == null || pemCertificates.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'pemCertificates' cannot be null or an empty String.");
        }
        if (pemPrivateKey == null || pemPrivateKey.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'pemPrivateKey' cannot be null or an empty String.");
        }
        pemCertificates = pemCertificates.trim();
        try {
            Collection<X509Certificate> certificates = CertificateManager.parseCertificates(pemCertificates);
            if (certificates.isEmpty()) {
                throw new CertificateStoreConfigException("No certificate was found in the input.");
            }
            List<X509Certificate> ordered = CertificateUtils.order(certificates);
            if (!IdentityStore.isForThisDomain(ordered.get(0))) {
                throw new CertificateStoreConfigException("The supplied certificate chain does not cover the domain of this XMPP service.");
            }
            PrivateKey privateKey = CertificateManager.parsePrivateKey(pemPrivateKey, passPhrase);
            this.removeAllDomainEntries();
            String alias = this.generateUniqueAlias();
            this.store.setKeyEntry(alias, privateKey, this.configuration.getPassword(), ordered.toArray(new X509Certificate[ordered.size()]));
            this.persist();
            Log.info("Replaced all private keys and corresponding certificate chains with a new private key and certificate chain.");
            return alias;
        }
        catch (IOException | KeyStoreException | CertificateException e) {
            this.reload();
            throw new CertificateStoreConfigException("Unable to install a certificate into an identity store.", e);
        }
    }

    public String installCertificate(String pemCertificates, String pemPrivateKey, String passPhrase) throws CertificateStoreConfigException {
        String alias = this.generateUniqueAlias();
        this.installCertificate(alias, pemCertificates, pemPrivateKey, passPhrase);
        return alias;
    }

    public void installCertificate(String alias, String pemCertificates, String pemPrivateKey, String passPhrase) throws CertificateStoreConfigException {
        if (alias == null || alias.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'alias' cannot be null or an empty String.");
        }
        if (pemCertificates == null || pemCertificates.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'pemCertificates' cannot be null or an empty String.");
        }
        if (pemPrivateKey == null || pemPrivateKey.trim().isEmpty()) {
            throw new IllegalArgumentException("Argument 'pemPrivateKey' cannot be null or an empty String.");
        }
        alias = alias.trim();
        pemCertificates = pemCertificates.trim();
        try {
            if (this.store.containsAlias(alias)) {
                throw new CertificateStoreConfigException("Certificate already exists for alias: " + alias);
            }
            Collection<X509Certificate> certificates = CertificateManager.parseCertificates(pemCertificates);
            if (certificates.isEmpty()) {
                throw new CertificateStoreConfigException("No certificate was found in the input.");
            }
            List<X509Certificate> ordered = CertificateUtils.order(certificates);
            if (!IdentityStore.isForThisDomain(ordered.get(0))) {
                throw new CertificateStoreConfigException("The supplied certificate chain does not cover the domain of this XMPP service.");
            }
            PrivateKey privateKey = CertificateManager.parsePrivateKey(pemPrivateKey, passPhrase);
            this.store.setKeyEntry(alias, privateKey, this.configuration.getPassword(), ordered.toArray(new X509Certificate[ordered.size()]));
            this.persist();
            Log.info("Installed a new private key and corresponding certificate chain.");
        }
        catch (IOException | KeyStoreException | CertificateException e) {
            this.reload();
            throw new CertificateStoreConfigException("Unable to install a certificate into an identity store.", e);
        }
    }

    public synchronized void ensureDomainCertificates(String ... algorithms) throws CertificateStoreConfigException {
        for (String algorithm : algorithms) {
            Log.debug("Verifying that a domain certificate ({} algorithm) is available in this store.", (Object)algorithm);
            if (this.containsDomainCertificate(algorithm)) continue;
            Log.debug("Store does not contain a domain certificate ({} algorithm). A self-signed certificate will be generated.", (Object)algorithm);
            this.addSelfSignedDomainCertificate(algorithm);
        }
    }

    public synchronized boolean containsDomainCertificate(String algorithm) throws CertificateStoreConfigException {
        String domainName = XMPPServer.getInstance().getServerInfo().getXMPPDomain();
        try {
            for (String alias : Collections.list(this.store.aliases())) {
                Certificate certificate = this.store.getCertificate(alias);
                if (!(certificate instanceof X509Certificate) || !certificate.getPublicKey().getAlgorithm().equalsIgnoreCase(algorithm)) continue;
                for (String identity : CertificateManager.getServerIdentities((X509Certificate)certificate)) {
                    if (!DNSUtil.isNameCoveredByPattern(domainName, identity)) continue;
                    return true;
                }
            }
            return false;
        }
        catch (KeyStoreException e) {
            throw new CertificateStoreConfigException("An exception occurred while searching for " + algorithm + " certificates that match the Openfire domain.", e);
        }
    }

    public synchronized void addSelfSignedDomainCertificate(String algorithm) throws CertificateStoreConfigException {
        String signAlgorithm;
        int keySize;
        switch (algorithm.toUpperCase()) {
            case "RSA": {
                keySize = JiveGlobals.getIntProperty("cert.rsa.keysize", 2048);
                signAlgorithm = "SHA256WITHRSAENCRYPTION";
                break;
            }
            case "DSA": {
                keySize = JiveGlobals.getIntProperty("cert.dsa.keysize", 1024);
                signAlgorithm = "SHA256withDSA";
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported algorithm '" + algorithm + "'. Use 'RSA' or 'DSA'.");
            }
        }
        String name = JiveGlobals.getProperty("xmpp.domain").toLowerCase();
        String alias = name + "_" + algorithm.toLowerCase();
        int validityInDays = 1825;
        Log.info("Generating a new private key and corresponding self-signed certificate for domain name '{}', using the {} algorithm (sign-algorithm: {} with a key size of {} bits). Certificate will be valid for {} days.", new Object[]{name, algorithm, signAlgorithm, keySize, 1825});
        try {
            KeyPair keyPair = IdentityStore.generateKeyPair(algorithm.toUpperCase(), keySize);
            X509Certificate cert = CertificateManager.createX509V3Certificate(keyPair, 1825, name, name, name, signAlgorithm);
            this.store.setKeyEntry(alias, keyPair.getPrivate(), this.configuration.getPassword(), new X509Certificate[]{cert});
            this.persist();
        }
        catch (IOException | GeneralSecurityException | CertificateStoreConfigException ex) {
            this.reload();
            throw new CertificateStoreConfigException("Unable to generate new self-signed " + algorithm + " certificate.", ex);
        }
    }

    protected static synchronized KeyPair generateKeyPair(String algorithm, int keySize) throws GeneralSecurityException {
        KeyPairGenerator generator = PROVIDER == null ? KeyPairGenerator.getInstance(algorithm) : KeyPairGenerator.getInstance(algorithm, PROVIDER);
        generator.initialize(keySize, new SecureRandom());
        return generator.generateKeyPair();
    }

    public static boolean isForThisDomain(X509Certificate certificate) {
        String domainName = XMPPServer.getInstance().getServerInfo().getXMPPDomain();
        List<String> serverIdentities = CertificateManager.getServerIdentities(certificate);
        for (String identity : serverIdentities) {
            if (!DNSUtil.isNameCoveredByPattern(domainName, identity)) continue;
            return true;
        }
        Log.info("The supplied certificate chain does not cover the domain of this XMPP service ('" + domainName + "'). Instead, it covers " + Arrays.toString(serverIdentities.toArray(new String[serverIdentities.size()])));
        return false;
    }

    protected synchronized String generateUniqueAlias() throws CertificateStoreConfigException {
        String domain = XMPPServer.getInstance().getServerInfo().getXMPPDomain();
        int index = 1;
        String alias = domain + "_" + index;
        try {
            while (this.store.containsAlias(alias)) {
                alias = domain + "_" + ++index;
            }
            return alias;
        }
        catch (KeyStoreException e) {
            throw new CertificateStoreConfigException("Unable to generate a unique alias for this identity store.", e);
        }
    }

    protected synchronized void removeAllDomainEntries() throws KeyStoreException {
        String domainName = XMPPServer.getInstance().getServerInfo().getXMPPDomain();
        HashSet<String> toDelete = new HashSet<String>();
        block0: for (String alias : Collections.list(this.store.aliases())) {
            Certificate certificate = this.store.getCertificate(alias);
            if (!(certificate instanceof X509Certificate)) continue;
            for (String identity : CertificateManager.getServerIdentities((X509Certificate)certificate)) {
                if (!DNSUtil.isNameCoveredByPattern(domainName, identity)) continue;
                toDelete.add(alias);
                continue block0;
            }
        }
        for (String alias : toDelete) {
            this.store.deleteEntry(alias);
        }
    }
}

