/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.nio.file.Path;
import java.security.KeyStore;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import org.elasticsearch.common.ssl.CompositeTrustConfig;
import org.elasticsearch.common.ssl.DefaultJdkTrustConfig;
import org.elasticsearch.common.ssl.EmptyKeyConfig;
import org.elasticsearch.common.ssl.KeyStoreUtil;
import org.elasticsearch.common.ssl.PemKeyConfig;
import org.elasticsearch.common.ssl.PemTrustConfig;
import org.elasticsearch.common.ssl.SslClientAuthenticationMode;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslConfiguration;
import org.elasticsearch.common.ssl.SslKeyConfig;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.SslVerificationMode;
import org.elasticsearch.common.ssl.StoreKeyConfig;
import org.elasticsearch.common.ssl.StoreTrustConfig;
import org.elasticsearch.common.ssl.TrustEverythingConfig;
import org.elasticsearch.common.ssl.X509Field;
import org.elasticsearch.core.Nullable;

public abstract class SslConfigurationLoader {
    static final List<String> DEFAULT_PROTOCOLS = Collections.unmodifiableList(SslConfiguration.ORDERED_PROTOCOL_ALGORITHM_MAP.containsKey("TLSv1.3") ? Arrays.asList("TLSv1.3", "TLSv1.2", "TLSv1.1") : Arrays.asList("TLSv1.2", "TLSv1.1"));
    private static final List<String> JDK12_CIPHERS = List.of("TLS_AES_256_GCM_SHA384", "TLS_AES_128_GCM_SHA256", "TLS_CHACHA20_POLY1305_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256", "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_256_GCM_SHA384", "TLS_RSA_WITH_AES_128_GCM_SHA256", "TLS_RSA_WITH_AES_256_CBC_SHA256", "TLS_RSA_WITH_AES_128_CBC_SHA256", "TLS_RSA_WITH_AES_256_CBC_SHA", "TLS_RSA_WITH_AES_128_CBC_SHA");
    static final List<String> DEFAULT_CIPHERS = JDK12_CIPHERS;
    private static final char[] EMPTY_PASSWORD = new char[0];
    public static final List<X509Field> GLOBAL_DEFAULT_RESTRICTED_TRUST_FIELDS = List.of(X509Field.SAN_OTHERNAME_COMMONNAME);
    private final String settingPrefix;
    private SslTrustConfig defaultTrustConfig;
    private SslKeyConfig defaultKeyConfig;
    private SslVerificationMode defaultVerificationMode;
    private SslClientAuthenticationMode defaultClientAuth;
    private List<String> defaultCiphers;
    private List<String> defaultProtocols;
    private List<X509Field> defaultRestrictedTrustFields;
    private Function<KeyStore, KeyStore> keyStoreFilter;

    public SslConfigurationLoader(String settingPrefix) {
        String string = this.settingPrefix = settingPrefix == null ? "" : settingPrefix;
        if (!this.settingPrefix.isEmpty() && !this.settingPrefix.endsWith(".")) {
            throw new IllegalArgumentException("Setting prefix [" + settingPrefix + "] must be blank or end in '.'");
        }
        this.defaultTrustConfig = new DefaultJdkTrustConfig();
        this.defaultKeyConfig = EmptyKeyConfig.INSTANCE;
        this.defaultVerificationMode = SslVerificationMode.FULL;
        this.defaultClientAuth = SslClientAuthenticationMode.OPTIONAL;
        this.defaultProtocols = DEFAULT_PROTOCOLS;
        this.defaultCiphers = DEFAULT_CIPHERS;
        this.defaultRestrictedTrustFields = GLOBAL_DEFAULT_RESTRICTED_TRUST_FIELDS;
    }

    public void setDefaultTrustConfig(SslTrustConfig defaultTrustConfig) {
        this.defaultTrustConfig = defaultTrustConfig;
    }

    public void setDefaultKeyConfig(SslKeyConfig defaultKeyConfig) {
        this.defaultKeyConfig = defaultKeyConfig;
    }

    public void setDefaultVerificationMode(SslVerificationMode defaultVerificationMode) {
        this.defaultVerificationMode = defaultVerificationMode;
    }

    public void setDefaultClientAuth(SslClientAuthenticationMode defaultClientAuth) {
        this.defaultClientAuth = defaultClientAuth;
    }

    public void setDefaultCiphers(List<String> defaultCiphers) {
        this.defaultCiphers = defaultCiphers;
    }

    public void setDefaultProtocols(List<String> defaultProtocols) {
        this.defaultProtocols = defaultProtocols;
    }

    public void setKeyStoreFilter(Function<KeyStore, KeyStore> keyStoreFilter) {
        this.keyStoreFilter = keyStoreFilter;
    }

    public void setDefaultRestrictedTrustFields(List<X509Field> x509Fields) {
        this.defaultRestrictedTrustFields = x509Fields;
    }

    protected abstract boolean hasSettings(String var1);

    protected abstract String getSettingAsString(String var1) throws Exception;

    protected abstract char[] getSecureSetting(String var1) throws Exception;

    protected abstract List<String> getSettingAsList(String var1) throws Exception;

    public SslConfiguration load(Path basePath) {
        Objects.requireNonNull(basePath, "Base Path cannot be null");
        List<String> protocols = this.resolveListSetting("supported_protocols", Function.identity(), this.defaultProtocols);
        List<String> ciphers = this.resolveListSetting("cipher_suites", Function.identity(), this.defaultCiphers);
        SslVerificationMode verificationMode = this.resolveSetting("verification_mode", SslVerificationMode::parse, this.defaultVerificationMode);
        SslClientAuthenticationMode clientAuth = this.resolveSetting("client_authentication", SslClientAuthenticationMode::parse, this.defaultClientAuth);
        List<X509Field> trustRestrictionsX509Fields = this.resolveListSetting("trust_restrictions.x509_fields", X509Field::parseForRestrictedTrust, this.defaultRestrictedTrustFields);
        SslKeyConfig keyConfig = this.buildKeyConfig(basePath);
        SslTrustConfig trustConfig = this.buildTrustConfig(basePath, verificationMode, keyConfig, Set.copyOf(trustRestrictionsX509Fields));
        if (protocols == null || protocols.isEmpty()) {
            throw new SslConfigException("no protocols configured in [" + this.settingPrefix + "supported_protocols]");
        }
        if (ciphers == null || ciphers.isEmpty()) {
            throw new SslConfigException("no cipher suites configured in [" + this.settingPrefix + "cipher_suites]");
        }
        boolean isExplicitlyConfigured = this.hasSettings(this.settingPrefix);
        return new SslConfiguration(this.settingPrefix, isExplicitlyConfigured, trustConfig, keyConfig, verificationMode, clientAuth, ciphers, protocols);
    }

    protected SslTrustConfig buildTrustConfig(Path basePath, SslVerificationMode verificationMode, SslKeyConfig keyConfig, @Nullable Set<X509Field> restrictedTrustFields) {
        List<String> certificateAuthorities = this.resolveListSetting("certificate_authorities", Function.identity(), null);
        String trustStorePath = this.resolveSetting("truststore.path", Function.identity(), null);
        if (certificateAuthorities != null && trustStorePath != null) {
            throw new SslConfigException("cannot specify both [" + this.settingPrefix + "certificate_authorities] and [" + this.settingPrefix + "truststore.path]");
        }
        if (!verificationMode.isCertificateVerificationEnabled()) {
            return TrustEverythingConfig.TRUST_EVERYTHING;
        }
        if (certificateAuthorities != null) {
            return new PemTrustConfig(certificateAuthorities, basePath);
        }
        if (trustStorePath != null) {
            char[] password = this.resolvePasswordSetting("truststore.secure_password", "truststore.password");
            String storeType = this.resolveSetting("truststore.type", Function.identity(), KeyStoreUtil.inferKeyStoreType(trustStorePath));
            String algorithm = this.resolveSetting("truststore.algorithm", Function.identity(), TrustManagerFactory.getDefaultAlgorithm());
            return new StoreTrustConfig(trustStorePath, password, storeType, algorithm, true, basePath);
        }
        return SslConfigurationLoader.buildDefaultTrustConfig(this.defaultTrustConfig, keyConfig);
    }

    protected static SslTrustConfig buildDefaultTrustConfig(SslTrustConfig trustConfig, SslKeyConfig keyConfig) {
        SslTrustConfig trust = keyConfig.asTrustConfig();
        if (trust == null) {
            return trustConfig;
        }
        return new CompositeTrustConfig(List.of(trustConfig, trust));
    }

    public SslKeyConfig buildKeyConfig(Path basePath) {
        String certificatePath = this.stringSetting("certificate");
        String keyPath = this.stringSetting("key");
        String keyStorePath = this.stringSetting("keystore.path");
        if (certificatePath != null && keyStorePath != null) {
            throw new SslConfigException("cannot specify both [" + this.settingPrefix + "certificate] and [" + this.settingPrefix + "keystore.path]");
        }
        if (certificatePath != null || keyPath != null) {
            if (keyPath == null) {
                throw new SslConfigException("cannot specify [" + this.settingPrefix + "certificate] without also setting [" + this.settingPrefix + "key]");
            }
            if (certificatePath == null) {
                throw new SslConfigException("cannot specify [" + this.settingPrefix + "key] without also setting [" + this.settingPrefix + "certificate]");
            }
            char[] password = this.resolvePasswordSetting("secure_key_passphrase", "key_passphrase");
            return new PemKeyConfig(certificatePath, keyPath, password, basePath);
        }
        if (keyStorePath != null) {
            char[] storePassword = this.resolvePasswordSetting("keystore.secure_password", "keystore.password");
            char[] keyPassword = this.resolvePasswordSetting("keystore.secure_key_password", "keystore.key_password");
            if (keyPassword.length == 0) {
                keyPassword = storePassword;
            }
            String storeType = this.resolveSetting("keystore.type", Function.identity(), KeyStoreUtil.inferKeyStoreType(keyStorePath));
            String algorithm = this.resolveSetting("keystore.algorithm", Function.identity(), KeyManagerFactory.getDefaultAlgorithm());
            return new StoreKeyConfig(keyStorePath, storePassword, storeType, this.keyStoreFilter, keyPassword, algorithm, basePath);
        }
        return this.defaultKeyConfig;
    }

    protected Path resolvePath(String settingKey, Path basePath) {
        return this.resolveSetting(settingKey, basePath::resolve, null);
    }

    private String expandSettingKey(String key) {
        return this.settingPrefix + key;
    }

    private char[] resolvePasswordSetting(String secureSettingKey, String legacySettingKey) {
        char[] securePassword = this.resolveSecureSetting(secureSettingKey, null);
        String legacyPassword = this.stringSetting(legacySettingKey);
        if (securePassword == null) {
            if (legacyPassword == null) {
                return EMPTY_PASSWORD;
            }
            return legacyPassword.toCharArray();
        }
        if (legacyPassword != null) {
            throw new SslConfigException("cannot specify both [" + this.settingPrefix + secureSettingKey + "] and [" + this.settingPrefix + legacySettingKey + "]");
        }
        return securePassword;
    }

    private String stringSetting(String key) {
        return this.resolveSetting(key, Function.identity(), null);
    }

    private <V> V resolveSetting(String key, Function<String, V> parser, V defaultValue) {
        try {
            String setting = this.getSettingAsString(this.expandSettingKey(key));
            if (setting == null || setting.isEmpty()) {
                return defaultValue;
            }
            return parser.apply(setting);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SslConfigException("cannot retrieve setting [" + this.settingPrefix + key + "]", e);
        }
    }

    private char[] resolveSecureSetting(String key, char[] defaultValue) {
        try {
            char[] setting = this.getSecureSetting(this.expandSettingKey(key));
            if (setting == null || setting.length == 0) {
                return defaultValue;
            }
            return setting;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SslConfigException("cannot retrieve secure setting [" + this.settingPrefix + key + "]", e);
        }
    }

    private <V> List<V> resolveListSetting(String key, Function<String, V> parser, List<V> defaultValue) {
        try {
            List<String> list = this.getSettingAsList(this.expandSettingKey(key));
            if (list == null || list.isEmpty()) {
                return defaultValue;
            }
            return list.stream().map(parser).collect(Collectors.toList());
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SslConfigException("cannot retrieve setting [" + this.settingPrefix + key + "]", e);
        }
    }
}

