/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509ExtendedTrustManager;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.core.Nullable;

public final class KeyStoreUtil {
    private KeyStoreUtil() {
        throw new IllegalStateException("Utility class should not be instantiated");
    }

    public static String inferKeyStoreType(String path) {
        String name;
        String string = name = path == null ? "" : path.toLowerCase(Locale.ROOT);
        if (name.endsWith(".p12") || name.endsWith(".pfx") || name.endsWith(".pkcs12")) {
            return "PKCS12";
        }
        return "jks";
    }

    public static KeyStore readKeyStore(Path path, String ksType, char[] password) throws GeneralSecurityException, IOException {
        KeyStore keyStore = KeyStore.getInstance(ksType);
        if (path != null) {
            try (InputStream in = Files.newInputStream(path, new OpenOption[0]);){
                keyStore.load(in, password);
            }
        }
        return keyStore;
    }

    public static KeyStore buildKeyStore(Collection<Certificate> certificateChain, PrivateKey privateKey, char[] password) throws GeneralSecurityException {
        KeyStore keyStore = KeyStoreUtil.buildNewKeyStore();
        keyStore.setKeyEntry("key", privateKey, password, certificateChain.toArray(new Certificate[0]));
        return keyStore;
    }

    public static KeyStore filter(KeyStore store, Predicate<KeyStoreEntry> filter) {
        KeyStoreUtil.stream(store, e -> new SslConfigException("Failed to apply filter to existing keystore", (Exception)e)).filter(filter.negate()).forEach(e -> e.delete());
        return store;
    }

    public static KeyStore buildTrustStore(Iterable<Certificate> certificates) throws GeneralSecurityException {
        assert (certificates != null) : "Cannot create keystore with null certificates";
        KeyStore store = KeyStoreUtil.buildNewKeyStore();
        int counter = 0;
        for (Certificate certificate : certificates) {
            store.setCertificateEntry("cert-" + counter, certificate);
            ++counter;
        }
        return store;
    }

    private static KeyStore buildNewKeyStore() throws GeneralSecurityException {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        try {
            keyStore.load(null, null);
        }
        catch (IOException e) {
            throw new SslConfigException("Unexpected error initializing a new in-memory keystore", e);
        }
        return keyStore;
    }

    public static X509ExtendedKeyManager createKeyManager(Certificate[] certificateChain, PrivateKey privateKey, char[] password) throws GeneralSecurityException, IOException {
        KeyStore keyStore = KeyStoreUtil.buildKeyStore(List.of(certificateChain), privateKey, password);
        return KeyStoreUtil.createKeyManager(keyStore, password, KeyManagerFactory.getDefaultAlgorithm());
    }

    public static X509ExtendedKeyManager createKeyManager(KeyStore keyStore, char[] password, String algorithm) throws GeneralSecurityException {
        KeyManager[] keyManagers;
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(algorithm);
        kmf.init(keyStore, password);
        for (KeyManager keyManager : keyManagers = kmf.getKeyManagers()) {
            if (!(keyManager instanceof X509ExtendedKeyManager)) continue;
            X509ExtendedKeyManager x509ExtendedKeyManager = (X509ExtendedKeyManager)keyManager;
            return x509ExtendedKeyManager;
        }
        throw new SslConfigException("failed to find a X509ExtendedKeyManager in the key manager factory for [" + algorithm + "] and keystore [" + keyStore + "]");
    }

    public static X509ExtendedTrustManager createTrustManager(@Nullable KeyStore trustStore, String algorithm) throws NoSuchAlgorithmException, KeyStoreException {
        TrustManager[] trustManagers;
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(algorithm);
        tmf.init(trustStore);
        for (TrustManager trustManager : trustManagers = tmf.getTrustManagers()) {
            if (!(trustManager instanceof X509ExtendedTrustManager)) continue;
            X509ExtendedTrustManager x509ExtendedTrustManager = (X509ExtendedTrustManager)trustManager;
            return x509ExtendedTrustManager;
        }
        throw new SslConfigException("failed to find a X509ExtendedTrustManager in the trust manager factory for [" + algorithm + "] and truststore [" + trustStore + "]");
    }

    public static X509ExtendedTrustManager createTrustManager(Collection<Certificate> certificates) throws GeneralSecurityException {
        KeyStore store = KeyStoreUtil.buildTrustStore(certificates);
        return KeyStoreUtil.createTrustManager(store, TrustManagerFactory.getDefaultAlgorithm());
    }

    public static Stream<KeyStoreEntry> stream(KeyStore keyStore, Function<GeneralSecurityException, ? extends RuntimeException> exceptionHandler) {
        try {
            return Collections.list(keyStore.aliases()).stream().map(a -> new KeyStoreEntry(keyStore, (String)a, exceptionHandler));
        }
        catch (KeyStoreException e) {
            throw exceptionHandler.apply(e);
        }
    }

    public static class KeyStoreEntry {
        private final KeyStore store;
        private final String alias;
        private final Function<GeneralSecurityException, ? extends RuntimeException> exceptionHandler;

        KeyStoreEntry(KeyStore store, String alias, Function<GeneralSecurityException, ? extends RuntimeException> exceptionHandler) {
            this.store = store;
            this.alias = alias;
            this.exceptionHandler = exceptionHandler;
        }

        public String getAlias() {
            return this.alias;
        }

        public X509Certificate getX509Certificate() {
            try {
                Certificate c = this.store.getCertificate(this.alias);
                if (c instanceof X509Certificate) {
                    X509Certificate x509Certificate = (X509Certificate)c;
                    return x509Certificate;
                }
                return null;
            }
            catch (KeyStoreException e) {
                throw this.exceptionHandler.apply(e);
            }
        }

        public boolean isKeyEntry() {
            try {
                return this.store.isKeyEntry(this.alias);
            }
            catch (KeyStoreException e) {
                throw this.exceptionHandler.apply(e);
            }
        }

        public PrivateKey getKey(char[] password) {
            try {
                Key key = this.store.getKey(this.alias, password);
                if (key instanceof PrivateKey) {
                    PrivateKey privateKey = (PrivateKey)key;
                    return privateKey;
                }
                return null;
            }
            catch (GeneralSecurityException e) {
                throw this.exceptionHandler.apply(e);
            }
        }

        public List<? extends X509Certificate> getX509CertificateChain() {
            try {
                Certificate[] certificates = this.store.getCertificateChain(this.alias);
                if (certificates == null || certificates.length == 0) {
                    return List.of();
                }
                return Stream.of(certificates).filter(c -> c instanceof X509Certificate).map(X509Certificate.class::cast).toList();
            }
            catch (KeyStoreException e) {
                throw this.exceptionHandler.apply(e);
            }
        }

        public void delete() {
            try {
                this.store.deleteEntry(this.alias);
            }
            catch (KeyStoreException e) {
                throw this.exceptionHandler.apply(e);
            }
        }
    }
}

